
package org.lsst.ccs.description;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import org.lsst.ccs.description.groovy.CCSBuilder;

/**
 * A ComponentNode builder class. 
 * It provides utility functions to build and manipulate ComponentNodes.
 * 
 * @author The LSST CCS Team
 */
public class ComponentNodeBuilder {

    private static final HashMap<String,ComponentNodeFactory> factories = new HashMap<>();
    
    //Register a GroovyComponentNodeFactory and a ClassComponentNodeFactory for serialized objects with the ComponentNodeBuilder.    
    static {
        ComponentNodeBuilder.registerComponentNodeFactory(new org.lsst.ccs.description.groovy.GroovyComponentNodeFactory());
        ComponentNodeBuilder.registerComponentNodeFactory(new org.lsst.ccs.description.classname.ClassComponentNodeFactory());
    }
    
    public static interface ComponentNodeFactory {

        /**
         * Build a descriptive node from the provided description String.
         * The description String is implementation specific and should be
         * documented.
         * 
         * @param descriptionStr String containing description information 
         *                       necessary to build a DescriptiveNode.
         * @param props          The properties to be used as part of the component
         *                       building process.
         * @return The DescriptiveNode built from the description String.
         *                       
         */
        public ComponentLookup buildComponentNode(String descriptionStr, Properties props);
        
        /**
         * Get the protocol for this GroovyComponentNodeFactory.
         * The protocol is used to identify the factory.
         * @return The factory's protocol
         */
        public String getProtocol();
       
        
    }
    
    private ComponentNodeBuilder() {
    }
    
    /**
     * Register a DescriptiveNodeFactory.
     * The protocol is used to register the factory and to pick it when needed.
     * 
     * @param factory The GroovyComponentNodeFactory to be registered.
     * 
     */
    public static void registerComponentNodeFactory(ComponentNodeFactory factory) {
        factories.put(factory.getProtocol(), factory);
    }
    
    /**
     * Build a Descriptive node given a fullDescription string of the form 
     * "protocol:descriptionStr".
     * Internally it picks the appropriate builder implementation to
     * create the corresponding DescriptiveNode.
     * 
     * @param fullDescription The full description to build a DescriptiveNode
     * @return The newly built DescriptiveNode.
     * 
     */
    public static ComponentLookup buildComponentNode(String fullDescription) {
        return buildComponentNode(fullDescription, null, new Properties());
    }
    
    public static ComponentLookup buildComponentNode(String fullDescription, String startupConfig, Properties props) {
        Map<String,String> buildArgs = new HashMap<>();
        buildArgs.put("startupConfig", startupConfig);
        buildArgs.put("descriptionName",fullDescription);
        return buildComponentNode(buildArgs, props);        
    }
    
    public static ComponentLookup buildComponentNode(Map<String,String> buildArgs, Properties props) {
                
        String descriptionProtocol = "groovy";
        String descriptionString = buildArgs.get("descriptionName");
        if (descriptionString.contains(":")) {
            descriptionProtocol = descriptionString.substring(0, descriptionString.indexOf(':'));
            descriptionString = descriptionString.substring(descriptionString.indexOf(':') + 1);
            buildArgs.put("descriptionName",descriptionString);
        }
        CCSBuilder.reset();
        ComponentNodeFactory factory = factories.get(descriptionProtocol);
        if ( factory == null ) {
            throw new IllegalArgumentException("There is no registered ComponentNodeFactory for protocol "+descriptionProtocol);
        }
        ComponentLookup lookup = factory.buildComponentNode(descriptionString, props);
        for (Entry<String,String> e : buildArgs.entrySet() ) {
            lookup.getTopComponentNode().addTag(e.getKey(), e.getValue());
        }
        return lookup;
    }
    
    
}
