package org.lsst.ccs.description;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;
import static org.lsst.ccs.description.ComponentNode.logger;

/**
 * This class helps find component of the subsystem as they are defined in
 * the description "tree".
 * 
 * @author The LSST CCS Team.
 * 
 * TO-DO: Allow to add components programmatically with a path
 * TO-DO: Should be possible to close this lookup, preventing further nodes from
 * being added. Or add listeners. Once a component lookup is added to a subsystem
 * it should not be possible to add more components.
 * 
 */
public class ComponentLookup {

    private final ComponentNode top;
    private final Map<String, ComponentNode> componentNodeDict = new HashMap<>();
    
    public ComponentLookup(ComponentNode top) {
        this.top = top;
        buildDict(top);
    }
    
    /**
     * Get the top ComponentNode, ie the tip of the tree structure.
     * 
     * @return The top ComponentNode.
     */
    public ComponentNode getTopComponentNode() {
        return top;
    }
    
    private void buildDict(ComponentNode<?> node) {
        addComponentNode(node);
        for (ComponentNode child : node.getChildren() ) {
            buildDict(child);
        }
    }
    
    private void addComponentNode(ComponentNode node) {
        componentNodeDict.put(node.getPath(), node);
    }

    public void setComponentNodeName(ComponentNode node, String name) {
        node.setKey(name);
        componentNodeDict.clear();
        buildDict(getTopComponentNode());
    }

    
    /**
     * 
     * @param parent the parent component to which the node is added.
     * @param node
     */
    public void addComponentNodeToLookup(ComponentNode parent, ComponentNode node) {
        if ( parent == null ) {
            throw new RuntimeException("Parent node cannot be null");
        }
        parent.addChild(node);
        buildDict(node);
    }

    /**
     * return any object registered by the componentNodeDictionaries of the service.
     *
     * @param path
     * @return
     */
    public Object getComponentByPath(String path) {
        ComponentNode node = componentNodeDict.get(path);
        if (node == null) {
            logger.log(Level.WARNING, "null reference associated to path {0}", path);        
            return null;
        } else {
            return node.getComponent();
        }
    }

    /**
     * return any object registered by the componentNodeDictionaries of the service.
     *
     * @param path
     * @return
     */
    public ComponentNode getNodeByPath(String path) {
        ComponentNode node = componentNodeDict.get(path);
        if (node == null) {
            logger.log(Level.WARNING, "no component for path {0}", path);
            return null;
        } else {
            return node;
        }
    }
    
    
    public String getNameOfComponent(Object obj) {
        ComponentNode componentNode = getComponentNodeForObject(obj);
        if (componentNode != null) {
            return componentNode.getKey();
        }
        return null;
    }
    
    public ComponentNode getComponentNodeForObject(Object obj) {
        for (Map.Entry<String, ComponentNode> entry : componentNodeDict.entrySet()) {
            if (entry.getValue().getComponent() == obj) {
                return entry.getValue();
            }
        }
        return null;
    }

    /**
     * gets the children of a Component in the ComponentNode and filters by
     * Class.
     *
     * @param node
     * @param classFilter the class filter. Any class if set to null
     * @param <N> the type of node
     * @return the children of {@code parentName}, mapped by their name. The
     * returned map preserves the order in which nodes were inserted in
     * the component node
     */
    public <N> Map<String, N> getChildren(ComponentNode node, Class<N> classFilter) {
        List<ComponentNode> children = new ArrayList<>();
        if (node == null) {
            children.add(getTopComponentNode());
        } else {
            ComponentNode current = node;
            if (current != null) {
                children.addAll(current.getChildren());
            } else return null;

        }
        Map<String, N> mapRes = new LinkedHashMap<>();
        for (ComponentNode child : children) {
            Object realValue = child.getComponent();
            if (classFilter == null || classFilter.isAssignableFrom(realValue.getClass())) {
                mapRes.put(child.getKey(), (N)realValue);
            }
        }
        return mapRes;
    }

    /**
     * Finds all the children of the given class in the sub-tree starting at the
     * node named {@code parentName}
     *
     * @param <N>
     * @param node the node to start getting the children
     * from. If set to null, the whole component tree is tested against the
     * filter, even the top node.
     * @param classFilter the class filter. Any class if set to null
     * @return
     */
    public <N> List<N> listDescendants(ComponentNode node, Class<N> classFilter) {
        List<ComponentNode> children = new ArrayList<>();
        if (node == null) {
            children.add(getTopComponentNode());
        } else {
            ComponentNode parentNode = node;
            if (parentNode == null) return null;
            children.addAll(parentNode.getChildren());
        }
        List<N> res = new ArrayList<>();
        res.addAll(listChildren(node, classFilter));
        for (ComponentNode c : children) {
            res.addAll(listDescendants(c, classFilter));
        }
        return res;
    }
    
    /**
     * lists the children of a component.
     *
     * @param parentName
     * @param <Q> the type of node
     * @param klass
     * @return an empty list if there are no children
     */
    public <Q> List<Q> listChildren(ComponentNode node, Class<Q> klass) {
        return getChildren(node, klass).values().stream()
                .collect(Collectors.toList());
    }

    public boolean containsComponent(Object obj) {
        return getComponentNodeForObject(obj) != null;
    }

}
