package org.lsst.ccs.description;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;
import static org.lsst.ccs.description.ComponentNode.logger;
import org.lsst.ccs.utilities.structs.UniquePair;

/**
 * This class helps find component of the subsystem as they are defined in
 * the description "tree".
 * 
 * @author The LSST CCS Team.
 * 
 * TO-DO: Allow to add components programmatically with a path
 * TO-DO: Should be possible to close this lookup, preventing further nodes from
 * being added. Or add listeners. Once a component lookup is added to a subsystem
 * it should not be possible to add more components.
 * 
 */
public class ComponentLookup {

    private final ComponentNode top;
    private final Map<String, ComponentNode> componentNodeDict = new HashMap<>();
    
    public ComponentLookup(ComponentNode top) {
        this.top = top;
        buildDict(top);
    }
    
    /**
     * Get the top ComponentNode, ie the tip of the tree structure.
     * 
     * @return The top ComponentNode.
     */
    public ComponentNode getTopComponentNode() {
        return top;
    }
    
    private void buildDict(ComponentNode<?> node) {
        addComponentNode(node);
        for (ComponentNode child : node.getChildren() ) {
            buildDict(child);
        }
    }
    
    private void addComponentNode(ComponentNode node) {
        componentNodeDict.put(node.getKey(), node);
//        if ( componentNodeObjectDict.containsKey(node.getComponent() )) {
//            throw new RuntimeException("Component "+node.getKey()+" has duplicate object");
//        }
    }
    
    /**
     * 
     * @param name the parent name.
     * @param node
     */
    public void addComponentNodeToLookup(String name, ComponentNode node) {
        ComponentNode parent = name == null ? top : getNodeByName(name);
        if (parent == null) {
            logger.warn("no node named : " + name + " : ignored");
            return;
        }
        parent.addChild(node);
        buildDict(node);
    }

    /**
     * return any object registered by the componentNodeDictionaries of the service.
     *
     * @param name
     * @return
     */
    public Object getComponentByName(String name) {
        ComponentNode node = componentNodeDict.get(name);
        if (node == null) {
            logger.log(Level.WARNING, "null reference associated to {0}", name);        
            return null;
        } else {
            return node.getComponent();
        }
    }

    /**
     * Check if the ComponentLookup already contains a ComponentNode for a given name.
     * 
     * @param name The name of the ComponentNode
     * @return true/false True if it exists, false otherwise.
     */
    public boolean containsComponentName(String name) {
        ComponentNode node = componentNodeDict.get(name);
        return node != null;
    }

    public ComponentNode getNodeByName(String name) {
        ComponentNode res = componentNodeDict.get(name);
        if (res == null) {
            logger.log(Level.WARNING, "no component named {0}", name);
        }
        return res;
    }

    public String getNameOfComponent(Object obj) {
        for (Map.Entry<String, ComponentNode> entry : componentNodeDict.entrySet()) {
            if (entry.getValue().getComponent() == obj) {
                return entry.getKey();
            }
        }
        return null;
    }
    
    /**
     * gets the children of a Component in the ComponentNode and filters by
     * Class.
     *
     * @param parentName
     * @param classFilter the class filter. Any class if set to null
     * @param <N> the type of node
     * @return the children of {@code parentName}, mapped by their name. The
     * returned map does not preserved the order in which nodes were inserted in
     * the component node
     */
    public <N> Map<String, N> getChildren(String parentName, Class<N> classFilter) {
        List<ComponentNode> children = new ArrayList<>();
        if (parentName == null) {
            children.add(getTopComponentNode());
        } else {
            ComponentNode current = getNodeByName(parentName);
            if (current != null) {
                children.addAll(current.getChildren());
            } else return null;

        }
        Map<String, N> mapRes = new HashMap<>();
        for (ComponentNode child : children) {
            Object realValue = child.getComponent();
            if (classFilter == null || classFilter.isAssignableFrom(realValue.getClass())) {
                mapRes.put(child.getKey(), (N)realValue);
            }
        }
        return mapRes;
    }

    /**
     * Finds all the children of the given class in the sub-tree starting at the
     * node named {@code parentName}
     *
     * @param <N>
     * @param parentName the name of the node to start getting the children
     * from. If set to null, the whole component tree is tested against the
     * filter, even the top node.
     * @param classFilter the class filter. Any class if set to null
     * @return
     */
    public <N> List<N> listDescendants(String parentName, Class<N> classFilter) {
        List<ComponentNode> children = new ArrayList<>();
        if (parentName == null) {
            children.add(getTopComponentNode());
        } else {
            ComponentNode parentNode = getNodeByName(parentName);
            if (parentNode == null) return null;
            children.addAll(parentNode.getChildren());
        }
        List<N> res = new ArrayList<>();
        res.addAll(listChildren(parentName, classFilter));
        for (ComponentNode c : children) {
            res.addAll(listDescendants(c.getKey(), classFilter));
        }
        return res;
    }
    
    /**
     * lists the children of a component.
     *
     * @param parentName
     * @param <Q> the type of node
     * @param klass
     * @return an empty list if there are no children
     */
    public <Q> List<Q> listChildren(String parentName, Class<Q> klass) {
        return getChildren(parentName, klass).values().stream()
                .collect(Collectors.toList());
    }

    /**
     * gets the Parent of a component in the <TT>ComponentNode</TT> tree.
     *
     * @param componentName
     * @return
     */
    public Map.Entry<String, Object> getParent(String componentName) {
        ComponentNode current = getNodeByName(componentName);
        if (current != null) {
            ComponentNode parent = current.getParent();
            if (parent == null) {
                return null;
            }
            String key = parent.getKey();
            Object value = parent.getComponent();
            return new UniquePair<String, Object>(key, value);
        } else {
            return null;
        }
    }

    /**
     * Returns the path of a component in the tree of components.
     * @param componentName
     * @return 
     */
    public String getFullPathFor(String componentName) {
        Map.Entry<String, Object> parent = getParent(componentName);
        if (parent == null) {
            return componentName;
        } else {
            return getFullPathFor(parent.getKey()) + "/" + componentName;
        }
    }
    
    public boolean containsComponent(Object obj) {
        return getNameOfComponent(obj) != null;
    }

}
