package org.lsst.ccs.description;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;
import static org.lsst.ccs.description.ComponentNode.logger;
import org.lsst.ccs.utilities.structs.UniquePair;

/**
 * This class helps find component of the subsystem as they are defined in
 * the description "tree".
 * 
 * @author The LSST CCS Team.
 * 
 * TO-DO: Allow to add components programmatically with a path
 * TO-DO: Should be possible to close this lookup, preventing further nodes from
 * being added. Or add listeners. Once a component lookup is added to a subsystem
 * it should not be possible to add more components.
 * 
 */
public class ComponentLookup {

    private final ComponentNode top;
    private final Map<String, ComponentNode> componentNodeDict = new HashMap<>();
    
    public ComponentLookup(ComponentNode top) {
        this.top = top;
        buildDict(top);
    }
    
    /**
     * Get the top ComponentNode, ie the tip of the tree structure.
     * 
     * @return The top ComponentNode.
     */
    public ComponentNode getTopComponentNode() {
        return top;
    }
    
    private void buildDict(ComponentNode<?> node) {
        addComponentNode(node);
        for (ComponentNode child : node.getChildren() ) {
            buildDict(child);
        }
    }
    
    private void addComponentNode(ComponentNode node) {
        componentNodeDict.put(node.getKey(), node);
//        if ( componentNodeObjectDict.containsKey(node.getComponent() )) {
//            throw new RuntimeException("Component "+node.getKey()+" has duplicate object");
//        }
    }
    
    /**
     * 
     * @param name the parent name.
     * @param node
     */
    public void addComponentNodeToLookup(String name, ComponentNode node) {
        ComponentNode parent = name == null ? top : getNodeByName(name);
        if (parent == null) {
            logger.warn("no node named : " + name + " : ignored");
            return;
        }
        parent.addChild(node);
        buildDict(node);
    }

    /**
     * return any object registered by the componentNodeDictionaries of the service.
     *
     * @param name
     * @return
     */
    public Object getComponentByName(String name) {
        ComponentNode node = componentNodeDict.get(name);
        if (node == null) {
            logger.log(Level.WARNING, "null reference associated to {0}", name);        
            return null;
        } else {
            return node.getComponent();
        }
    }

    /**
     * Check if the ComponentLookup already contains a ComponentNode for a given name.
     * 
     * @param name The name of the ComponentNode
     * @return true/false True if it exists, false otherwise.
     */
    public boolean containsComponentName(String name) {
        ComponentNode node = componentNodeDict.get(name);
        return node != null;
    }

    public ComponentNode getNodeByName(String name) {
        ComponentNode res = componentNodeDict.get(name);
        if (res == null) {
            logger.log(Level.WARNING, "no component named {0}", name);
        }
        return res;
    }

    public String getNameOfComponent(Object obj) {
        for (Map.Entry<String, ComponentNode> entry : componentNodeDict.entrySet()) {
            if (entry.getValue().getComponent() == obj) {
                return entry.getKey();
            }
        }
        return null;
    }
    
    /**
     * gets the children of a Component in the ComponentNode and filters by
     * Class. BEWARE: the map should be a LinkedHashMap!
     *
     * @param parentName
     * @param classFilter
     * @param <N> the type of node
     * @return
     * 
     * TO-DO: Must the return be a LinkedHashMap? Can it just be a Map?
     */
    public <N> LinkedHashMap<String, N> getChildren(String parentName, Class<N> classFilter) {
        ComponentNode current = getNodeByName(parentName);
        if (current != null) {
            List<ComponentNode> children = current.getChildren();
            LinkedHashMap<String, N> mapRes = new LinkedHashMap<>();
            for (ComponentNode child : children) {
                Object realValue = child.getComponent();
                if (classFilter == null || classFilter.isAssignableFrom(realValue.getClass())) {
                    N realChild = (N) realValue;
                    String key = child.getKey();
                    mapRes.put(key, realChild);
                }
            }
            return mapRes;
        } else {
            return null;
        }
    }

    /**
     * lists the children of a component.
     *
     * @param parentName
     * @param <Q> the type of node
     * @param klass
     * @return an empty list if there are no children
     */
    public <Q> List<Q> listChildren(String parentName, Class<Q> klass) {
        return getChildren(parentName, klass).values().stream()
                .collect(Collectors.toList());
    }

    /**
     * gets the Parent of a component in the <TT>ComponentNode</TT> tree. The
     * parent of the top node is named _EDEN_ and has a null value.
     *
     * @param componentName
     * @return
     */
    public Map.Entry<String, Object> getParent(String componentName) {
        ComponentNode current = getNodeByName(componentName);
        if (current != null) {
            //todo : use property isRootNode instead !
            ComponentNode parent = current.getParent();
            if (parent == null) {
                return new UniquePair<>("_EDEN_", null);
            }
            String key = parent.getKey();
            Object value = parent.getComponent();
            return new UniquePair<String, Object>(key, value);
        } else {
            return null;
        }
    }

    /**
     * Returns the path of a component in the tree of components.
     * @param componentName
     * @return 
     */
    public String getFullPathFor(String componentName) {
        String parentName = getParent(componentName).getKey();
        if ("_EDEN_".equals(parentName)) {
            return componentName;
        } else {
            return getFullPathFor(parentName) + "/" + componentName;
        }
    }
    
    public boolean containsComponent(Object obj) {
        return getNameOfComponent(obj) != null;
    }

}
