package org.lsst.ccs.description.groovy;

import groovy.util.BuilderSupport;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import java.util.Map;
import java.util.Set;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Class dedicated to building nodes from a groovy description file.
 * All it does is build DescriptiveNodes from the provided arguments.
 * It also sets the parent/child relationship as the tree is built.
 *
 * This class is overwritten by CCSBuilder in the startup package.
 * CCSBuilder is used directly in the groovy files.
 *
 * @author The LSST CCS Team
 */
class GroovyComponentBuilder extends BuilderSupport {

    private final Map<String,ComponentNode> internalDictionary = new HashMap<>();
        
    @Override
    public void setParent(Object o, Object o1) {
        GroovyComponentNode parent = (GroovyComponentNode) o;
        parent.addChild((GroovyComponentNode) o1);
    }

    @Override
    public Object createNode(Object name) {
        throw new IllegalStateException("class missing for node " + name);
    }

    @Override
    public Object createNode(Object name, Object value) {
        return createNode(name, Collections.EMPTY_MAP, value);
    }

    @Override
    public Object createNode(Object name, Map attributes) {
        throw new IllegalStateException("class missing for node " + name);
    }

    @Override
    public Object createNode(Object name, Map attributes, Object value) {
        String key;
        if(attributes.containsKey("name")) {
            // the component name is the value of the 'name' field
            key = (String)attributes.get("name");
            if (!key.equals(name)) {
                Logger.getLogger("org.lsst.ccs.description.groovy").warn(
                "the component name will be " + key + " instead of " + name +
                        ". This behaviour should not be used as it will be discontinued. See LSSTCCS-945"
                );
            }
        } else {
            // the component name is the node name
            key = (String)name;
        }
        if (getInternalDictionary().containsKey(key)) {
            throw new IllegalArgumentException("node with name : " + key + " already exists");
        }
        try {
            GroovyComponentNode node = new GroovyComponentNode(getCurrentNode(), key, attributes, (Class) value);
            getInternalDictionary().put(key, node);
            return node;
        } catch (Exception ex) {
            throw new RuntimeException("at node : " + name + ": " + ex.getMessage() , ex);
        }
    }

    protected GroovyComponentNode getCurrentNode() {
        GroovyComponentNode current = (GroovyComponentNode) getCurrent();
        return current;
    }
    
    protected Map<String,ComponentNode> getInternalDictionary() {
        return internalDictionary;
    }

    /**
     * A hook to allow nodes to be processed once they have had all of their children applied.
     * We use this hook to set the appropriate constructor to this node, given the list of attributes
     *
     * @param parent the parent of the node being processed
     * @param node the current node being processed
     */
    @Override
    public void nodeCompleted(Object parent, Object node) {
        GroovyComponentNode gNode = (GroovyComponentNode) node;

        try {
            gNode.attributes = (Map)resolveReferencesIn(gNode.attributes);
            Object component =  new GroovyConstructorInvoker().invokeConstructor(gNode.getCls(), gNode.getAttributes());            
            gNode.setComponent(component);
        } catch (Exception ex) {
            throw new RuntimeException("At node : " + gNode.getKey() + " : " + ex.getMessage(), ex);
        }
    }

    private Object resolveReferencesIn(Object obj) {
        if (obj == null) {
            return null;
        } else if(obj instanceof GroovyReferencePlaceHolder) {
            return resolveReference(null, (GroovyReferencePlaceHolder)obj);
        } else if(Map.class.isAssignableFrom(obj.getClass())) {
            for ( Object key : ((Map)obj).keySet() ) {
                Object value = ((Map)obj).get(key);
                if(value instanceof GroovyReferencePlaceHolder) {
                    ((Map)obj).put(key, resolveReference((String)key, (GroovyReferencePlaceHolder)value));
                } else {
                    ((Map)obj).put(key, resolveReferencesIn(value));
                }
            }
            return obj;
        } else if (obj.getClass().isArray()) {
            for(int i = 0; i<Array.getLength(obj); i++) {
                Array.set(obj, i, resolveReferencesIn(Array.get(obj,i)));
            }
            return obj;
        } else if (java.util.List.class.isAssignableFrom(obj.getClass())) {
            List resolvedList = new ArrayList<>();
            for (Object o : (List)obj) {
                resolvedList.add(resolveReferencesIn(o));
            }
            return resolvedList;
        } else if (java.util.Set.class.isAssignableFrom(obj.getClass())) {
            Set resolvedSet = new HashSet<>();
            for (Object o : (Set)obj) {
                resolvedSet.add(resolveReferencesIn(o));
            }
            return resolvedSet;
        } else {
            return obj;
        }
    }
    
    private Object resolveReference(String key, GroovyReferencePlaceHolder ph) {
        String ref = ph.getReference() == null ? key : ph.getReference();
        if (ref == null) {
            throw new RuntimeException("cannot infer reference name");
        }
        ComponentNode compNode = getInternalDictionary().get(ref);
        if ( compNode != null ) {               
            return compNode.getComponent();
        } else {
            throw new RuntimeException("null reference associated to " + ref);
        }
    }
    
}
