
package org.lsst.ccs.description;

import java.util.HashMap;

/**
 * A ComponentNode builder class. 
 * It provides utility functions to build and manipulate ComponentNodes.
 * 
 * @author The LSST CCS Team
 */
public class ComponentNodeBuilder {

    public static String alias;
    
    private static final HashMap<String,ComponentNodeFactory> factories = new HashMap<>();
    
    //Register a GroovyComponentNodeFactory and a ClassComponentNodeFactory for serialized objects with the ComponentNodeBuilder.    
    static {
        ComponentNodeBuilder.registerComponentNodeFactory(new org.lsst.ccs.description.groovy.GroovyComponentNodeFactory());
        ComponentNodeBuilder.registerComponentNodeFactory(new org.lsst.ccs.description.classname.ClassComponentNodeFactory());
    }
    
    public static interface ComponentNodeFactory {

        /**
         * Build a descriptive node from the provided description String.
         * The description String is implementation specific and should be
         * documented.
         * 
         * @param descriptionStr String containing description information 
         *                       necessary to build a DescriptiveNode.
         * @return The DescriptiveNode built from the description String.
         *                       
         */
        public ComponentLookup buildComponentNode(String descriptionStr);
        
        /**
         * Get the protocol for this GroovyComponentNodeFactory.
         * The protocol is used to identify the factory.
         * @return The factory's protocol
         */
        public String getProtocol();
       
        
    }
    
    private ComponentNodeBuilder() {
    }
    
    /**
     * Register a DescriptiveNodeFactory.
     * The protocol is used to register the factory and to pick it when needed.
     * 
     * @param factory The GroovyComponentNodeFactory to be registered.
     * 
     */
    public static void registerComponentNodeFactory(ComponentNodeFactory factory) {
        factories.put(factory.getProtocol(), factory);
    }
    
    /**
     * Build a Descriptive node given a fullDescription string of the form 
     * "protocol:descriptionStr".
     * Internally it picks the appropriate builder implementation to
     * create the corresponding DescriptiveNode.
     * 
     * @param fullDescription The full description to build a DescriptiveNode
     * @return The newly built DescriptiveNode.
     * 
     */
    public static ComponentLookup buildComponentNode(String fullDescription) {
        return buildComponentNode(fullDescription, null, null);
    }
    
    public static ComponentLookup buildComponentNode(String fullDescription, String alias, String startupConfig) {
        ComponentNodeBuilder.alias = alias;
        
        String descriptionProtocol = "groovy";
        String descriptionString = fullDescription;
        if (fullDescription.contains(":")) {
            descriptionProtocol = fullDescription.substring(0, fullDescription.indexOf(':'));
            descriptionString = fullDescription.substring(fullDescription.indexOf(':') + 1);
        }
        ComponentNodeFactory factory = factories.get(descriptionProtocol);
        if ( factory == null ) {
            throw new IllegalArgumentException("There is no registered ComponentNodeFactory for protocol "+descriptionProtocol);
        }
        ComponentLookup lookup = factory.buildComponentNode(descriptionString);
        lookup.getTopComponentNode().addTag("descriptionName", descriptionString);
        lookup.getTopComponentNode().addTag("startupConfig", startupConfig);
        return lookup;
    }
}
