package org.lsst.ccs.description;

import java.io.Serializable;
import java.util.*;
import java.util.logging.Logger;

/**
 * Abstract class for a node in the concrete implementation of a subsystem description.
 * It is a standard tree node structure containing in addition a key (the name
 * of the component) and the actual component that will be part of the modular 
 * structure of the subsystem.
 * Subclasses define how the component is built, by overriding {@code setComponent()}
 * 
 * @param <T> The particular instance of the ComponentNode. Either DescriptiveNode or EffectiveNode.
 *
 * @author The LSST CCS Team
 */
public class ComponentNode <T extends ComponentNode<T>> implements Serializable {
		
    private static final long serialVersionUID = 284121597899337112L;
    
    protected final static Logger logger = Logger.getLogger("org.lsst.ccs.description");
    
    /** The key of the node. */
    private String key ;

    /** The parent of the node. */
    private T parent;
    
    /** The list of children. */
    private final List<T> children = new ArrayList<>();

    /** The component that belongs to the node. It is lazy initialized. */
    private Object component;
    
    private String protocol;
    
    public ComponentNode(T parent, String name) {
        this(parent, name, null);
    }

    public ComponentNode(T parent, String name, Object component) {
        this.parent = parent;
        this.key = name;
        this.component = component;
    }

    public String getProtocol() {
        return protocol;
    }
    public void setProtocol(String protocol) {
        this.protocol = protocol;
    }
    
    public Object getComponent() {
        return component;
    }

    public void setComponent(Object component) {
        if ( this.component != null ) {
            throw new RuntimeException("Cannot set component twice.");
        }
        this.component = component;
    }

    public String getKey() {
        return key;
    }

    /**
     * @return the list of child nodes, <TT>null</TT> if there aren't any
     */
    public List<T> getChildren() {
        return Collections.unmodifiableList(children);
    }
    
    /**
     * adds a ComponentNode as a child of the current node
     *
     * @param child to be added to the node
     * @throws NullPointerException if child is null
     */
    public void addChild(T child) throws NullPointerException {
        // is this a root or a pseudo-root
        if (child.getParent() != null && child.getParent() != this) {
            throw new IllegalArgumentException("child added but is part of another node tree");
        }
        this.children.add(child);
    }

    @Override
    public String toString() {
        String valueString ;
        try {
           valueString = String.valueOf(component)  ;
        } catch (Exception exc) {
           valueString = "can't trace value [" + exc +"] "  ;
        }
        return String.valueOf(key) + "("
                + valueString + ") {"
                + String.valueOf(children) + "}";
    }
    
    public T getParent() {
        return parent;
    }
}
