package org.lsst.ccs.description.groovy;

import groovy.util.BuilderSupport;
import java.math.BigDecimal;
import java.util.Collections;

import java.util.Map;
import java.util.Set;
import org.codehaus.groovy.runtime.GStringImpl;
import org.lsst.ccs.description.DescriptiveNode;
import org.lsst.gruth.jutils.MapArgs;

/**
 * Class dedicated to building nodes from a groovy description file.
 * All it does is build DescriptiveNodes from the provided arguments.
 * It also sets the parent/child relationship as the tree is built.
 *
 * This class is overwritten by CCSBuilder in the startup package.
 * CCSBuilder is used directly in the groovy files.
 *
 * @author The LSST CCS Team
 */
public class ComponentBuilder extends BuilderSupport {

    @Override
    public void setParent(Object o, Object o1) {
        DescriptiveNode parent = (DescriptiveNode) o;
        parent.addChild((DescriptiveNode) o1);
    }

    @Override
    public Object createNode(Object name) {
        throw new IllegalStateException("class missing for node " + name);
    }

    @Override
    public Object createNode(Object name, Object value) {
        return createNode(name, Collections.EMPTY_MAP, value);
    }

    @Override
    public Object createNode(Object name, Map attributes) {
        throw new IllegalStateException("class missing for node " + name);
    }

    @Override
    public Object createNode(Object name, Map attributes, Object value) {
        // Big Decimal and GStringImpl special case
        Set<Map.Entry> entries = attributes.entrySet();
        for (Map.Entry entry : entries) {
            Object val = entry.getValue();
            if (val instanceof GStringImpl) {
                attributes.put(entry.getKey(), val.toString());
            } else if(val instanceof BigDecimal) {
                attributes.put(entry.getKey(), ((BigDecimal)val).doubleValue());
            }
        }
        return new DescriptiveNode(getCurrentNode(), (String)name, attributes, (Class)value);
    }
    
    protected DescriptiveNode getCurrentNode() {
        DescriptiveNode current = (DescriptiveNode) getCurrent();
        return current;
    }
    
    /**
     * A hook to allow nodes to be processed once they have had all of their children applied.
     * We use this hook to set the appropriate constructor to this node, given the list of attributes
     * @param parent the parent of the node being processed
     * @param node the current node being processed
     */
    @Override
    public void nodeCompleted(Object parent, Object node) {
        if (((DescriptiveNode)node).getAttributes() instanceof MapArgs) {
            ((DescriptiveNode)node).setConstructor();
        }
    }
}
