package org.lsst.ccs.description.groovy;

import groovy.lang.Binding;
import groovy.lang.GroovyShell;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import org.lsst.ccs.utilities.logging.Logger;
import org.codehaus.groovy.control.CompilationFailedException;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.description.ComponentNodeBuilder;
import org.lsst.ccs.description.DescriptiveNode;

/**
 * The ComponentNodeFactory for Groovy.
 *
 * The LSST CCS Team.
 */
public class ComponentNodeFactory implements ComponentNodeBuilder.ComponentNodeFactory {

    @Override
    public DescriptiveNode buildDescriptiveNode(String descriptionStr) {
        //Some checks on the content of descriptionStr
        if (descriptionStr.contains("/")) {
            throw new IllegalArgumentException("description files must be located at the root of a resource directory");
        }
        if (descriptionStr.endsWith(".groovy")) {
            Logger.getLogger("org.lsst.ccs.description.groovy").warn("Description file extension has to be skipped in the --description argument");
        }

        if (!descriptionStr.endsWith(".groovy")) {
            descriptionStr += ".groovy";
        }

        InputStream is = BootstrapResourceUtils.getBootstrapResource(descriptionStr, ComponentNodeFactory.class);

        if (is == null) {
            throw new IllegalArgumentException("Could not find resource for " + descriptionStr);
        }
        try {
            GroovyShell shell = shellFactory();
            InputStreamReader reader = new InputStreamReader(is, "ISO-8859-1");
            Object res = shell.evaluate(reader);
            if (!(res instanceof DescriptiveNode)) {
                throw new IllegalArgumentException("The described object is not a DescriptiveNode: " + res.getClass());
            }
            DescriptiveNode node = (DescriptiveNode)res;
            node.setProtocol(getProtocol());
            return node;
        } catch (UnsupportedEncodingException | CompilationFailedException | IllegalArgumentException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getProtocol() {
        return "groovy";
    }
    static GroovyShell shellFactory() {
        Binding binding = new Binding();

        ComponentBuilder builder = new ComponentBuilder();
        binding.setProperty("builder", builder);

        GroovyShell shell = new GroovyShell(binding);
        return shell;
    }

}
