package org.lsst.ccs.description;

import java.util.logging.Logger;
import java.util.logging.Level;

/**
 * used to trace execution of gru.
 * Many errors when writing a test or a node tree are just
 * fairly hard to find: use the tracer to understand what is wrong in
 * the interpretation of your code.
 * <P>
 *  Gru internal code should use <TT>Tracer.trace</TT> at critical points.
 *  <P/>
 *  Gru users should set a property (or environment variable)
 *  with name 'gruth.traces' the value is a comma separated list
 *  of <TT>Tracer</TT> values.
 *
 * @author bamade
 */
 // Date: 13/09/12
 
public enum Tracer {
    NODE_BUILD, NODE_EVAL, NODE_MODIF, NODE_CALLS, TEST_BUILD, TEST_EVAL,
    PRE, POST, XPECT;

    /**
     * each enum member has a corresponding bit in a bit pattern.
     * this bit is build according to the ordinal of the value.
     *
     */
    int bitPosition;

    /**
     * according to the value of 'gruth.traces' this mask is use
     * to pick up the corresponding calls to <TT>trace</TT>
     */
    static final int traceMask ;
    static {
        int mask =0 ;
        String traceSpecs = System.getenv("gruth.traces");
        if(traceSpecs == null) {
            traceSpecs = System.getProperty("gruth.traces");
        }
        if(traceSpecs!= null) {
            String[] specs = traceSpecs.split(",");
            for(String spec: specs) {
                Tracer tracer = Tracer.valueOf(spec);
                //todo OR equals
                mask |=  tracer.bitPosition;
            }
        }
        traceMask = mask ;
    }

    Tracer()  {
       bitPosition = 1 << this.ordinal();
    }

    static Logger logger =  Logger.getLogger("org.lsst.gruth");

    public static void trace (Tracer tracer, Object... args) {
        if( (traceMask & tracer.bitPosition) == 1) {
            StringBuilder builder = new StringBuilder();
           for(Object obj : args) {
               if(obj == null ){
                   builder.append(" %no trace (null value)% ");
               }
               try {
                    builder.append(" $obj");
               } catch (Exception exc) {
                   builder.append(" %no trace (null value in toString() method)% ");
               }
           }

            Level level = logger.getLevel();
            if(level != null && level.intValue() <= Level.INFO.intValue()  ) {
               logger.log(level, builder.toString());
            } else {
                System.out.println(builder.toString());
            }
        }

    }

}