package org.lsst.ccs.description;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Map;
import org.lsst.ccs.description.ComponentNode.IndirectHash;
import org.lsst.gruth.jutils.EmptyParmException;
import org.lsst.gruth.jutils.MapArgs;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Builds <TT>EffectiveNode</TT> structure of actual objects out of <TT>DescriptiveNode</TT> structure.
 * <P/>
 * The tree is built "from the ground up" (starting from the "leaves").
 * @author bamade
 *
 */
public class ComponentFactory {
    
    private static Logger log = Logger.getLogger("org.lsst.ccs.description");
    public DescriptiveNode topNode;

    /**
     * creates a Factory that will operate from a subtree of this topNode
     * @ImplNote
     * Keep this constructor and deprecate buildObjectTreeFrom
     * @param topNode
     */
    public ComponentFactory(DescriptiveNode topNode) {
        this.topNode = topNode;
    }

    EffectiveNode build() {
        EffectiveNode effectiveNode = buildObjectTreeFrom( topNode);
        return effectiveNode;
    }
    
    
    

    /**
     * builds an <TT>EffectiveNode</TT> tree from a DescriptiveNode
     * @ImplNote
     * todo: deprecate and build from the topNode only.
     * @param node
     * @return
     */
    EffectiveNode buildObjectTreeFrom(DescriptiveNode node) {
        IndirectMap<String, ComponentNode> map = new IndirectHash<>();
        EffectiveNode res = buildFrom(map, node);
        return res;
    }

    /**
     * builds an EffectiveNode from a Descriptive Node.
     * the build is operating "from the ground up" (children are build before parent).
     *
     * @param registry a Map common to all Nodes. This means that the build of a Node can use a reference to an Object build during child node build (or before)
     * @param node
     * @return
     */
    //ComponentNode buildFrom(IndirectMap<String, Object> registry, ComponentNode node) {
    EffectiveNode buildFrom(IndirectMap<String, ComponentNode> registry, DescriptiveNode node) {
        //start from children
        ArrayList<EffectiveNode> newChildren = null;
        ArrayList<DescriptiveNode> children = node.getChildren();
        if (children != null) {
            int nbChildren = children.size();
            if (nbChildren > 0) {
                newChildren = new ArrayList<EffectiveNode>(nbChildren);
                for (DescriptiveNode child: children) {
                    newChildren.add(buildFrom(registry, child));
                }
            }
        }
        // gets the key
        String key = node.getKey();

        // gets the parameters of constructor
        Object actualNodeValue = null;
        Map rawArgs = node.getAttributes();
        if (rawArgs instanceof MapArgs) {
            MapArgs originArgs = (MapArgs) rawArgs;
            // Do we still need this ?
            if ( originArgs.isSafe()) {
                // means that we can't use the default groovy strategy : null ctor + map
                // builds the object value
                //println "doing constructor call $clazz"
                node.setConstructor();
                Constructor c = node.getConstructor();
                Object[] arrayArgs = originArgs.asFullArray(registry);
                actualNodeValue = createNewInstance(registry, key, c, arrayArgs);
            } else {
                throw new EmptyParmException("arguments list with empty values");
            }
        } else if (rawArgs instanceof Map) {
            try {
                // First try : looking for a constructor that matches the map values types in order
                node.setConstructor();
                Constructor c  = node.getConstructor();
                Object[] arrayArgs = rawArgs.values().toArray();
                actualNodeValue = createNewInstance(registry, key, c, arrayArgs);
            } catch (Exception ex1) {
                /**
                 * Second try : groovy strategy : looking for an empty constructor, and
                 * passing a map the keys of which are the names of the fields.
                 */
                try {
                    actualNodeValue = new GroovyConstructorInvoker().invokeEmptyConstructor(node.getRealValue(), rawArgs);
                } catch (Exception ex2) {
                    throw new RuntimeException(ex1.getMessage() + "\n" + ex2.getMessage(), ex2);
                }
            }
        } else {
            throw new IllegalArgumentException("node with arguments that are not passed as a map " + rawArgs.getClass());
        }
        //TODO : else
        // builds the object value
        // builds a new Component Node which should be an Effective Node
        //ComponentNode res = ComponentNode.createOrphan(registry, node.getName(), key, actualNodeValue)
        //todo: add the properties from the descriptiveNode ?
        EffectiveNode res = new EffectiveNode(registry, node.getName(), key, actualNodeValue,node);
        // register children
        if (newChildren != null) {
            res.adoptOrphans(registry, newChildren, res);
        }
        return res;

    }
    
    public static Object createNewInstance(IndirectMap registry, String key, Constructor c, Object[] args) {
        Object[] newArray = args;
        try {
            if (c.isVarArgs()) {
                // Constructor is a varargs : varargs arguments have to be grouped together in an array.
                int varargsStartIndex = c.getParameterCount() - 1;
                int varargsLength = args.length - varargsStartIndex;
                Class varargsClass = c.getParameterTypes()[varargsStartIndex];
                Object[] varargs = new Object[varargsLength];
                for (int i = 0; i<varargsLength; i++) {
                    varargs[i] = args[varargsStartIndex+i];
                }
                newArray = Arrays.copyOf(args, c.getParameterCount());
                newArray[newArray.length-1]= Arrays.copyOf(varargs, varargs.length, varargsClass);
            }
            c.setAccessible(true);
            return c.newInstance(newArray);
        } catch(Exception ex) {
            String s = "(";
            for (int i = 0 ; i <newArray.length; i++) {
                s+= args[i].getClass(); 
                if ( i != newArray.length - 1 ) {
                    s += ", ";
                }
            }
            s += ")";
            throw new ConstructorInvocationException("error invoking constructor " + c + " for node " + key + " and arguments of type : " + s + "\n"+ex.getMessage(), ex);
        }
    } 
}

