package org.lsst.ccs.description;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.utilities.structs.UniquePair;

/**
 * This interface helps find component of the susbsystem
 * as they are defined in the description "tree"
 * @author bamade
 */
// Date: 17/11/12

public class ComponentLookupService {
    private final EffectiveNode top ;
    
    public ComponentLookupService(EffectiveNode top) {
        this.top = top;
    }
    
    /**
     * return any object registered by the dictionaries of the service.
     * Though most are <TT>Configurable</TT> objects of the corresponding <TT>EffectiveNode</TT>
     * In some case a use can register other objects through the <TT>alias</TT> method method
     * @param name
     * @return
     */
    public Object getComponentByName(String name) {
        Object res = top.getIndirect(name);
        return res ;
    }

    /**
     * gets the children of a Component in the EffectiveNode and filters by Class.
     * BEWARE: the map should be a LinkedHashMap!
     * @param parentName
     * @param classFilter
     * @param <T>
     * @return
     */
    public <T> LinkedHashMap<String,T> getChildren(String parentName, Class<T> classFilter) {
        EffectiveNode current = top.getNodeByName(parentName);
        ArrayList<EffectiveNode> children =  current.getChildren();
        LinkedHashMap<String, T> mapRes = new LinkedHashMap<>() ;
        for(EffectiveNode child: children) {
            Object realValue = child.getRealValue();
            if(classFilter == null || classFilter.isAssignableFrom(realValue.getClass())) {
                T realChild = (T) realValue ;
                String key = child.getKey() ;
                mapRes.put(key, realChild) ;
            }
        }
        return mapRes;
    }

    /**
     *  lists the children of a component.
     * @param parentName
     * @param <T>
     * @param klass
     * @return an empty list if there are no children
     */
    public <T> List<T> listChildren(String parentName, Class<T> klass) {
        return getChildren(parentName, klass).values().stream()
                .collect(Collectors.toList());
    }

    /**
     * gets the Parent of a component in the <TT>EffectiveNode</TT> tree.
     * The parent of the top node is named _EDEN_ and has a null value.
     * @param componentName
     * @return
     */
    public Map.Entry<String,Object> getParent(String componentName) {
        EffectiveNode current = top.getNodeByName(componentName);
        //todo : use property isRootNode instead !
        EffectiveNode parent = current.getParent();
        if (parent == null) {
            return new UniquePair<>("_EDEN_", null);
        }
        String key = parent.getKey() ;
        Object value = parent.getRealValue() ;
        return  new UniquePair<String, Object>(key, value) ;
    }

    public String getFullPathFor(String componentName) {
         String parentName = getParent(componentName).getKey();
        if ( "_EDEN_".equals(parentName) ) {
            return componentName;
        } else {
            return getFullPathFor(parentName)+"/"+componentName;
        }
    }
    
    
    /**
     * Tree walking starting from the top node. If the class of the node's inner
     * object is assignable to the specified class, the pre and post consumers
     * are applied to it.
     * @param <T>
     * @param startingNodeName the name of the node to start the procedure from.
     * @param klass
     * @param pre the procedure to apply in pre order
     * @param post the procedure to apply in post order
     */
    public <T> void proceduralWalk(String startingNodeName, Class<T> klass, Consumer<T> pre, Consumer<T> post) {
        EffectiveNode startFromNode = (startingNodeName == null)?top : top.getNodeByName(startingNodeName);
        startFromNode.proceduralWalk(klass, pre, post);
    }
    
    /**
     * Tree walking starting from the top node. Consumers are targeted to the node 
     * itself.
     * @param startingNodeName
     * @param pre
     * @param post 
     */
    public void proceduralNodeWalk(String startingNodeName, Consumer<EffectiveNode> pre, Consumer<EffectiveNode> post) {
        EffectiveNode startFromNode = (startingNodeName == null)?top : top.getNodeByName(startingNodeName);
        startFromNode.proceduralNodeWalk(pre, post);
    }
    
    /**
     * Walks the components tree recursively, applying rules of <TT>TreeWalkerDiag</TT> (the walk
     * can be interrupted by HANDLING_CHILDREN (the code is not invoked on children)
     * or STOP (the code abruptly ends).
     * REVIEW : proceduralWalk and proceduralNodeWalk are particular cases of treeWalk.
     * @param <T>
     * @param startingNodeName
     * @param klass
     * @param func
     * @param post
     */
    public <T> void treeWalk(String startingNodeName, Class<T> klass, Function<T,TreeWalkerDiag> func, Consumer<T> post) {
        EffectiveNode startFromNode = (startingNodeName == null)?top : top.getNodeByName(startingNodeName);
        startFromNode.treeWalk(klass, func, post);
    }
}
