package org.lsst.ccs.description;

import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.framework.TreeWalkerDiag;

/**
 * A node in the concrete implementation of a subsystem description.
 * There are two possible instances of ComponentNode:
 *  - DescriptiveNodes are first built with the information needed to create an 
 * actual implementation of the object in the tree
 *  - EffectiveNodes are then built from a corresponding DescriptiveNode and 
 * contain the actual instance of the objects in described the provided description.
 * 
 * @param <T> The particular instance of the ComponentNode. Either DescriptiveNode or EffectiveNode.
 * @param <Z> The object contained in the ComponentNode as derived from the description.
 * For a DescriptiveNode it's the Class of the object to be built, for an EffectiveNode
 * it's the instance of the created object.
 *
 * @author The LSST CCS Team
 */
class ComponentNode <T extends ComponentNode<T,Z>, Z> implements Serializable {

    protected final static Logger logger = Logger.getLogger("org.lsst.ccs.description");

    /**
     * this class supposes that the creation of the dictionary is made top down.
     * @param <K>
     * @param <V>
     */
    static class IndirectHash<K, V> extends LinkedHashMap<K, V> implements IndirectMap<K, V> {
        @Override
        public Object getIndirect(Object key) {
            Object obj = get(key);
            if (obj instanceof ComponentNode) {
                obj = ((ComponentNode) obj).getRealValue();
            }
            //Tracer.trace(Tracer.NODE_BUILD, "reference to ", key, " ", obj);
	    // bizarre bug!
            Tracer.trace(Tracer.NODE_BUILD, new Object[] {"reference to ", key, " ", obj});
            if (obj == null) {
                //TODO : fire error?
                logger.log(Level.WARNING, "null reference associated to {0}", key);
            }
            return obj;
        }
    }

    /**
     * A dictionary of all nodes in the tree. name duplications should be detected
     */
    protected /*@NotNull*/ IndirectMap<String, ComponentNode> nodeDict;
    /**
     * the key of the node
     */
    protected /*NotNull*/ String key;

    // standard NODE functions changed from Node
    protected T parent;
    protected ArrayList<T> children = new ArrayList<>();

    /**
     * todo: righ now key and name should be the same .. if separate one must change the use of the dictionary
     *
     */
    protected String name;


    /**
     * for DescriptiveNode the Class of the future object
     * for an EffectiveNode the actual object
     */
    protected Z realValue;
    /**
     * for a DescriptiveNode the description of parameters
     * not used for EffectiveNode (but might be with a copy of parameters description)
     */
    protected Map attributes;


    public IndirectMap getNodeDict() {
        return this.nodeDict;
    }

    @Deprecated
    public void setNodeDict(IndirectMap<String, ComponentNode> registry) {
        this.nodeDict = registry;
    }

    /**
     * prepare the setDict call
     * <B>Beware</B> should be used only in top-down build of the tree
     * @param parent
     * @param name
     * @param attributes
     */
    protected void checkDict(T parent, String name, Map attributes) {
        IndirectMap<String, ComponentNode> dict;
        // todo check if not used by EffectiveNode
        if (parent == null) {
            dict = new IndirectHash<>();
        } else {
            dict = parent.getNodeDict();
        }
        setDict(dict, name, attributes);
    }

    /**
     * set the dictionary of the node and set the entry of this Node in the Dictionary
     * @param registry
     * @param name
     * @param attributes
     */
    protected void setDict(IndirectMap<String, ComponentNode> registry, String name, Map attributes) {
        String keyStr = name;

        /** 
         * If "name" is part of the attributes map, it is used to register the 
         * component, instead of the actual name provided in groovy.
         * TODO: WHY do we do this?
         */
        if (attributes != null) {
            Object valName = attributes.get("name");
            if (valName != null) {
                keyStr = String.valueOf(valName);
            }
        } 
        this.key = keyStr;
        this.name = keyStr;
        
        // Review : This is the only place we add a new entry to the dictionary :
        // nodeDict is a Map of
        registry.put(keyStr, this);
        this.nodeDict = registry;
    }


    @Deprecated
     ComponentNode(T parent, String name) {
        this(parent, name, (Map) null, null);
        System.out.println("*** Why are we using this constructor. "+name);
    }

    ComponentNode(T parent, String name, Z value) {
        this(parent, name, (Map) null, value);
    }

    @Deprecated
     ComponentNode(T parent, String name, Map attributes) {
        this(parent, name, attributes, null);
        System.out.println("*** Why are we using this constructor. "+name+" "+attributes);
    }

    ComponentNode(T parent, String name, Map attributes, Z value) {
        this.parent = parent;
        this.attributes = attributes;
        this.realValue = value;
        checkDict(parent, name, attributes);
    }

    /**
     * this constructor to be used by EffectiveNode (does not use checkDict!)
     * @param registry
     * @param name
     * @param key
     * @param attributes
     * @param value
     */
    protected ComponentNode(IndirectMap<String, ComponentNode> registry, String name, String key, Map attributes, Z value) {
        this.attributes = attributes;
        this.realValue = value;
        //TODO: fix the ambigity between name and kay when building "orphan" this is a feature bug!
        //setDict(registry, key, attributes);
        setDict(registry, name, attributes);
    }


    public String getName() {
        return name;
    }

    public Z getRealValue() {
        return realValue;
    }

    public  ComponentNode getNodeByName(String name) {
        return this.nodeDict.get(name);
    }

    public Object getIndirect(Object key) {
        Object node = this.nodeDict.get(key);
        if (node instanceof ComponentNode) {
            return ((ComponentNode) node).getRealValue();
        }
        return node;
    }

    public Iterator<String> getAllKeys() {
        return nodeDict.keySet().iterator();
    }

    public Iterable<String> getAllNames() {
        Collection<ComponentNode>col = nodeDict.values() ;
        HashSet<String> res = new HashSet<>() ;
        for(Object obj : col) {
            if(obj instanceof ComponentNode) {
               res.add(((ComponentNode)obj).getName())  ;
            }
        }
        return res ;
    }


    public Map getAttributes() {
        return attributes;
    }

    public String getKey() {
        return key;
    }


    /**
     * @return the list of child nodes, <TT>null</TT> if there aren't any
     */
    //todo: generalize the type of children
    public /*Nullable*/ ArrayList<T> getChildren() {
        return children;
    }

    /**
     * adds a ComponentNode as a child of the current node
     *
     * @param child to be added to the node
     * @throws NullPointerException if child is null
     */
    //todo: rewrite to distinguish the case of DescriptiveNode and EffectiveNode
    public  void    addChild(/*NotNull*/ T child) throws NullPointerException {
        // is this a root or a pseudo-root
        if (child.parent == null) {
            //when building from the ground up
            //todo: clumsy dictionary build: the child dictionary is replaced by a new one!
            if (child.nodeDict != null && child.nodeDict != this.nodeDict) {
                this.nodeDict.putAll(child.getNodeDict());
                child.nodeDict = this.nodeDict;
            } else {//add child to dict
                child.setDict(this.nodeDict, child.getName(), child.getAttributes());
            }
        } else if (child.parent == this) {
            //System.out.println(" parent already set");
            //return ;
        } else {
            throw new IllegalArgumentException("child added but is part of another node tree");
        }
        //System.out.println("adding " + child.getName() + " to " + this.getName());
        this.children.add(child);

    }

    @Override
    public /*NotNull*/ String toString() {
        String valueString ;
        try {
           valueString = String.valueOf(realValue)  ;
        } catch (Exception exc) {
           valueString = "can't trace value [" + exc +"] "  ;
        }
        return String.valueOf(name) + "("
                + valueString + ","
                + String.valueOf(attributes) + ") {"
                + String.valueOf(children) + "}";
    }

    public T getParent() {
        return parent;
    }


    public <V> void proceduralWalk(Class<V> klass, Consumer<V> pre, Consumer<V> post) {
        if(pre != null) {
            Object component = getRealValue();
            if (klass.isAssignableFrom(component.getClass())){
                pre.accept((V)component);
            }
        }
        for(ComponentNode child: this.getChildren()) {
            child.proceduralWalk(klass, pre, post);
        }
        if(post != null) {
            Object component = getRealValue();
            if (klass.isAssignableFrom(component.getClass())){
                post.accept((V)component);
            }
        }
    }
    
    
    public void proceduralNodeWalk(Consumer<T> pre, Consumer<T> post) {
        if(pre != null) {
            pre.accept((T)this);
        }
        for(ComponentNode child: this.getChildren()) {
            child.proceduralNodeWalk(pre, post);
        }
        if(post != null) {
            post.accept((T)this);
        }
    }
    
    /**
     *  Walks the components tree recursively, applying rules of <TT>TreeWalkerDiag</TT> (the walk
     *  can be interrupted by HANDLING_CHILDREN (the code is not invoked on children)
     *  or STOP (the code abruptly ends)
     * @param func
     * @param post
     * @return a {@code TreeWalkerDiag} indicating how the next Configurable
     * should be handled
     */
    public <V> TreeWalkerDiag treeWalk(Class<V> klass, Function<V, TreeWalkerDiag> func, Consumer<V> post) {
        TreeWalkerDiag diag = TreeWalkerDiag.GO ;
        if(func != null) {
            Object component = getRealValue();
            if (klass.isAssignableFrom(component.getClass())){
                diag = func.apply((V)component);
            }
        }
        
        switch (diag) {
            case STOP:
                return diag;
            case HANDLING_CHILDREN:
                return diag;// should return GO
            case GO:
            default:
                for (ComponentNode child : this.getChildren()) {
                    TreeWalkerDiag childRes = child.treeWalk(klass, func, post) ;
                    if (childRes.equals(TreeWalkerDiag.STOP)) return childRes;
                }
                if(post != null) {
                    Object component = getRealValue();
                    if (klass.isAssignableFrom(component.getClass())){
                        post.accept((V)this);
                    }
                }
                return diag;
        }
    }
    
}
