package org.lsst.gruth.jutils;

import groovy.util.Eval;
import org.apache.commons.beanutils.MethodUtils;
import org.lsst.gruth.types.IncompatibleTypeException;
import org.lsst.gruth.types.GArray;
import org.lsst.gruth.types.GStruct;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Map;

/**
 * A set of static methods to check constraints.
 * @implNote
 * since this class depends heavily on Groovy it must be moved to gruth package.
 * @author bamade
 */
// Date: 31/05/12

public class Constraints {
    
    private Constraints() {}
    
    /**
     * A specific code to control a value.
     * instance of this class are supposed to have a no-arg constructor.
     */
    public static interface Controler {
        /**
         * should transform the String argument as a valid value and check it.
         * @param value
         * @return the correct object or throws an exception
         */
        public Object control(String value);
    }
    
    /**
     * Builds an object out of a type and a String representation and tests it against
     * the specified constraints.
     * @param type the expected type of the object
     * @param value the String representation of the object
     * @param constraints the constraints the final object must satisfy
     * @return an object that satisfies the constraints
     */
    public static Object check (String type, String value , String constraints) {
        
        type = type.trim() ;
        //TODO: whatif value == null or "".equals(value) ?
        value = value.trim() ;
        Object realValue = buildObject(type, value);
        
        if (constraints == null ) return realValue;
        constraints = constraints.trim();
        
        if ("". equals(constraints)) return realValue;
        
        // There is a constraint
        String[] split = constraints.split("\\.\\.");
        if (split.length == 2) {
            // constraint is a range
            return checkRange(realValue.getClass(), value, split[0], split[1]);
        }
        try {
            if(constraints.startsWith("##")) {
                // constraint is a predicate
                String[] splits = constraints.split("##") ;
                String argName = splits[1] ;
                String expr = splits[2] ;
                Object checks = Eval.me(argName, realValue, expr) ;
                if(checks  instanceof  Boolean) {
                    boolean doIt = ((Boolean)checks).booleanValue();
                    if(doIt) {
                        return realValue ;
                    }
                }
                throw new IllegalArgumentException(" predicate " + expr + " FAILED") ;
                
            } else {
                // constraint is a controler
                Controler controler = (Controler) Class.forName(constraints).newInstance();
                return checkWithControler(value, controler);
            }
        } catch (Exception exc) {
            throw new IllegalArgumentException("constraint :" + constraints + " " + exc);
        }
    }
    
    /**
     * Builds an object out of a type and a String representation.
     * @param type the String representation of the type
     * @param value the String representation of the object
     * @return an object of type {@code type}
     */
    public static Object buildObject(String type, String value){
        Object realValue = null ;
        // TODO: following code to be separated to be used by predicates (see below)
        Class thatClass = null;
        try {
            // hack : java.lang.Character does not have a valueOf(String) method
            if("java.lang.Character".equals(type)) {
                type = "java.lang.String" ;
            }
            thatClass = Class.forName(type);
            if(thatClass.isArray()) {
                realValue = GArray.valueOf(value, thatClass.getName()) ;
            } else if (null!= SStructParm.structClassUsesList(thatClass)) {
                realValue = GStruct.valueOf(thatClass, value) ;
            } else {
                realValue = MethodUtils.invokeStaticMethod(thatClass, "valueOf", value);
            }
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException(e);
        } catch (NoSuchMethodException e) {
            if( Map.class.isAssignableFrom(thatClass)||
                    List.class.isAssignableFrom(thatClass)){
                try {
                    realValue = Eval.me(value) ;
                } catch (Exception exc) {
                    throw new IllegalArgumentException(exc);
                }
                Class realClass = realValue.getClass();
                if(! thatClass.isAssignableFrom(realClass)) {
                    throw new IncompatibleTypeException(thatClass, realClass);
                }
            } else {
                throw new IllegalArgumentException(e);
            }
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(e);
        } catch (InvocationTargetException e) {
            throw new IllegalArgumentException(e);
        } catch (Exception exc) {
            return null ;
        }
        return realValue ;
        
    }
    
    /**
     * checks if the value of type  is between lower and upper (included).
     * @param thatClass a Class. the class <B>should</B> have a static factory method <TT>valueOf(String val)</TT>
     *             (but <TT>java.lang.Character</TT> is still ok)
     * @param value
     * @param lower
     * @param upper
     * @return the real value
     */
    private static Object checkRange(Class thatClass, String value, String lower, String upper) {
        Object realValue = null ;
        try {
            realValue = MethodUtils.invokeStaticMethod(thatClass, "valueOf", value);
            Object realUpper = MethodUtils.invokeStaticMethod(thatClass, "valueOf", upper);
            Object realLower = MethodUtils.invokeStaticMethod(thatClass, "valueOf", lower);
            Comparable compValue = (Comparable) realValue ;
            if((compValue.compareTo(realLower) >= 0) && (compValue.compareTo(realUpper) <= 0)) {
                return realValue ;
            } else {
                throw new IllegalArgumentException(realValue + "not in range [" +realLower + ".." + realUpper+"]") ;
            }
        } catch (Exception exc) {
            throw new IllegalArgumentException("range " + lower + ".." + upper +
                    " and value :" + value + " raises :" + exc);
        }
        
    }
    
    private static Object checkWithControler(String value, Controler controler) {
        return controler.control(value);
    }
}
