package org.lsst.gruth.jutils;

import java.text.MessageFormat;
import java.text.ParseException;

/**
 * This class helps creating Strings with incremental number value in it
 * (such as incremental fileNames).
 * <P/>
 * Instances must be created with a simplified pattern where the character
 * '#' is the place holder for a number (integer).
 * <P/>
 * Then other Strings containing the same pattern can be submitted
 * with method <TT>submit</TT>  and in the end the <TT>getNewString()</TT> method will yield
 * the next numbered String.
 * <P/>
 * So for example :
 * <PRE>
 *      String pattern = "aName#.something";
 *  NumberedStringGenerator generator = new NumberedStringGenerator(pattern) ;
 *  System.out.println(" first :" + generator.getNewString());
 *  // will yield : "aName0.something"
 *  generator.submit("aName0.something") ;
 *  generator.submit("aName11.something") ;
 *  generator.submit("aName1.something") ;
 *  System.out.println(" max :" + generator.getNewString());
 *  // will yield : "aName12.something"
 * </PRE>
 *
 * @author bamade
 */
// Date: 29/08/12

public class NumberedStringGenerator {
    private MessageFormat messageFormat ;
    private int max =-1;
    private int min = -1 ;

    /**
     * creates a generator that will compare Strings and will yield the
     * next incremented String.
     * @param simplePattern a String with a '#' character as number placeholder
     * @throws  IllegalArgumentException if no '#' character in argument
     */
    public NumberedStringGenerator(String simplePattern) {
        if(-1 == simplePattern.indexOf("#")) {
            throw new IllegalArgumentException("no # in simplePattern String") ;
        }
        String realPattern = simplePattern.replace("#", "{0,number,integer}");
        messageFormat = new MessageFormat(realPattern) ;
    }

    /**
     *
     * @param candidate any String that possibly matches the initial pattern
     * @return true if matched false otherwise
     */
    public boolean submit(String candidate) {
        try {
            Object[] parsed = messageFormat.parse(candidate) ;
            int val = ((Long) parsed[0]).intValue();
            if(val > max) {
                max =val ;
            }
            if(min == -1) { min = val ;}
            else if (val < min ) {
                min = val ;
            }
            return true ;
        } catch (ParseException e) {
           return false ;
        }
    }

    /**
     *
     * @return a new String that has the highest number immediately after the highest matched String
     */
    public String getNewString() {
        return messageFormat.format(new Integer[] {max+1});
    }

    /**
     *
     * @return the highest ranking submitted name
     */
    public String getMaxString() {
        return messageFormat.format(new Integer[] {max});
    }

    /**
     *
     * @return the max number found in submitted Strings; -1 if no string matched
     */
    public int getMax() {
        return max ;
    }

    /**
     *
     * @return the smallest numbered String being submitted or null if none matched
     */
    public String getMinString() {
        if(min == -1) return null;
        return messageFormat.format(new Integer[] {min});
    }


    /**
     *
     * @param number
     * @return a formatted String using the parameter
     */
    public String formatWith(int number) {
        return messageFormat.format(new Integer[] {number});
    }


}
