package org.lsst.gruth.jutils;


import org.lsst.gruth.types.TypeUtils;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;

/**
 * An hybrid object: it is  an immutable Map<String,Object>
 * but has some behaviour of an immutable List.
 * It could be used to specify arguments to a constructor or method call.
 * These arguments could be named (then use static method <TT>argMap</TT>)
 * or not (then use static method <TT>argList</TT>).
 * <p/>
 * The order of arguments might be important because the values (of the map, or of the list)
 * will be passed "as such" to the constructor or method invocations if there is no use
 * of <TT>ConstructorProperties</TT> to annotate the corresponding constructor
 * </P>
 * The objects that are part of this Map/list can be "hollow" (see <TT>HollowParm</TT>).
 * so to get a full Map/list where all objects have been set test <TT>isSafe</TT>
 * then <TT>asFullMap</TT> (or <TT>asFullArray</TT>, or <TT>asFullList</TT>) to get data
 * that can be used as a set of parameters  for constructors or methods.
 * @author bamade
 */
public class MapArgs implements Map<String, Object>, Iterable<Object>, Serializable , Cloneable{
    /**
     * <TT>Map.Entry</TT> implementation.
     */
    public static class LEntry implements Map.Entry<String, Object>, Serializable , Cloneable{
         String key;
         Object value;

        public LEntry(String key, Object value) {
            if (key == null) {
                throw new IllegalArgumentException("no null key");
            }
            this.key = key;
            this.value = value;
        }

        // for bean compatiiblity : do not use
        public LEntry() {

        }
        // for bean compatiiblity : do not use
        public void setKey(String key) {
            this.key = key ;

        }

        public String getKey() {
            return this.key;
        }

        public Object getValue() {
            return this.value;
        }

        public Object setValue(Object o) {
            throw new UnsupportedOperationException("immutable entry");
        }

        public boolean equals(Object obj) {
            if (!(obj instanceof LEntry)) return false;
            LEntry other = (LEntry) obj;
            if (this.value == null) {
                return this.key.equals(other.key) && other.value == null;
            }
            return (this.key.equals(other.key) && this.value.equals(other.value)); //brutal
        }

        public int hashCode() {
            return this.key.hashCode();
        }

        public LEntry clone() {
            LEntry res = null ;
            if(value instanceof HollowParm)  {
                HollowParm data = (HollowParm) this.value ;
                res =  new LEntry(this.key, data.clone()) ;
            } else {
                try {
                    res = (LEntry)super.clone() ;
                } catch (CloneNotSupportedException e) {
                    // should not happen
                }
            }
            return res ;
        }

        public String toString() {
            String vlstr;
            if (this.value instanceof String) {
                vlstr = "'" + this.value + "'";
            } else {
                //vlstr = String.valueOf(value);
                vlstr = TypeUtils.stringify(value) ;
            }
            return this.key + ": " + vlstr;
        }

    }

    /**
     * the structure maintains two arrays: one of Entries and one of values.
     * this is a duplicate (since values from <TT>entries</TT> are duplicated in
     * <TT>vals</TT>. The reverse is not true: there could be values only but
     * no Map entry.
     * <P>
     *    to test that use <TT>prefersList()</TT>
     */
    private final LEntry[] entries;
    private final Object[] vals;

    /**
     * creates a complete Map/list with the corresponding entries
     * @param entries
     */
    private MapArgs(LEntry... entries) {
        this.entries = entries;
        int size = entries.length;
        this.vals = new Object[size];
        for (int ix = 0; ix < size; ix++) {
            LEntry lentry = entries[ix] ;
            if(lentry == null) {
                throw new IllegalArgumentException(
                        "ambiguous null value in arg number " + (ix+1) + ": use declaration such as 'null as String'");
            }
            vals[ix] = lentry.value;
        }
    }

    /**
     * creates a "list-only" structure (<TT>entries</TT> is of size 0.
     * @param values
     */
    private MapArgs(Object... values) {
        this.vals = values;
        this.entries = new LEntry[0];
    }

    /**
     * Convenience factory for creating an Entry.
     * @param key
     * @param value
     * @return
     */
    private static LEntry _m(String key, Object value) {
        return new LEntry(key, value);
    }

    // FACTORIES

    /**
     * the <TT>argMap</TT> methods are convenience factories for creating pair of String key-Object value
     * @return
     */
    public static MapArgs argMap() {
        //return new MapArgs();
        return new MapArgs(new LEntry[0]);
    }

    //TODO suppress all these methods because it's confusing
    /*
    protected static MapArgs argMap(String k1, Object v1) {
        return new MapArgs(_m(k1, v1));
    }

    protected static MapArgs argMap(String k1, Object v1, String k2, Object v2) {
        return new MapArgs(_m(k1, v1), _m(k2, v2));
    }

    protected static MapArgs argMap(String k1, Object v1,
                                 String k2, Object v2,
                                 String k3, Object v3) {
        return new MapArgs(_m(k1, v1), _m(k2, v2), _m(k3, v3));
    }

    protected static MapArgs argMap(String k1, Object v1,
                                 String k2, Object v2,
                                 String k3, Object v3,
                                 String k4, Object v4) {
        return new MapArgs(_m(k1, v1), _m(k2, v2), _m(k3, v3), _m(k4, v4));
    }

    protected static MapArgs argMap(String k1, Object v1,
                                 String k2, Object v2,
                                 String k3, Object v3,
                                 String k4, Object v4,
                                 String k5, Object v5) {
        return new MapArgs(_m(k1, v1), _m(k2, v2), _m(k3, v3), _m(k4, v4), _m(k5, v5));
    }

    protected static MapArgs argMap(String k1, Object v1,
                                 String k2, Object v2,
                                 String k3, Object v3,
                                 String k4, Object v4,
                                 String k5, Object v5,
                                 String k6, Object v6) {
        return new MapArgs(_m(k1, v1), _m(k2, v2), _m(k3, v3), _m(k4, v4), _m(k5, v5), _m(k6, v6));
    }

    protected static MapArgs argMap(String k1, Object v1,
                                 String k2, Object v2,
                                 String k3, Object v3,
                                 String k4, Object v4,
                                 String k5, Object v5,
                                 String k6, Object v6,
                                 String k7, Object v7) {
        return new MapArgs(_m(k1, v1), _m(k2, v2), _m(k3, v3), _m(k4, v4), _m(k5, v5), _m(k6, v6), _m(k7, v7));
    }
    */

    /**
     * used in combination with Entry factory methods such as <TT>a</TT>, <TT>anInt</TT>,
     * <TT>aDbl</TT>, <TT>aString</TT>, <TT>aRef</TT>
     * @param args
     * @return
     */
    public static MapArgs argMap(LEntry... args) {
        return new MapArgs(args);
    }

    /**
     * to use only when the corresponding object has a constructor with all needed arguments.
     * each object parameter which is a modifiable "named" parameter will be used as such,
     * other objects will just get a default name (which is the argument number) and will be
     * an immutable parameter.
     * <BR/>
     * Remember: this only works with classes that have the proper constructor (since a method
     * like <TT>set3</TT> may be called otherwise!).
     * @param args
     * @return
     */
    //TODO overloading and varargs no recommended !
    public static MapArgs argMap(Object... args) {
        LEntry[] entries = new LEntry[args.length] ;
        for(int ix =0 ; ix < args.length; ix++){
            Object obj = args[ix] ;
            if(obj instanceof LEntry) {
                entries[ix] = (LEntry) obj ;
            } else {
                entries[ix] = a(String.valueOf(ix), obj) ;
            }
        }
        return new MapArgs(entries) ;
    }

    /**
     * factory method for specifying a list of parameter: the ensuing <TT>MapArg</TT>
     * has only list features.
     * @param args
     * @return
     */
    public static MapArgs argList(Object... args) {
        Object[] resArgs = new Object[args.length];
        for (int ix = 0; ix < args.length; ix++) {
            Object arg = args[ix];
            if (arg instanceof LEntry) {
                LEntry entry = (LEntry) arg;
                arg = entry.getValue();
                if (arg instanceof HollowParm) {
                    HollowParm<?> parm = (HollowParm) arg;
                    if (!parm.isSet()) {
                        throw new EmptyParmException(entry.getKey(), parm);
                    }
                    arg = parm.getValue();
                }
            }
            resArgs[ix] = arg;
        }
        return new MapArgs(resArgs);
    }

    /**
     * creates a dummy list with one argument (kept for future "proxy" enhancements)
     * @param obj
     * @return
     */
    public static MapArgs proxy(Object obj) {
        return argList(obj) ;
    }

    /**
     * hybrid list that may contains HollowParms
     * @param args
     * @return
     */
    public static MapArgs rawList(Object... args) {
        Object[] resArgs = new Object[args.length];
        for (int ix = 0; ix < args.length; ix++) {
            Object arg = args[ix];
            if (arg instanceof LEntry) {
                LEntry entry = (LEntry) arg;
                arg = entry.getValue();
            }
            resArgs[ix] = arg;
        }
        return new MapArgs(resArgs);
    }

    //////////// LENTRY FACTORIES

    /**
     * declares a parameter that is not open for modification.
     * @param key
     * @param value
     * @return
     */
    public static LEntry a(String key, Object value) {
        // TODO trace System.out.println(" a -> " + key + "= " + String.valueOf(value));
        return _m(key, value);
    }

    /**
     * declares a modifiable parameter of type <TT>String</TT>
     * and sets the initials value.
     * @param key
     * @param value
     * @return
     */
    public static LEntry aString(String key, String value) {
        return _m(key, new StringParm(value));
    }
    /**
     * declares a modifiable parameter of type <TT>String</TT>
     * sets the initials value and annotates some properties
     * @param key
     * @param value
     * @param map keys and values will be transformed in Strings
     * @return
     */
    public static LEntry aString(String key, String value, Map<?,?> map) {
        return _m(key, new StringParm(value, map));
    }

    /**
     * declares a modifiable parameter of type <TT>String</TT>
     * but does not initialize it.
     * @param key
     * @return
     */
    public static LEntry aString(String key) {
        return _m(key, new StringParm());
    }

    /**
     * declares a modifiable parameter of type <TT>Object</TT>
     * whose reference will be later found using a name.
     * sets the initials value.
     * @param key
     * @param value
     * @return
     */
    public static LEntry aRef(String key, String value) {
        return _m(key, new NamedRefParm(value));
    }
    public static LEntry aRef(String key, String value, Map<?,?> map) {
        return _m(key, new NamedRefParm(value, map));
    }

    public static LEntry ref(String key) {
        return _m(key, new NamedRefParm(true, key));
    }

    /**
     * declares a modifiable parameter of type <TT>Object</TT>
     * whose reference will be later found using a name.
     * does not initialize it.
     * @param key
     * @return
     */
    public static LEntry aRef(String key) {
        return _m(key, new NamedRefParm());
    }

    /**
     * declares a modifiable parameter of type <TT>Integer</TT>
     * and sets the initial value.
     * @param key
     * @param value
     * @return
     */
    public static LEntry anInt(String key, Integer value) {
        return _m(key, new IntParm(value));
    }
    public static LEntry anInt(String key, Integer value, Map<?,?> map) {
        return _m(key, new IntParm(value, map));
    }

    /**
     * declares a modifiable parameter of type <TT>Integer</TT>
     * and sets the initial value with a (numeric) <TT>String</TT>
     * @param key
     * @param strVal
     * @return
     */
    public static LEntry anInt(String key, String strVal) {
        return _m(key, new IntParm(strVal));
    }
    public static LEntry anInt(String key, String strVal, Map<?,?> map) {
        return _m(key, new IntParm(strVal, map));
    }

    /**
     * declares a modifiable parameter of type <TT>Integer</TT>
     * and does not initialize it.
     * @param key
     * @return
     */
    public static LEntry anInt(String key) {
        return _m(key, new IntParm());
    }

    public static LEntry aBool(String key, Boolean value) {
        return _m(key, new BoolParm(value));
    }
    public static LEntry aBool(String key, Boolean value, Map<?,?> map) {
        return _m(key, new BoolParm(value, map));
    }
    public static LEntry aBool(String key, String strVal) {
        return _m(key, new BoolParm(strVal));
    }
    public static LEntry aBool(String key, String strVal, Map<?,?> map) {
        return _m(key, new BoolParm(strVal, map));
    }

    /**
     * declares a modifiable parameter of type boolean
     * and initializes it to <TT>false</TT>
     * @param key
     * @return
     */
    public static LEntry aBool(String key) {
        return _m(key, new BoolParm());
    }

    /**
     * declares a modifiable parameter of type <TT>Double</TT>
     * and sets the initial value.
     * @param key
     * @param value
     * @return
     */
    public static LEntry aDbl(String key, Double value) {
        return _m(key, new DblParm(value));
    }
    public static LEntry aDbl(String key, Double value, Map<?,?> map) {
        return _m(key, new DblParm(value, map));
    }

    /**
     * declares a modifiable parameter of type <TT>BigDecimal</TT>
     * and sets the initial value.
     * @param key
     * @param value
     * @return
     */
    public static LEntry aDbl(String key, BigDecimal value) {
        return _m(key, new DblParm(value));
    }
    public static LEntry aDbl(String key, BigDecimal value, Map<?,?> map) {
        return _m(key, new DblParm(value, map));
    }

    /**
     * declares a modifiable parameter of type <TT>BigDecimal</TT>
     * and sets the initial value with a (numeric) <TT>String</TT>
     * @param key
     * @param strVal
     * @return
     */
    public static LEntry aDbl(String key, String strVal) {
        return _m(key, new DblParm(strVal));
    }
    public static LEntry aDbl(String key, String strVal, Map<?,?> map) {
        return _m(key, new DblParm(strVal, map));
    }

    /**
     * declares a modifiable parameter of type <TT>Double</TT> or <TT>BigDecimal</TT>
     * and does not initialize it.
     * @param key
     * @return
     */
    /* was
    public static LEntry aDbl(String key) {
        return _m(key, new IntParm());
    } */
    public static LEntry aDbl(String key) {
        return _m(key, new DblParm());
    }

    /**
     * declares a modifiable parameter which is a list of simple values
     * (do not use for references)
     * @param key
     * @param list
     * @return
     */
    public static LEntry aList(String key, List list){
        return _m(key, new ListParm(list)) ;
    }

    public static LEntry aList(String key, double[] arrayDouble){
        return aList(key, Arrays.asList(arrayDouble)) ;
    }
    public static LEntry aList(String key, float[] arrayFloat){
        return aList(key, Arrays.asList(arrayFloat)) ;
    }

    public static LEntry aList(String key, List list, Map<?,?> map){
        return _m(key, new ListParm(list, map)) ;
    }

    public static LEntry aList(String key, String strList){
        return _m(key, new ListParm(strList)) ;
    }
    public static LEntry aList(String key, String strList, Map<?,?> map){
        return _m(key, new ListParm(strList,map)) ;
    }

    /**
     * declares a modifiable parameter which is a Map where values
     * are of a simple type (do not use for references)
     * @param key
     * @param map
     * @return
     */
    public static LEntry aMap(String key, Map map){
        return _m(key, new MapParm(map)) ;
    }
    public static LEntry aMap(String key, Map map, Map props){
        return _m(key, new MapParm(map, props)) ;
    }
    public static LEntry aMap(String key, String strMap){
        return _m(key, new MapParm(strMap)) ;
    }
    public static LEntry aMap(String key, String strMap, Map props){
        return _m(key, new MapParm(strMap, props)) ;
    }

    /**
     * declares a modifiable parameter of an array type.
     * @param key
     * @param array
     * @return
     */
    public static LEntry anArray(String key, int[] array) {
       return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, int[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }
    public static LEntry anArray(String key, byte[] array) {
        return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, byte[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }
    public static LEntry anArray(String key, char[] array) {
        return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, char[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }
    public static LEntry anArray(String key, float[] array) {
        return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, float[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }
    public static LEntry anArray(String key, double[] array) {
        return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, double[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }
    public static LEntry anArray(String key, boolean[] array) {
        return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, boolean[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }
    public static LEntry anArray(String key, Object[] array) {
        return _m(key, new ArrayParm(array)) ;
    }
    public static LEntry anArray(String key, Object[] array, Map map) {
        return _m(key, new ArrayParm(array, map)) ;
    }

    /**
     * parameter is a list used to create a struct object.
     * that is Serializable, method toString returns a List in groovy format
     * and has a constructor with corresponding members of the list
     * @param key
     * @param clazz
     * @param list
     * @return
     */
    public static  LEntry aStruct(String key, Class clazz, List list) {
        return _m(key, new SStructParm(clazz, list)) ;
    }
    public static  LEntry aStruct(String key, Class clazz, Map map) {
        return _m(key, new SStructParm(clazz, map)) ;
    }
    public static  LEntry aStruct(String key, Class clazz, List list, Map properties) {
        return _m(key, new SStructParm(clazz, list, properties)) ;
    }
    public static  LEntry aStruct(String key, Class clazz, Map map, Map properties) {
        return _m(key, new SStructParm(clazz, map, properties)) ;
    }

    /**
     * to be used for
     * any object that has :
     * <UL>
     *     <LI/> <TT>Serializable</TT> implementation
     *     <LI/> a symmetric way of dealing with strings:
     *     the <TT>toString()</TT> method returns a String that
     *     can be used to create an Object through static method
     *     <TT>valueOf(thatString)</TT>
     * </UL>
     * A typical example are <TT>Enums</TT>
     * @param key
     * @param val
     * @return
     */
    public static LEntry aVal(String key, Serializable val) {
        return _m(key, new ValParm( val)) ;
    }
    public static LEntry aVal(String key, Serializable val, Map properties) {
        return _m(key, new ValParm( val, properties)) ;
    }

    /**
     * same as  <TT>aVal (key, value)</TT> but here the name of the key is generated from
     * the Class Name of the object.
     * <BR/>
     * So if the value is of type <TT>RunningWay</TT> (an Enum)
     * then the key will be <TT>runningWay</TT>
     * @param val
     * @return
     */
    public static LEntry aVal(Serializable val) {
        String name = val.getClass().getSimpleName() ;
        char firstLetter = name.charAt(0) ;
        String key = Character.toLowerCase(firstLetter) + name.substring(1) ;
        return _m(key, new ValParm( val)) ;
    }
    public static LEntry aVal(Serializable val, Map properties) {
        String name = val.getClass().getSimpleName() ;
        char firstLetter = name.charAt(0) ;
        String key = Character.toLowerCase(firstLetter) + name.substring(1) ;
        return _m(key, new ValParm( val, properties)) ;
    }

    /**
     * to be used in constraints definition to declare an expression that will be evaluated with a single variable
     * @param argName the name of the variable in the expression
     * @param expr the code of the expression
     * @return a String to be evaluated by constraints
     */
    public static String predicate(String argName, String expr) {
        return "##" + argName.trim() +"##" +expr.trim() ;
    }

    /**
     * declares an entry with a Hollow value of type <TT>Class</TT>.
     * When set the corresponding class must implement the corresponding interface.
     * The key of the Entry is the simple name of the interface.
     * <P>
     *     this is not generally used for parameters but for defining the type of Nodes in trees of Objects.
     * </P>
     * @param anInterface
     * @param <T>
     * @return
     */
    public static <T> LEntry anImpl(Class<T> anInterface) {
        return _m(anInterface.getSimpleName(), new ImplPlaceHolder<T>(anInterface));
    }

    /**
     * declares a modifiable class : any modification to this value must implement the interface.
     * @param anInterface
     * @param aClass
     * @param <T>
     * @return
     */
    public static <T> LEntry anImpl(Class<T> anInterface, Class<? extends T> aClass) {
        return _m(anInterface.getSimpleName(), new ImplPlaceHolder<T>(anInterface, aClass));
    }


    public static Property kv(String key, String value) {
        return new Property(key, value) ;
    }

    //////////////////////////////////////////////////////

    public String toString() {
        if (entries.length != 0) {
            return Arrays.toString(entries);
        }
        return Arrays.toString(vals);
    }

    public MapArgs clone() {
        if(entries.length == 0){
            Object[] newVals = new Object[vals.length] ;
            for(int ix = 0 ; ix < vals.length; ix++) {
                Object curObj = vals[ix] ;
                if(curObj instanceof HollowParm) {
                    newVals[ix] = ((HollowParm) curObj).clone() ;
                } else {
                    newVals[ix] = curObj ;
                }
            }
            return new MapArgs(newVals) ;

        } else {
            LEntry[] newEnt = new LEntry[entries.length] ;
            for (int ix = 0 ; ix < entries.length; ix++) {
                newEnt[ix] = entries[ix] .clone() ;
            }
            return new MapArgs(newEnt) ;
        }
    }

    /**
     * this Map/list has only some list features and not those of a Map.
     * @return
     */
    public boolean prefersList() {
        return entries.length == 0 ;
    }

    // INTERFACE MAP
    public int size() {
        return vals.length;
    }

    public boolean isEmpty() {
        return vals.length == 0;
    }

    /**
     * as all "get" methods from the <TT>Map</TT> interface contract
     * this method is implemented by testing all the values in the structure.
     * Though generally inefficient compared to a true Map it is considered that, due
     * to the low number of values, it is still an acceptable compromise.
     * @param o
     * @return
     */
    public boolean containsKey(Object o) {
        for (int ix = 0; ix < entries.length; ix++) {
            if (entries[ix].key.equals(o)) {
                return true;
            }
        }
        return false;
    }

    public boolean containsValue(Object o) {
        for (int ix = 0; ix < vals.length; ix++) {
            if (vals[ix].equals(o)) {
                return true;
            }
        }
        return false;
    }

    public Object get(Object o) {
        for (int ix = 0; ix < entries.length; ix++) {
            LEntry entry = entries[ix];
            if (entry.key.equals(o)) {
                return entry.value;
            }
        }
        return null;
    }

    /**
     * the structure being immutable and having list features this
     * method is both efficient and guaranteed to return always the same value.
     * @param index
     * @return
     */
    public Object get(int index) {
        return vals[index];
    }

    /**
     * the structure being immutable this is unsupported.
     * unless it is to change a modifiable hollow Parameter
     * @param s
     * @param o
     * @return
     */
    public Object put(String s, Object o) {
        Object obj = this.get(s) ;
        if(obj instanceof HollowParm) {
            HollowParm hollowParm = (HollowParm) obj ;
            if( hollowParm.isReadOnly()) {
                throw new IllegalArgumentException(" unmodifiable parameter " + s) ;
            }
            Object res = hollowParm.getValue() ;
            hollowParm.modifyChecked(obj);
            return res ;
        }
        throw new UnsupportedOperationException("immutable map");
    }

    /**
     * the structure being immutable this is unsupported.
     * @param o
     * @return
     */
    public Object remove(Object o) {
        throw new UnsupportedOperationException("immutable map");
    }

    /**
     * the structure being immutable this is unsupported.
     * @param map
     */
    public void putAll(Map<? extends String, ? extends Object> map) {
        throw new UnsupportedOperationException("immutable map");
    }

    /**
     * the structure being immutable this is unsupported.
     *
     */
    public void clear() {
        throw new UnsupportedOperationException("immutable map");
    }

    public Set<String> keySet() {
        String[] keys = new String[entries.length];
        for (int ix = 0; ix < entries.length; ix++) {
            keys[ix] = entries[ix].key;
        }
        return new HashSet<String>(Arrays.asList(keys));
    }

    public Collection<Object> values() {
        return Arrays.asList(vals);
    }

    /**
     * returns an array of values contained in this structure (without the corresponding keys).
     * <B>Beware</B> the corresponding values may not be used "as is" since  some may be
     * <TT>HollowParm</TT> : to get a safe array of object use <TT>isSafe</TT>
     * and  then <TT>asFullArray</TT>, otherwise to be sure that the array can be used directly test
     * <TT>isValuesOnly()</TT>
     * @return
     */
    public Object[] toArray() {
        return vals;
    }

    /**
     * returns a List of values contained in this structure (without the corresponding keys).
     * <B>Beware</B> the corresponding values may not be used "as is" since  some may be
     * <TT>HollowParm</TT> : to get a safe List of object use <TT>isSafe</TT>
     * and  then <TT>asFullList</TT>, otherwise to be sure that the list can be used directly test
     * <TT>isValuesOnly()</TT>
     * @return
     */
    public List<Object> asList() {
        return Arrays.asList(vals);
    }

    public Set<Entry<String, Object>> entrySet() {
        return new HashSet<Entry<String, Object>>(Arrays.asList(entries));
    }

    public LEntry[] getEntries() {
        return entries;
    }

    public Iterator<Object> iterator() {
        return Arrays.asList(vals).iterator();
    }

    /**
     * tells if all "hollow" parameters have been initialized.
     * @return
     */
    public boolean isSafe() {
        for (int ix = 0; ix < vals.length; ix++) {
            Object arg = vals[ix];
            if (arg instanceof HollowParm) {
                HollowParm<?> parm = (HollowParm) arg;
                if (!parm.isSet()) {
                    //System.out.println(parm);
                    return false;
                }
            } else if(arg instanceof BuiltParm) {
                if(!((BuiltParm)arg).isSafe()) {
                    return false ;
                }
            }
        }
        return true;
    }

    /**
     * tells that the structure does not contain any <TT>HollowParm</TT>
     * object (so the contained values can be used "as is")
     * @return
     */
    public boolean isValuesOnly() {
        for (int ix = 0; ix < vals.length; ix++) {
            Object arg = vals[ix];
            if (arg instanceof HollowParm) {
                    return false;
            } else if(arg instanceof BuiltParm){
                if(!((BuiltParm)arg).isValuesOnly()) {
                    return false ;
                }
            }
        }
        return true;
    }


    /**
     * better calls <TT>isSafe</TT> before calling this method otherwise
     * will throw an EmptyParmException.
     * @param objectDictionary  a map that can resolve an object reference using its name
     * if <TT>null</TT> it will be considered that there are no "Hollow" Object in the structure
     * (see <TT>isValuesOnly</TT>) so use at your own risk!
     * @return
     */
    public MapArgs asFullMap(IndirectMap<String,Object> objectDictionary) {
        if(entries.length == 0 ) {
            return new MapArgs(asFullArray(objectDictionary));
        }
        LEntry[] newEntries = new LEntry[entries.length] ;
        for(int ix = 0 ; ix < entries.length; ix++) {
            LEntry curEntry = entries[ix] ;
            Object curObj = curEntry.getValue() ;
            if(objectDictionary!= null && curObj instanceof HollowParm) {
                HollowParm parm = (HollowParm) curObj ;
                if(! parm.isSet()) {
                    throw new EmptyParmException(curEntry.getKey(), parm) ;
                }
                if (parm instanceof NamedRefParm) {
                   curObj = objectDictionary.getIndirect(parm.getValue()) ;
                } else {
                    curObj = parm.getValue();
                }
            } else if( curObj instanceof BuiltParm) {
                curObj = ((BuiltParm) curObj).fullValue(objectDictionary) ;
            }
            newEntries[ix] = new LEntry(curEntry.getKey(), curObj) ;
        }
        return new MapArgs(newEntries) ;
    }

    /**
     * same remarks as <TT>asFullMap</TT>
     * @param objectDictionary
     * @return
     */
    public Object[] asFullArray(IndirectMap<String,Object> objectDictionary) {
        Object[] newValues = new Object[vals.length];
        for(int ix = 0 ; ix < vals.length ; ix++) {
            Object curObj = vals[ix] ;
            if(objectDictionary!= null && curObj instanceof HollowParm) {
                HollowParm parm = (HollowParm) curObj ;
                if(! parm.isSet()) {
                    throw new EmptyParmException( parm) ;
                }
                if (parm instanceof NamedRefParm) {
                    curObj = objectDictionary.getIndirect(parm.getValue()) ;
                } else {
                    curObj = parm.getValue();
                }
            }else if( curObj instanceof BuiltParm) {
                curObj = ((BuiltParm) curObj).fullValue(objectDictionary) ;
            }
            newValues[ix] = curObj ;
        }
        return newValues ;
    }

    /**
     * same remarks as <TT>asFullMap</TT>
     * @param objectDictionary
     * @return
     */
    public List<Object> asFullList(IndirectMap<String, Object> objectDictionary) {
        return Arrays.asList(asFullArray(objectDictionary)) ;
    }

    /**
     * modifies a local Entry with another entry with the same key
     * since modification could only operate on hollow parameters
     * this is used if the entry reference a <TT>HollowParm</TT> object.
     * @param acceptNullValues do we accept setting values to <TT>null</TT>
     * @param entry
     * @return true if an assignment has been operated
     * @throws org.lsst.gruth.types.IncompatibleTypeException if the assignment can not be made
     * due to type incompatible org.lsst.gruth.types.
     */
    public boolean overlap(boolean acceptNullValues, LEntry entry) {
        Object value = entry.value ;
        if(value instanceof HollowParm) {
            value = ((HollowParm)value).getValue() ;
        }
        if(( !acceptNullValues) && value == null) return false ;
        Object stored = get(entry.getKey()) ;
        if(stored == null) return false ;
        if(stored instanceof HollowParm) {
            HollowParm parm = (HollowParm) stored ;
            // values are modified in both vals and entries
            parm.modifyChecked(value);
            return true ;
        }
        return false ;
    }

    /**
     * operates a set of assignments  of valued <TT>HollowParm</TT> objects
     * in the argument to the corresponding <TT>HollowParm</TT> object in the current structure.
     * <P>
     *     see overlap with <TT>Object[]</TT> for more details.
     * <P>
     *     <B>beware</B> ; implementation not synchronized!
     * </P>
     * @param acceptNullvalues if null values in the other arguments should be assigned to objects in the result
     * @param other another MapArgs object
     * @return a new modified object
     * @throws org.lsst.gruth.types.IncompatibleTypeException if incompatible assignment
     * @throws ParmStructureException if incompatible structure
     */
    //TODO:  synchronize ?
    public MapArgs overlap(boolean acceptNullvalues, MapArgs other) {
        if(other.entries.length == 0 ) {
            return overlap(acceptNullvalues,other.vals) ;
        }
        for(int ix=0 ; ix < other.entries.length; ix++ ) {
            if(ix >= entries.length) throw new ParmStructureException(ix) ;
            LEntry curEntry = entries[ix] ;
            LEntry otherEntry = other.entries[ix] ;
            if(! curEntry.key.equals(otherEntry.key))throw new ParmStructureException(ix) ;
        }
        MapArgs res =  overlap(acceptNullvalues, other.vals ) ;
        return res ;
    }

    /**
     * modifies the <TT>HollowObjects</TT> values with the values contained in the argument.
     * the structure of the current <TT>Mapargs</TT> and the argument should be analog
     * but not necessarily the same:
     * <P>
     *     for each element in the array argument
     * <UL>
     *     <LI>
     *         if it is  a <TT>HollowParm</TT>  and the local corresponding data is
     *         an <TT>HollowParm</TT> then an assignment of values is tried
     *     <LI>
     *         if it is  a <TT>HollowParm</TT>  and the local corresponding data is
     *         not a <TT>ParmStructureException</TT> is fired
     *     <LI>
     *         if it is not  a <TT>HollowParm</TT>  and the local corresponding data is
     *         an <TT>HollowParm</TT> then an assignment of values is tried
     *         But if org.lsst.gruth.types are incompatible there will be a <TT>ParmStructureException</TT> fired
     *     <LI>
     *         if it is not  a <TT>HollowParm</TT>  and the local corresponding data is
     *         not an <TT>HollowParm</TT> nothing happens at this stage
     *
     * </UL>
     * @param acceptNullvalues
     * @param array
     * @return another MapArgs with modified values
     */
    public MapArgs overlap(boolean acceptNullvalues , Object[] array) {
        MapArgs res = this.clone() ;
        for(int ix=0 ; ix < array.length; ix++ ) {
            Object obj = array[ix] ;
            if (obj instanceof HollowParm) {
                if(ix >= res.vals.length) throw new ParmStructureException(ix) ;
                Object val = res.vals[ix];
                if(! (val instanceof HollowParm)) {
                    throw new ParmStructureException(ix) ;
                }
                if(! val.getClass().equals(obj.getClass())) {
                    throw new ParmStructureException(ix) ;
                }
                Object newValue = ((HollowParm)obj).getValue();
                if(newValue == null && ! acceptNullvalues) {
                    // nothing
                } else {
                    ((HollowParm) val).modifyChecked(((HollowParm)obj).getValue());
                }
            }  else if (obj instanceof BuiltParm) {
                if(ix >= res.vals.length) throw new ParmStructureException(ix) ;
                Object val = res.vals[ix];
                if(! (val instanceof BuiltParm)) {
                    throw new ParmStructureException(ix) ;
                }
                BuiltParm builtL = (BuiltParm) val ;
                BuiltParm builtR = (BuiltParm) obj ;
                if(builtL.getType() != builtR.getType() ) {
                    throw new ParmStructureException(ix) ;
                }
                builtL.overlap(acceptNullvalues, builtR.getArgs());

            } else { // special case where there is a hollow object in vals
                if(ix < res.vals.length && res.vals[ix] instanceof HollowParm) {
                    HollowParm valParm = (HollowParm) res.vals[ix] ;
                    if(obj == null && ! acceptNullvalues) {
                        // nothing
                    } else  {
                        valParm.modifyChecked(obj);
                    }
                }
            }
        }
        // now modify entries
        for(int ix = 0 ; ix < res.entries.length ; ix++) {
            LEntry entry = res.entries[ix] ;
            res.entries[ix] = new LEntry(entry.getKey(), res.vals[ix] ) ;
        }
        return res ;
    }

    public MapArgs overlap(boolean acceptNullValues , List<Object> list) {
        return overlap(acceptNullValues, list.toArray()) ;
    }

}
