package org.lsst.gruth.jutils;

import org.lsst.gruth.types.IncompatibleTypeException;
import org.lsst.gruth.types.TypeUtils;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Map;
import java.util.Properties;

/**
 * (could not re-use java.lang.Reference)
 * A Placeholder for a parameter value (for constructor or method).
 * Such an object has a value that can be modified : it can be empty (then method <TT>isSet</TT>
 * yields false)  or initialized.
 * <P>
 *     A special feature is about null values: normally setting up the value with null
 *     makes it uninitialized so making it nul AND initializes should be performed
 *     either by  <TT>setNullValue</TT> or through  <TT>modifyChecked</TT>
 *     which is a generic way of setting a value (with type control).
 * </P>
 *
 * @author bamade
 */
public  abstract class HollowParm<T extends Serializable>  implements Serializable, Cloneable{
    protected T value ;
    protected boolean set ;
    protected boolean readOnly = false ;
    protected Properties properties ;

    protected HollowParm() {}
    protected HollowParm(T value) {
        this.value = value ;
        if(value != null) {
            set = true ;
        }
    }

    protected HollowParm(T value, Map<?,?> map){
        this(value) ;
        setProperties(map);
    }

    final void setProperties(Map<?,?> map) {
        properties = new Properties() ;
        for(Map.Entry<?,?> entry : map.entrySet()){
            Object valProp = entry.getValue() ;
            String valPropStr = TypeUtils.stringify(valProp);
            if(valProp instanceof BigDecimal) {
                valPropStr = ((BigDecimal)valProp).toPlainString() ;
            } else {
                valPropStr = String.valueOf(valProp) ;
            }
            properties.setProperty(entry.getKey().toString(), valPropStr) ;
        }
    }


    public Properties getProperties() {
        return properties;
    }

    public T getValue() {
        return value;
    }

    public void setValue(T value) {
        if (readOnly) return ;
        this.value = value;
        if(value != null) {
            set = true ;
        } else {
            set = false ;
        }
    }

    public void setNullValue() {
        if (readOnly) return ;
        this.value = null ;
        set = true ;
    }

    public void clear() {
        if (readOnly) return ;
        this.setValue(null);
    }

    public void modifyChecked(Object obj) {
        if (readOnly) return ;
        if(obj == null) {
            setNullValue();
            return ;
        }
        Class clazz = obj.getClass() ;
        Class myClass = getValueClass() ;
        if(! myClass.isAssignableFrom(clazz)) {
            throw new IncompatibleTypeException(myClass, clazz) ;
        }
        this.value = (T)obj ;
    }


    public boolean isSet() {
        return set;
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public String toString() {
        //TODO: check why changing this provokes a bug!!
        //return this.getClass().getSimpleName() + " " + String.valueOf(value) ;
        //return  String.valueOf(value) ;
        return TypeUtils.stringify(value) ;
    }
    public abstract Class<T> getValueClass() ;

    public HollowParm<T> addProperties(Map.Entry<String, String>... entries){
        if(this.properties == null) {
            this.properties = new Properties() ;
        }
        for(Map.Entry<String, String> entry : entries){
            properties.setProperty(entry.getKey(), entry.getValue()) ;
        }
        return this ;
    }

    /**
     * first level clone: object that are referenced are not duplicated
     * @return
     */
    public  HollowParm clone() {
        HollowParm nextOne = null ;
        try {
            nextOne = (HollowParm) super.clone() ;
            if(properties!= null){
                nextOne.properties = (Properties) properties.clone() ;
            }
        } catch (CloneNotSupportedException e) {
            /* should not happen */
        }
        return nextOne ;
    }

}
