package org.lsst.ccs.elog;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.codec.binary.Base64;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 *
 * @author turri
 */
public class ElogEntry {

    private String author, category, formName = "default";
    private boolean formatted = false;
    private final List<String> tags = new ArrayList<>();
    private final Map<String, String> formFields = new HashMap<>();
    private final Map<String, byte[]> images = new HashMap<>();
    private final Map<String, byte[]> attachments = new HashMap<>();

    public ElogEntry() {
    }

    public void addTag(String tagName) {
        tags.add(tagName);
    }

    public void setAuthor(String author) {
        this.author = author;
    }

    public void setCategory(String category) {
        this.category = category;
    }

    public void setFormatted(boolean formatted) {
        this.formatted = formatted;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    public void setFormField(String fieldName, String fieldValue) {
        formFields.put(fieldName, fieldValue);
    }
    
    public void addImage(String name, byte[] imageBytes) {
        images.put(name, imageBytes);        
    }

    public void addAttachment(String name, byte[] attachmentBytes) {
        attachments.put(name, attachmentBytes);        
    }

    public String getAuthor() {
        return author;
    }

    public String getCategory() {
        return category;
    }

    public String getFormName() {
        return formName;
    }

    public boolean isFormatted() {
        return formatted;
    }

    public Set<String> getImages() {
        return images.keySet();
    }
    
    public Set<String> getAttachments() {
        return attachments.keySet();
    }
    
    public static boolean isFileImage(String fileName) {
        String fn = fileName.toLowerCase();
        return ( fn.endsWith("gif") || fn.endsWith("jpeg") || fn.endsWith("png") || fn.endsWith("tiff") || fn.endsWith("jpg"));        
    }

    public void addFile(String name, File file) throws FileNotFoundException, IOException {
        boolean isImage = isFileImage(name);
        byte[] byteArray = Files.readAllBytes(file.toPath());
        if ( isImage ) {
            addImage(name, byteArray);
        } else {
            addAttachment(name, byteArray);
        }
    }
    
    public void removeAttachment(String name, boolean isImage)  {
        if (isImage) {
            images.remove(name);
        } else {
            attachments.remove(name);
        }
    }
    
    public String toXML() {

        Element entryElement = new Element("entry");
        if (author != null) {
            entryElement.setAttribute("author", author);
        }
        if (category != null) {
            entryElement.setAttribute("category", category);
        }
        if (formatted) {
            entryElement.setAttribute("formatted", "formatted");
        }

        Element formElement = new Element("form");
        entryElement.addContent(formElement);
        formElement.setAttribute("name", formName);


        for (String field : formFields.keySet()) {
            Element fieldElement = new Element("field");
            fieldElement.setAttribute("name", field);
            fieldElement.setText(formFields.get(field));
            formElement.addContent(fieldElement);
        }

        for (String tag : tags) {
            Element tagElement = new Element("tag");
            tagElement.setAttribute("name", tag);
            entryElement.addContent(tagElement);
        }

        for (String image: images.keySet() ) {
            Element imageElement = new Element("attachment");
            imageElement.setAttribute("type", "image");
            imageElement.setAttribute("filename", image);
            imageElement.setText(new String(Base64.encodeBase64(images.get(image))));
            entryElement.addContent(imageElement);                        
        }
        
        for (String attachment: attachments.keySet() ) {
            Element attachmentElement = new Element("attachment");
            attachmentElement.setAttribute("type", "file");
            attachmentElement.setAttribute("filename", attachment);
            attachmentElement.setText(new String(Base64.encodeBase64(attachments.get(attachment))));
            entryElement.addContent(attachmentElement);                        
        }
        
        Document myDocument = new Document(entryElement);
        XMLOutputter outputter = new XMLOutputter(Format.getCompactFormat());
        String xmlString = outputter.outputString(myDocument).replace("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", "").trim();        
        return xmlString;


    }
    
    public void reset() {
        tags.clear();
        images.clear();
        attachments.clear();
        formFields.clear();
    }
}
