package org.lsst.ccs.elog;

import java.io.Closeable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * A connection to an instance of an eLog. Methods on this class allow
 * operations to be performed on the eLog.
 *
 * @author tonyj
 */
public class ELogConnection implements Closeable {

    private final URI baseURI;
    private final Random random = new SecureRandom();
    private final String eLogXmlUser = System.getProperty("org.lsst.ccs.elog.xml.user", "xml-user");
    private final String eLogXmlPwd = System.getProperty("org.lsst.ccs.elog.xml.password", "xml-lsst");
    private final int timeout = Integer.getInteger("org.lsst.ccs.elog.timeout", 5000);
    private final CloseableHttpClient client;

    /**
     * Create a new electronic logbook connection.
     *
     * @param baseURL The base URL for the logbook, e.g.
     * "http://dbweb0.fnal.gov:8080/ECL/lsst_camera"
     * @throws java.net.URISyntaxException If the baseURL is invalid
     */
    public ELogConnection(String baseURL) throws URISyntaxException {
        this(new URI(baseURL));
    }

    /**
     * Create a new electronic logbook connection.
     *
     * @param baseURI The URI for the logbook.
     */
    public ELogConnection(URI baseURI) {
        this.baseURI = baseURI;
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(timeout)
                .setConnectionRequestTimeout(timeout)
                .setSocketTimeout(timeout).build();
        client = HttpClientBuilder.create().setDefaultRequestConfig(config).build();
    }

    private String getSalt() {
        int i = random.nextInt(1234567890);
        i++;
        return convertToMD5Hex(String.valueOf(i));
    }

    private String convertToMD5Hex(String someText) {

        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(someText.getBytes());
            byte byteData[] = md.digest();

            //convert the byte to hex format method 1
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < byteData.length; i++) {
                sb.append(Integer.toString((byteData[i] & 0xff) + 0x100, 16).substring(1));
            }

            return sb.toString();

        } catch (NoSuchAlgorithmException ex) {
            throw new RuntimeException("Error during convertToMD5Hex", ex);
        }

    }

    private void addSignature(HttpRequestBase method, String args, String body) {
        String signature = args + ":" + eLogXmlPwd + ":" + body;
        method.addHeader("X-Signature-Method", "md5");
        method.addHeader("X-User", eLogXmlUser);
        method.addHeader("X-Signature", convertToMD5Hex(signature));

    }

    /**
     * Get the list of categories available for this log book
     *
     * @return A list of the available categories.
     * @throws IOException if an error occurs, or the method returns a status
     * other than 200 (OK)
     */
    public List<String> getCategories() throws IOException {

        String salt = "salt=" + getSalt();
        URI requestURI = URIUtils.resolve(baseURI, "A/xml_category_list?" + salt);
        HttpGet get = new HttpGet(requestURI);
        try {
            addSignature(get, salt, "");
            // execute the POST
            try (CloseableHttpResponse response = client.execute(get)) {

                if (response.getStatusLine().getStatusCode() == 200) {
                    SAXBuilder builder = new SAXBuilder();
                    try {
                        Document d = builder.build(response.getEntity().getContent());
                        Element root = d.getRootElement();
                        List<Element> cats = root.getChildren("category");
                        List<String> categories = new ArrayList<>();

                        for (Element cat : cats) {
                            categories.add(cat.getAttributeValue("path"));
                        }
                        return categories;
                    } catch (JDOMException e) {
                        throw new IOException("XML exception while getting categories", e);
                    }
                } else {
                    throw new IOException("Bad return code: " + response.getStatusLine().getStatusCode());
                }
            }

        } finally {
            // release any connection resources used by the method
            get.releaseConnection();
        }
    }

    /**
     * Get the list of tags available for this log book.
     *
     * @return A list of the available tags.
     * @throws IOException if an error occurs, or the method returns a status
     * other than 200 (OK)
     */
    public List<String> getTags() throws IOException {

        String salt = "salt=" + getSalt();
        URI requestURI = URIUtils.resolve(baseURI, "A/xml_tag_list?" + salt);
        HttpGet get = new HttpGet(requestURI);
        try {
            addSignature(get, salt, "");
            // execute the POST
            try (CloseableHttpResponse response = client.execute(get)) {
                if (response.getStatusLine().getStatusCode() == 200) {
                    SAXBuilder builder = new SAXBuilder();
                    Document d = builder.build(response.getEntity().getContent());
                    Element root = d.getRootElement();
                    List<Element> allTags = root.getChildren("tag");
                    List<String> tags = new ArrayList<>();
                    for (Element tag : allTags) {
                        tags.add(tag.getAttributeValue("name"));
                    }
                    return tags;

                } else {
                    throw new IOException("Bad return code: " + response.getStatusLine().getStatusCode());
                }
            } catch (JDOMException e) {
                throw new IOException("XML exception while getting tags", e);
            }
        } finally {
            // release any connection resources used by the method
            get.releaseConnection();
        }
    }

    /**
     * Post an entry to the log book.
     *
     * @param entry The entry to post
     * @return A result containing both the HTTP status code, and the text
     * returned from the post.
     * @throws IOException If an error occurs
     */
    public ElogTransactionResult postEntryToElog(ElogEntry entry) throws IOException {

        String salt = "salt=" + getSalt();
        URI requestURI = URIUtils.resolve(baseURI, "E/xml_post?" + salt);

        HttpPost post = new HttpPost(requestURI);
        try {
            String body = entry.toXML();
            addSignature(post, salt, body);

            HttpEntity entity = new StringEntity(body, "text/xml", "UTF-8");
            post.setEntity(entity);
            //Content type for posts only
            post.setHeader("Content-type", "text/xml");
            // execute the POST
            try (CloseableHttpResponse response = client.execute(post)) {
                return new ElogTransactionResult(response.getStatusLine().getStatusCode(), EntityUtils.toString(response.getEntity()));
            }
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        } finally {
            // release any connection resources used by the method
            post.releaseConnection();
        }
    }

    @Override
    public void close() throws IOException {
        client.close();
    }
}
