package org.lsst.ccs.drivers.yaskawa;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Yaskawa A1000 series motor drive..
 *
 *  @author  Owen Saxton
 */
public class A1000 {

    /**
     *  Public constants.
     */
    public enum Language { ENGLISH }
    public enum AccessLevel { OPR_ONLY, USR_PARM, FULL }
    public enum ControlMode { VF, VF_PG, OLV, CLV, CLV_PM, OLV_PM, AOLV_PM }
    public enum FreqRefSrc { OPERATOR, ANALOG, MODBUS, OPTN_PCB, PULSE }
    public enum RunCmndSrc { OPERATOR, DIGITAL, MODBUS, OPTN_PCB }
    public enum StopMethod { COAST, COAST_TMR, INJECT, RAMP }
    public enum RunProg { PROG_ONLY, RUN_PROG, RUN_ONLY }
    public enum BaudRate { B1200, B2400, B4800, B9600, B19200, B38400, B57600, B76800, B115200 }
    public enum Parity { EVEN, ODD, NONE }
    public enum ErrorAction { RAMP_STOP, COAST_STOP, FAST_STOP, ALARM_ONLY }
    public enum VoltageUnits { ONES, TENTHS }
    public enum VfSelection { HZ50, HZ60_SAT, HZ50_SAT, HZ72, HZ50_VT1, HZ50_VT2, HZ60_VT1, HZ60_VT2,
                              HZ50_HST1, HZ50_HST2, HZ60_HST1, HZ60_HST2, HZ90, HZ120, HZ180, CUSTOM }

    public static final int
        STATUS_RUNNING     = 0x01,
        STATUS_ZERO_SPEED  = 0x02,
        STATUS_REVERSE     = 0x04,
        STATUS_FAULT_RESET = 0x08,
        STATUS_SPEED_AGREE = 0x10,
        STATUS_READY       = 0x20,
        STATUS_ALARM       = 0x40,
        STATUS_FAULT       = 0x80;

    /**
     *  Private constants.
     */
    private static final short
        REG_COMMAND_MF_IN  = 0x001,
        CMND_STOP          = 0x00,
        CMND_FORWARD       = 0x01,
        CMND_FR_REVERSE    = 0x02,
        CMND_OD_REVERSE    = 0x03,
        CMND_RESET_FAULT   = 0x08,
        REG_FREQ_REFC      = 0x002,
        REG_VOLTAGE_GAIN   = 0x003,
        REG_TORQUE_REFC    = 0x004,
        REG_TORQUE_COMP    = 0x005,
        REG_PID_TARGET     = 0x006,
        REG_ANAL_OUT_FM    = 0x007,
        REG_ANAL_OUT_AM    = 0x008,
        REG_DIGITAL_OUT    = 0x009,
        REG_PULSE_OUT      = 0x00a,
        REG_CTRL_SLCTN     = 0x00f,

        REG_ANAL_MONITOR_1 = 0x01b,
        REG_ANAL_MONITOR_2 = 0x01c,
        REG_DIG_OUT_OPTN   = 0x01d,
        REG_DRIVE_STATUS_1 = 0x020,
        REG_FAULT_1        = 0x021,
        REG_LINK_STATUS    = 0x022,
        REG_FREQ_REF_IN    = 0x023,
        REG_OUTPUT_FRQ     = 0x024,
        REG_VOLTAGE_REF    = 0x025,
        REG_OUTPUT_CUR     = 0x026,
        REG_OUTPUT_PWR     = 0x027,
        REG_TORQUE_REF_IN  = 0x028,
        REG_FAULT_2        = 0x029,
        REG_ALARM_1        = 0x02a,
        REG_INPUT_TRM      = 0x02b,
        REG_DRIVE_STATUS_2 = 0x02c,
        REG_OUTPUT_TRM     = 0x02d,
        REG_FREQ_REF_BIAS  = 0x02f,
        REG_DC_BUS_VOLTAGE = 0x031,
        REG_TORQUE_REF_IN2 = 0x032,
        REG_PRODUCT_CODE_1 = 0x034,
        REG_PRODUCT_CODE_2 = 0x035,
        REG_COMM_ERROR     = 0x03d,
        REG_OUT_FREQ_RPM   = 0x03e,
        REG_OUT_FREQ_PCT   = 0x03f,

        REG_FREQ_REFC_IN   = 0x040,  //U1-01
        REG_OUTPUT_FREQ    = 0x041,  //U1-02
        REG_OUTPUT_CURR    = 0x042,  //U1-03
        REG_CTRL_METHOD_IN = 0x043,  //U1-04
        REG_MOTOR_SPEED    = 0x044,  //U1-05
        REG_VOLTAGE_REFC   = 0x045,  //U1-06
        REG_BUS_VOLTAGE    = 0x046,  //U1-07
        REG_OUTPUT_POWER   = 0x047,  //U1-08
        REG_INPUT_TERM     = 0x049,  //U1-10
        REG_OUTPUT_TERM    = 0x04a,  //U1-11
        REG_DRIVE_STATUS   = 0x04b,  //U1-12

        REG_OPERATION_TIME = 0x04c,  //U4-01 (hr)
        REG_RUN_CMND_COUNT = 0x075,  //U4-02
        REG_FAN_OPRN_TIME  = 0x067,  //U4-03 (hr)
        REG_FAN_LIFE_PCT   = 0x07e,  //U4-04 (%)
        REG_CAP_LIFE_PCT   = 0x07c,  //U4-05 (%)
        REG_SCBR_LIFE_PCT  = 0x7d6,  //U4-06 (%)
        REG_IGBT_LIFE_PCT  = 0x7d7,  //U4-07 (%)
        REG_HEATSINK_TEMP  = 0x068,  //U4-08 (C)
        REG_LED_CHECK      = 0x05e,  //U4-09
        REG_KWH_LOWER      = 0x05c,  //U4-10 (0.1 KWH)
        REG_KWH_UPPER      = 0x05d,  //U4-11 (MWH)
        REG_PEAK_HOLD_CURR = 0x7cf,  //U4-13 (0.01 A)
        REG_PEAK_HOLD_FREQ = 0x7d0,  //U4-14 (0.01 Hz)
        REG_OVERLOAD_EST   = 0x7d8,  //U4-16 (0.1 %)

        REG_OPER_TIME_SET   = 0x50b,  //o4-01 (10 hr)
        REG_OPER_TIME_TYPE  = 0x50c,  //o4-02
        REG_FAN_OP_TIME_SET = 0x50e,  //o4-03 (10 hr)
        REG_CAP_LT_PCT_SET  = 0x51d,  //o4-05 (%)
        REG_SCBR_LT_PCT_SET = 0x523,  //o4-07 (%)
        REG_IGBT_LT_PCT_SET = 0x525,  //o4-09 (%)

        REG_ALARM_CODE     = 0x07f,
        REG_CURR_FAULT     = 0x080,
        REG_PREV_FAULT     = 0x081,

        REG_ACC_OP_TIME_HI = 0x098,
        REG_ACC_OP_TIME_LO = 0x099,
        REG_FAN_OP_TIME_HI = 0x09a,
        REG_FAN_OP_TIME_LO = 0x09b,
        REG_RATED_CURRENT  = 0x0ab,
        REG_MOTOR_SPD_RPM  = 0x0ac,
        REG_MOTOR_SPD_PCT  = 0x0ad,
        REG_FREQ_REF_S_RPM = 0x0b5,
        REG_FREQ_REF_S_PCT = 0x0b6,
        REG_FREQ_REF_RPM   = 0x0b7,
        REG_FREQ_REF_PCT   = 0x0b8,
        REG_OP_ERROR_CODE  = 0x0bf,
        REG_FAULT_3        = 0x0c0,
        REG_FAULT_4        = 0x0c1,
        REG_FAULT_5        = 0x0c2,
        REG_FAULT_6        = 0x0c3,
        REG_FAULT_7        = 0x0c4,
        REG_FAULT_8        = 0x0c5,
        REG_FAULT_9        = 0x0c6,
        NUM_FAULT_REGS     = REG_FAULT_9 - REG_FAULT_3 + 1,
        REG_ALARM_2        = 0x0c8,
        REG_ALARM_3        = 0x0c9,
        REG_ALARM_4        = 0x0ca,
        REG_ALARM_5        = 0x0cb,
        REG_ALARM_6        = 0x0cc,
        NUM_ALARM_REGS     = REG_ALARM_6 - REG_ALARM_2 + 1,
        REG_CPF_1          = 0x0d0,
        REG_CPF_2          = 0x0d1,
        REG_CPF_3          = 0x0d2,

        REG_LANGUAGE       = 0x100,  //A1-00
        LANG_ENGLISH       = 0,  // Default
        LANG_JAPANESE      = 1,
        LANG_GERMAN        = 2,
        LANG_FRENCH        = 3,
        LANG_ITALIAN       = 4,
        LANG_SPANISH       = 5,
        LANG_PORTUGUESE    = 6,
        LANG_CHINESE       = 7,
        LANG_CZECH         = 8,
        LANG_RUSSIAN       = 9,
        LANG_TURKISH       = 10,
        LANG_POLISH        = 11,
        LANG_GREEK         = 12,
        REG_ACCESS_LEVEL   = 0x101,  //A1-01
        ACC_LEVEL_OPR_ONLY = 0,
        ACC_LEVEL_USR_PARM = 1,
        ACC_LEVEL_FULL     = 2,  // Default
        REG_CTRL_MODE      = 0x102,  //A1-02
        CTRL_MODE_VF       = 0,
        CTRL_MODE_VF_PG    = 1,
        CTRL_MODE_OLV      = 2,  // Default
        CTRL_MODE_CLV      = 3,
        CTRL_MODE_OLV_PM   = 5,
        CTRL_MODE_AOLV_PM  = 6,
        CTRL_MODE_CLV_PM   = 7,
        REG_INIT_PARAMS    = 0x103,  //A1-03
        INIT_NONE          = 0,  // Default
        INIT_USER          = 0x1110,
        INIT_2WIRE         = 0x2220,
        INIT_3WIRE         = 0x3330,
        INIT_OPE04_RESET   = 0x5550,
        REG_PW_ENTRY       = 0x104,  //A1-04
        REG_PW_SETTING     = 0x105,  //A1-05

        REG_FREQ_REF_SRC   = 0x180,  //b1-01
        FREQ_REF_OPERATOR  = 0,
        FREQ_REF_ANALOG    = 1,  // Default
        FREQ_REF_MODBUS    = 2,
        FREQ_REF_OPTN_PCB  = 3,
        FREQ_REF_PULSE     = 4,
        REG_RUN_CMND_SRC   = 0x181,  //b1-02
        RUN_CMND_OPERATOR  = 0,
        RUN_CMND_DIGITAL   = 1,  // Default
        RUN_CMND_MODBUS    = 2,
        RUN_CMND_OPTN_PCB  = 3,
        REG_STOP_METHOD    = 0x182,  //b1-03
        STOP_METH_RAMP     = 0,  // Default
        STOP_METH_COAST    = 1,
        STOP_METH_INJECT   = 2,
        STOP_METH_COAST_TM = 3,
        REG_REV_PERMIT     = 0x183,  //b1-04
        REV_ENABLED        = 0,  // Default
        REV_DISABLED       = 1,
        REG_DIG_INPUT_CONF = 0x185,  //b1-06
        DIG_CONF_DISABLED  = 0,
        DIG_CONF_ENABLED   = 1,  // Default
        REG_RUN_CMND_CYCLE = 0x186,  //b1-07
        RUN_CYCLE_ENABLED  = 0,  // Default
        RUN_CYCLE_DISABLED = 1,
        REG_RUN_CMND_PROG  = 0x187,  //b1-08
        PROG_MODE_ONLY     = 0,  // Default
        RUN_PROG_OKAY      = 1,
        RUN_MODE_ONLY      = 2,

        REG_FREQ_REF_MAX   = 0x289,  //d2-01 (0.1 %)
        REG_FREQ_REF_MIN   = 0x28a,  //d2-01 (0.1 %)

        REG_INPUT_VOLTAGE  = 0x300,  //E1-01 (0.1 V)
        REG_VF_SELECTION   = 0x302,  //E1-03
        VF_50HZ            = 0,
        VF_60HZ_SAT        = 1,
        VF_50HZ_SAT        = 2,
        VF_72HZ            = 3,
        VF_50HZ_VT1        = 4,
        VF_50HZ_VT2        = 5,
        VF_60HZ_VT1        = 6,
        VF_60HZ_VT2        = 7,
        VF_50HZ_HST1       = 8,
        VF_50HZ_HST2       = 9,
        VF_60HZ_HST1       = 10,
        VF_60HZ_HST2       = 11,
        VF_90HZ            = 12,
        VF_120HZ           = 13,
        VF_180HZ           = 14,
        VF_CUSTOM          = 15,  // Default

        REG_NODE_ADDRESS   = 0x425,  //H5-01
        REG_BAUD_RATE      = 0x426,  //H5-02
        BAUD_RATE_1200     = 0,
        BAUD_RATE_2400     = 1,
        BAUD_RATE_4800     = 2,
        BAUD_RATE_9600     = 3,  // Default
        BAUD_RATE_19200    = 4,
        BAUD_RATE_38400    = 5,
        BAUD_RATE_57600    = 6,
        BAUD_RATE_76800    = 7,
        BAUD_RATE_115200   = 8,
        REG_COMM_PARITY    = 0x427,  //H5-03
        PARITY_NONE        = 0,  // Default
        PARITY_EVEN        = 1,
        PARITY_ODD         = 2,
        REG_ERROR_ACTION   = 0x428,  //H5-04
        ERROR_RAMP_STOP    = 0,  // Default
        ERROR_COAST_STOP   = 1,
        ERROR_FAST_STOP    = 2,
        ERROR_ALARM_ONLY   = 3,
        REG_COMM_FAULT_DET = 0x429,  //H5-05
        FAULT_DETC_OFF     = 0,
        FAULT_DETC_ON      = 1,  // Default
        REG_XMIT_WAIT_TIME = 0x42a,  //H5-06 (msec)
        REG_RTS_CONTROL    = 0x42b,  //H5-07
        RTS_ALWAYS_ON      = 0,
        RTS_ONLY_ON_SEND   = 1,  // Default
        REG_ERROR_DET_TIME = 0x435,  //H5-09 (0.1 sec)
        REG_VOLTAGE_UNITS  = 0x436,  //H5-10
        VOLT_UNITS_TENTHS  = 0,  // Default
        VOLT_UNITS_ONES    = 1,
        REG_ENTER_FUNC     = 0x43c,  //H5-11
        ENTER_REQUIRED     = 0,  // Default
        ENTER_NOT_REQUIRED = 1,
        REG_RUN_CMND_TYPE  = 0x43d,  //H5-12
        RUN_CMND_FWD_REV   = 0,  // Default
        RUN_CMND_OPER_DIRN = 1,

        REG_DRIVE_MODEL    = 0x508,  //o2-04
        MODEL_4A0002       = 0x92,
        MODEL_4A0004       = 0x93,
        MODEL_4A0005       = 0x94,
        MODEL_4A0007       = 0x95,
        MODEL_4A0009       = 0x96,

        REG_ENTER_EEPROM   = 0x900,
        REG_ENTER_RAM      = 0x910,
        REG_ = 0x0,
        z = 0;

    public static final int
        ALARM_NONE                = 0x00,
        ALARM_UNDERVOLTAGE        = 0x01,
        ALARM_OVERVOLTAGE         = 0x02,
        ALARM_HEATSINK_OVERHEAT   = 0x03,
        ALARM_DRIVE_OVERHEAT      = 0x04,
        ALARM_OVERTORQUE_1        = 0x05,
        ALARM_OVERTORQUE_2        = 0x06,
        ALARM_RUN_INPUT_ERROR     = 0x07,
        ALARM_DRIVE_BASEBLOCK     = 0x08,
        ALARM_EXTERNAL_FAULT_3    = 0x09,
        ALARM_EXTERNAL_FAULT_4    = 0x0a,
        ALARM_EXTERNAL_FAULT_5    = 0x0b,
        ALARM_EXTERNAL_FAULT_6    = 0x0c,
        ALARM_EXTERNAL_FAULT_7    = 0x0d,
        ALARM_EXTERNAL_FAULT_8    = 0x0e,
        ALARM_COOLING_FAN         = 0x0f,
        ALARM_OVERSPEED           = 0x10,
        ALARM_EXCESS_SPEED_DEVN   = 0x11,
        ALARM_PG_DISCONNECTED     = 0x12,
        ALARM_MODBUS_COMM_ERR     = 0x14,
        ALARM_OPTION_COMM_ERR     = 0x15,
        ALARM_SERIAL_COMM_ERR     = 0x16,
        ALARM_MOTOR_OVERLOAD      = 0x17,
        ALARM_DRIVE_OVERLOAD      = 0x18,
        ALARM_OPTION_CARD_FAULT   = 0x1a,
        ALARM_SWITCH_DURING_RUN   = 0x1b,
        ALARM_SERIAL_TRANS_ERR    = 0x1d,
        ALARM_UNDERTORQUE_1       = 0x1e,
        ALARM_UNDERTORQUE_2       = 0x1f,
        ALARM_MODBUS_TEST_FAULT   = 0x20,
        ALARM_MOTOR_OVERHEAT      = 0x22,
        ALARM_PID_FEEDBACK_LOSS   = 0x27,
        ALARM_PID_EXC_FEEDBACK    = 0x28,
        ALARM_DRIVE_DISABLED      = 0x2a,
        ALARM_PG_DISCONNECTEDX    = 0x2b,
        ALARM_SI_T3_WATCHDOG      = 0x31,
        ALARM_SI_T3_ADDR_SETTING  = 0x32,
        ALARM_SI_T3_CYCLE_SETTING = 0x33,
        ALARM_HIGH_CURRENT        = 0x34,
        ALARM_FAN_MAINT_TIME      = 0x35,
        ALARM_CAP_MAINT_TIME      = 0x36,
        ALARM_SI_S_EEPROM_ERROR   = 0x38,
        ALARM_EXTERNAL_FAULT_1    = 0x39,
        ALARM_EXTERNAL_FAULT_2    = 0x3a,
        ALARM_SAFE_DISABLE_INPUT  = 0x3b,
        ALARM_SAFE_DISABLE_INPUTX = 0x3c,
        ALARM_MECH_WEAKENING_1    = 0x3d,
        ALARM_MECH_WEAKENING_2    = 0x3e,
        ALARM_PLC_1               = 0x3f,
        ALARM_PLC_2               = 0x40,
        ALARM_VOLTAGE_DETC_FAULT  = 0x41,
        ALARM_IGBT_MAINT_TIME_90  = 0x42,
        ALARM_SCBR_MAINT_TIME     = 0x43,
        ALARM_IGBT_MAINT_TIME_50  = 0x44,
        ALARM_MOTOR_OVERHEAT_NTC  = 0x48,
        ALARM_DRIVE_WORKS_EZ      = 0x49,
        ALARM_THERMISTOR_DISC     = 0x4d;

    private static final Map<Integer, String> alarmDesc = new HashMap<>();
    static {
        alarmDesc.put(ALARM_UNDERVOLTAGE, "Undervoltage (Uv)");
        alarmDesc.put(ALARM_OVERVOLTAGE, "Overvoltage (ov)");
        alarmDesc.put(ALARM_HEATSINK_OVERHEAT, "Heatsink overheat (oH)");
        alarmDesc.put(ALARM_DRIVE_OVERHEAT, "Drive overheat (oH2)");
        alarmDesc.put(ALARM_OVERTORQUE_1, "Overtorque 1 (oL3)");
        alarmDesc.put(ALARM_OVERTORQUE_2, "Overtorque 2 (oL4)");
        alarmDesc.put(ALARM_RUN_INPUT_ERROR, "Run commands input error (EF)");
        alarmDesc.put(ALARM_DRIVE_BASEBLOCK, "Drive baseblock (bb)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_3, "External fault 3, input terminal S3 (EF3)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_4, "External fault 4, input terminal S4 (EF4)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_5, "External fault 5, input terminal S5 (EF5)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_6, "External fault 6, input terminal S6 (EF6)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_7, "External fault 7, input terminal S7 (EF7)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_8, "External fault 8, input terminal S8 (EF8)");
        alarmDesc.put(ALARM_COOLING_FAN, "Cooling fan error (FAN)");
        alarmDesc.put(ALARM_OVERSPEED, "Overspeed (oS)");
        alarmDesc.put(ALARM_EXCESS_SPEED_DEVN, "Excessive speed deviation (dEv");
        alarmDesc.put(ALARM_PG_DISCONNECTED, "PG disconnected (PGo)");
        alarmDesc.put(ALARM_MODBUS_COMM_ERR, "Modbus communication error (cE)");
        alarmDesc.put(ALARM_OPTION_COMM_ERR, "Option communication error (bUS)");
        alarmDesc.put(ALARM_SERIAL_COMM_ERR, "Serial communication transmission error (CALL)");
        alarmDesc.put(ALARM_MOTOR_OVERLOAD, "Motor overload (oL1)");
        alarmDesc.put(ALARM_DRIVE_OVERLOAD, "Drive overload (oL2)");
        alarmDesc.put(ALARM_OPTION_CARD_FAULT, "Option card external fault (EF0)");
        alarmDesc.put(ALARM_SWITCH_DURING_RUN, "Motor command switch input during run (rUN)");
        alarmDesc.put(ALARM_SERIAL_TRANS_ERR, "Serial communication transmission error (CALL)");
        alarmDesc.put(ALARM_UNDERTORQUE_1, "Undertorque detection 1 (UL3)");
        alarmDesc.put(ALARM_UNDERTORQUE_2, "Undertorque detection 2 (UL4)");
        alarmDesc.put(ALARM_MODBUS_TEST_FAULT, "Modbus test mode fault (SE)");
        alarmDesc.put(ALARM_MOTOR_OVERHEAT, "Motor overheat (oH3)");
        alarmDesc.put(ALARM_PID_FEEDBACK_LOSS, "PID feedback loss (FbL)");
        alarmDesc.put(ALARM_PID_EXC_FEEDBACK, "Excessive PID feedback (FbH)");
        alarmDesc.put(ALARM_DRIVE_DISABLED, "Drive disabled (dnE)");
        alarmDesc.put(ALARM_PG_DISCONNECTEDX, "PG disconnected (PGo)");
        alarmDesc.put(ALARM_SI_T3_WATCHDOG, "S!-T3 watchdog error (E5)");
        alarmDesc.put(ALARM_SI_T3_ADDR_SETTING, "S!-T3 station address setting error (AEr)");
        alarmDesc.put(ALARM_SI_T3_CYCLE_SETTING, "SI-T3 comm. cycle setting error (CyC)");
        alarmDesc.put(ALARM_HIGH_CURRENT, "High current (HCA)");
        alarmDesc.put(ALARM_FAN_MAINT_TIME, "Cooling fan maintenance time (LT-1)");
        alarmDesc.put(ALARM_CAP_MAINT_TIME, "Capacitor maintenance time (LT-2)");
        alarmDesc.put(ALARM_SI_S_EEPROM_ERROR, "SI-S EEPROM error (EEP)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_1, "External fault 1, input terminal S1 (EF1)");
        alarmDesc.put(ALARM_EXTERNAL_FAULT_2, "External fault 2, input terminal S2 (EF2)");
        alarmDesc.put(ALARM_SAFE_DISABLE_INPUT, "Safe disable input (HbbF)");
        alarmDesc.put(ALARM_SAFE_DISABLE_INPUTX, "Safe disable input (Hbb)");
        alarmDesc.put(ALARM_MECH_WEAKENING_1, "Mechanical weakening detection 1 (oL5)");
        alarmDesc.put(ALARM_MECH_WEAKENING_2, "Mechanical weakening detection 2 (UL5)");
        alarmDesc.put(ALARM_PLC_1, "PLC alarm (PA1)");
        alarmDesc.put(ALARM_PLC_2, "PLC alarm (PA2)");
        alarmDesc.put(ALARM_VOLTAGE_DETC_FAULT, "Output voltage detection fault (voF)");
        alarmDesc.put(ALARM_IGBT_MAINT_TIME_90, "IGBT maintenance time (90%) (TrPC)");
        alarmDesc.put(ALARM_SCBR_MAINT_TIME, "Soft charge bypass relay maint. time (LT-3)");
        alarmDesc.put(ALARM_IGBT_MAINT_TIME_50, "IGBT maintenance time (50%) (LT-4)");
        alarmDesc.put(ALARM_MOTOR_OVERHEAT_NTC, "Motor overheat (NTC input) (oH5)");
        alarmDesc.put(ALARM_DRIVE_WORKS_EZ, "DriveWorks EZ alarm (dWAL)");
        alarmDesc.put(ALARM_THERMISTOR_DISC, "Thermistor disconnect (THo)");
    }


    public static final int
        FAULT_NONE                   = 0x000,
        FAULT_BLOWN_FUSE             = 0x001,
        FAULT_UNDERVOLTAGE           = 0x002,
        FAULT_CPS_UNDERVOLTAGE       = 0x003,
        FAULT_SOFT_CHARGE_CIRCUIT    = 0x004,
        FAULT_GROUND_FAULT           = 0x006,
        FAULT_OVERCURRENT            = 0x007,
        FAULT_OVERVOLTAGE            = 0x008,
        FAULT_HEATSINK_OVERHEAT      = 0x009,
        FAULT_HEATSINK_OVERHEAT1     = 0x00a,
        FAULT_MOTOR_OVERLOAD         = 0x00b,
        FAULT_DRIVE_OVERLOAD         = 0x00c,
        FAULT_OVERTORQUE_1           = 0x00d,
        FAULT_OVERTORQUE_2           = 0x00e,
        FAULT_DYN_BRAKING_TRANS      = 0x00f,
        FAULT_BRAKING_RESISTOR       = 0x010,
        FAULT_EXTERNAL_FAULT_3       = 0x011,
        FAULT_EXTERNAL_FAULT_4       = 0x012,
        FAULT_EXTERNAL_FAULT_5       = 0x013,
        FAULT_EXTERNAL_FAULT_6       = 0x014,
        FAULT_EXTERNAL_FAULT_7       = 0x015,
        FAULT_EXTERNAL_FAULT_8       = 0x016,
        FAULT_COOLING_FAN_FAIL       = 0x017,
        FAULT_OVERSPEED              = 0x018,
        FAULT_EXCESS_SPEED_DEVN      = 0x019,
        FAULT_PG_DISCONNECT          = 0x01a,
        FAULT_INPUT_PHASE_LOSS       = 0x01b,
        FAULT_OUTPUT_PHASE_LOSS      = 0x01c,
        FAULT_MOTOR_OVERHEAT_PTC1    = 0x01d,
        FAULT_DIGITAL_OPER_CONN      = 0x01e,
        FAULT_EEPROM_WRITE_ERROR     = 0x01f,
        FAULT_MOTOR_OVERHEAT_PTC2    = 0x020,
        FAULT_MODBUS_COMM_ERROR      = 0x021,
        FAULT_OPTION_COMM_ERROR      = 0x022,
        FAULT_CONTROL                = 0x025,
        FAULT_ZERO_SERVO             = 0x026,
        FAULT_OPTION_EXTERNAL        = 0x027,
        FAULT_PID_FEEDBACK_LOSS      = 0x028,
        FAULT_UNDERTORQUE_1          = 0x029,
        FAULT_UNDERTORQUE_2          = 0x02a,
        FAULT_HARDWARE               = 0x030,
        FAULT_OUTPUT_CURR_IMBAL      = 0x036,
        FAULT_PG_HARDWARE            = 0x038,
        FAULT_EXCESS_SS_RESTARTS     = 0x03b,
        FAULT_EXC_PID_FEEDBACK       = 0x041,
        FAULT_EXTERNAL_FAULT_1       = 0x042,
        FAULT_EXTERNAL_FAULT_2       = 0x043,
        FAULT_MECH_WEAKENING_1       = 0x044,
        FAULT_MECH_WEAKENING_2       = 0x045,
        FAULT_CURRENT_OFFSET         = 0x046,
        FAULT_PLC_ERROR_1            = 0x047,
        FAULT_PLC_ERROR_2            = 0x048,
        FAULT_DRIVEWORKS_EZ          = 0x049,
        FAULT_OUTPUT_VOLT_DETC       = 0x04d,
        FAULT_MOTOR_OVERHEAT_NTC     = 0x050,
        FAULT_NODE_SETUP             = 0x052,
        FAULT_THERMISTOR_DISC        = 0x053,
        FAULT_CURR_UNBALANCE         = 0x060,
        FAULT_PS_UNDERVOLTAGE        = 0x061,
        FAULT_MC_FAN_POWER_FAIL      = 0x062,
        FAULT_THERMAL_CONVERTER      = 0x063,
        FAULT_ABNORMAL_FAN_DIODE     = 0x064,
        FAULT_ABNORMAL_FAN_PANEL     = 0x065,
        FAULT_AD_CONV_ERROR_1        = 0x083,
        FAULT_PWM_FEEDBACK           = 0x084,
        FAULT_EEPROM_DATA_ERROR      = 0x087,
        FAULT_TERM_BOARD_CONN_ERR    = 0x088,
        FAULT_EEPROM_SERIAL_COMM     = 0x089,
        FAULT_RAM_FAULT              = 0x08c,
        FAULT_FLASH_MEM_EXCPTN       = 0x08d,
        FAULT_WATCHDOG_EXCPTN        = 0x08e,
        FAULT_CONTROL_CIRCUIT_1      = 0x08f,
        FAULT_CLOCK_FAULT            = 0x091,
        FAULT_TIMING_FAULT           = 0x092,
        FAULT_CONTROL_CIRCUIT_2      = 0x093,
        FAULT_CONTROL_CIRCUIT_3      = 0x094,
        FAULT_HW_FAULT_POWERUP       = 0x095,
        FAULT_HW_FAULT_COMM_START    = 0x096,
        FAULT_AD_CONV_FAULT          = 0x097,
        FAULT_PWM_FEEDBACK_FAULT     = 0x098,
        FAULT_DRIVE_SIGNAL_FAULT     = 0x099,
        FAULT_ASIC_BB_CIRC_ERROR     = 0x09b,
        FAULT_ASIC_PWM_SET_ERROR     = 0x09c,
        FAULT_ASIC_PWM_PATT_ERROR    = 0x09d,
        FAULT_ASIC_ONDELAY_ERROR     = 0x09e,
        FAULT_ASIC_BBON_ERROR        = 0x09f,
        FAULT_ASIC_CODE_ERROR        = 0x0a0,
        FAULT_ASIC_STARTUP_ERROR     = 0x0a1,
        FAULT_WATCHDOG_ERROR         = 0x0a2,
        FAULT_ASIC_POWER_ERROR       = 0x0a3,
        FAULT_EXT_AD_CONV_ERROR      = 0x0a4,
        FAULT_CONTROL_CIRC_ERR_1     = 0x0a9,
        FAULT_CONTROL_CIRC_ERR_2     = 0x0aa,
        FAULT_CONTROL_CIRC_ERR_3     = 0x0ab,
        FAULT_CONTROL_CIRC_ERR_4     = 0x0ac,
        FAULT_CONTROL_CIRC_ERR_5     = 0x0ad,
        FAULT_CONTROL_CIRC_ERR_6     = 0x0ae,
        FAULT_OPTION_COMPAT_ERR_A    = 0x101,
        FAULT_OPTION_CONN_ERR_A      = 0x102,
        FAULT_DUPL_OPTION_ERR_A      = 0x103,
        FAULT_AD_CONV_ERROR_A        = 0x106,
        FAULT_OPTION_RESP_ERR_A      = 0x107,
        FAULT_OPTION_RAM_FAULT_A     = 0x111,
        FAULT_OPTION_OPMODE_FAULT_A  = 0x112,
        FAULT_DRIVE_RCV_CRC_ERR_A    = 0x113,
        FAULT_DRIVE_RCV_FRAME_ERR_A  = 0x114,
        FAULT_DRIVE_RCV_ABORT_ERR_A  = 0x115,
        FAULT_OPTION_RCV_CRC_ERR_A   = 0x116,
        FAULT_OPTION_RCV_FRAME_ERR_A = 0x117,
        FAULT_OPTION_RCV_ABORT_ERR_A = 0x118,
        FAULT_COMM_ID_ERR_A          = 0x131,
        FAULT_MODEL_CODE_ERR_A       = 0x132,
        FAULT_CHECKSUM_ERR_A         = 0x133,
        FAULT_OPTION_TIMEOUT_A      = 0x134,
        FAULT_MODBUS_TIMEOUT_A       = 0x135,
        FAULT_DRIVE_TIMEOUT1_A        = 0x136,
        FAULT_CI_CHECK_ERR_A         = 0x137,
        FAULT_DRIVE_TIMEOUT2_A       = 0x138,
        FAULT_CTRL_COMMMAND_ERR_A      = 0x139,
        FAULT_DRIVE_TIMEOUT3_A       = 0x13a,
        FAULT_CTRL_RESP_ERR1_A       = 0x13b,
        FAULT_DRIVE_TIMEOUT4_A       = 0x13c,
        FAULT_CTRL_RESP_ERR2_A       = 0x13d,
        FAULT_CTRL_RESP_ERR3_A       = 0x13e,
        FAULT_OPTION_CONN_ERR_B      = 0x201,
        FAULT_DUPL_OPTION_ERR_B      = 0x202,
        FAULT_AD_CONV_ERR_B          = 0x205,
        FAULT_OPTION_RESP_ERR_B      = 0x206,
        FAULT_OPTION_RAM_FAULT_B     = 0x210,
        FAULT_OPTION_OPMODE_FAULT_B  = 0x211,
        FAULT_DRIVE_RCV_CRC_ERR_B    = 0x212,
        FAULT_DRIVE_RCV_FRAME_ERR_B  = 0x213,
        FAULT_DRIVE_RCV_ABORT_ERR_B  = 0x214,
        FAULT_OPTION_RCV_CRC_ERR_B   = 0x215,
        FAULT_OPTION_RCV_FRAME_ERR_B = 0x216,
        FAULT_OPTION_RCV_ABORT_ERR_B = 0x217,
        FAULT_COMM_ID_ERROR_B        = 0x231,
        FAULT_MODEL_CODE_ERR_B       = 0x232,
        FAULT_CHECKSUM_ERR_B         = 0x233,
        FAULT_OPTION_TIMEOUT_B       = 0x234,
        FAULT_MODBUS_TIMEOUT_B       = 0x235,
        FAULT_DRIVE_TIMEOUT1_B       = 0x236,
        FAULT_CI_CHECK_ERROR_B       = 0x237,
        FAULT_DRIVE_TIMEOUT2_B       = 0x238,
        FAULT_CTRL_COMMMAND_ERR_B    = 0x239,
        FAULT_DRIVE_TIMEOUT3_B       = 0x23a,
        FAULT_CTRL_RESP_ERR1_B       = 0x23b,
        FAULT_DRIVE_TIMEOUT4_B       = 0x23c,
        FAULT_CTRL_RESP_ERR2_B       = 0x23d,
        FAULT_CTRL_RESP_ERR3_B       = 0x23e,
        FAULT_OPTION_COMPAT_ERR_C    = 0x300,
        FAULT_OPTION_CONN_ERR_C      = 0x301,
        FAULT_DUPL_OPTION_ERR_C      = 0x302,
        FAULT_AD_CONV_ERR_C          = 0x305,
        FAULT_OPTION_RESP_ERR_C      = 0x306;

    private static final Map<Integer, String> faultDesc = new HashMap<>();
    static {
        faultDesc.put(FAULT_BLOWN_FUSE, "Blown fuse (PUF)");
        faultDesc.put(FAULT_UNDERVOLTAGE, "Undervoltage (Uv1)");
        faultDesc.put(FAULT_CPS_UNDERVOLTAGE, "Control p[ower supply undervoltage (Uv2)");
        faultDesc.put(FAULT_SOFT_CHARGE_CIRCUIT, "Soft charge circuit fault (Uv3)");
        faultDesc.put(FAULT_GROUND_FAULT, "Ground fault (GF)");
        faultDesc.put(FAULT_OVERCURRENT, "Overcurrent (oC)");
        faultDesc.put(FAULT_OVERVOLTAGE, "Overvoltage (ov)");
        faultDesc.put(FAULT_HEATSINK_OVERHEAT, "Heatsink overheat (oH)");
        faultDesc.put(FAULT_HEATSINK_OVERHEAT1, "Heatsink overheat (oH1)");
        faultDesc.put(FAULT_MOTOR_OVERLOAD, "Motor overload (oL1)");
        faultDesc.put(FAULT_DRIVE_OVERLOAD, "Drive overload (oL2)");
        faultDesc.put(FAULT_OVERTORQUE_1, "Overtorque 1 (oL3)");
        faultDesc.put(FAULT_OVERTORQUE_2, "Overtorque 2 (oL4)");
        faultDesc.put(FAULT_DYN_BRAKING_TRANS, "Dynamic braking transistor (rr)");
        faultDesc.put(FAULT_BRAKING_RESISTOR, "Braking resistor (rH)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_3, "External fault at input terminal S3 (EF3)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_4, "External fault at input terminal S4 (EF4)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_5, "External fault at input terminal S5 (EF5)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_6, "External fault at input terminal S6 (EF6)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_7, "External fault at input terminal S7 (EF7)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_8, "External fault at input terminal S8 (EF8)");
        faultDesc.put(FAULT_COOLING_FAN_FAIL, "Drive cooling fan failure (FAn)");
        faultDesc.put(FAULT_OVERSPEED, "Overspeed (oS)");
        faultDesc.put(FAULT_EXCESS_SPEED_DEVN, "Excessive speed deviation (dEv)");
        faultDesc.put(FAULT_PG_DISCONNECT, "PG disconnect (PGo)");
        faultDesc.put(FAULT_INPUT_PHASE_LOSS, "Input phase loss (PF)");
        faultDesc.put(FAULT_OUTPUT_PHASE_LOSS, "Output phase loss (LF)");
        faultDesc.put(FAULT_MOTOR_OVERHEAT_PTC1, "Motor overheat (PTC input) (oH3)");
        faultDesc.put(FAULT_DIGITAL_OPER_CONN, "Digital operator connection (oPr)");
        faultDesc.put(FAULT_EEPROM_WRITE_ERROR, "EEPROM write error (Err)");
        faultDesc.put(FAULT_MOTOR_OVERHEAT_PTC2, "Motor overheat (PTC input) (oH4)");
        faultDesc.put(FAULT_MODBUS_COMM_ERROR, "Modbus commuication error (CE)");
        faultDesc.put(FAULT_OPTION_COMM_ERROR, "Option communication error (bUS)");
        faultDesc.put(FAULT_CONTROL, "Control fault (CF)");
        faultDesc.put(FAULT_ZERO_SERVO, "Zero-server fault (SvE)");
        faultDesc.put(FAULT_OPTION_EXTERNAL, "Option external fault (EF0)");
        faultDesc.put(FAULT_PID_FEEDBACK_LOSS, "PID feedback loss (FbL)");
        faultDesc.put(FAULT_UNDERTORQUE_1, "Undertorque 1 (UL3)");
        faultDesc.put(FAULT_UNDERTORQUE_2, "Undertorque 2 (UL4)");
        faultDesc.put(FAULT_HARDWARE, "Hardware fault (including oFx)");
        faultDesc.put(FAULT_OUTPUT_CURR_IMBAL, "Output current imbalance (LF2)");
        faultDesc.put(FAULT_PG_HARDWARE, "PG hardware fault (PGoH)");
        faultDesc.put(FAULT_EXCESS_SS_RESTARTS, "Too many speed search restarts (SEr)");
        faultDesc.put(FAULT_EXC_PID_FEEDBACK, "Excessive PID feedback (FbH)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_1, "External fault at input terminal S1 (EF1)");
        faultDesc.put(FAULT_EXTERNAL_FAULT_2, "External fault at input terminal S2 (EF2)");
        faultDesc.put(FAULT_MECH_WEAKENING_1, "Mechanical weakening 1 (oL5)");
        faultDesc.put(FAULT_MECH_WEAKENING_2, "Mechanical weakening 2 (UL5)");
        faultDesc.put(FAULT_CURRENT_OFFSET, "Current offset fault (CoF)");
        faultDesc.put(FAULT_PLC_ERROR_1, "PLC error 1 (PE1)");
        faultDesc.put(FAULT_PLC_ERROR_2, "PLC error 2 (PE2)");
        faultDesc.put(FAULT_DRIVEWORKS_EZ, "DriveWorks EZ fault (dWFL)");
        faultDesc.put(FAULT_OUTPUT_VOLT_DETC, "Output voltage detection fault (voF)");
        faultDesc.put(FAULT_MOTOR_OVERHEAT_NTC, "Motor overheat (NTC inout) (oH5)");
        faultDesc.put(FAULT_NODE_SETUP, "Node setup fault (nSE)");
        faultDesc.put(FAULT_THERMISTOR_DISC, "Thermistor disconnect (THo)");
        faultDesc.put(FAULT_CURR_UNBALANCE, "Current unbalance (UnbC)");
        faultDesc.put(FAULT_PS_UNDERVOLTAGE, "Power supply module undervoltage (Uv4)");
        faultDesc.put(FAULT_MC_FAN_POWER_FAIL, "MC fan power failure (Uv5)");
        faultDesc.put(FAULT_THERMAL_CONVERTER, "Thermal converter fault (oH6)");
        faultDesc.put(FAULT_ABNORMAL_FAN_DIODE, "Abnormal fan diode module (dFAn)");
        faultDesc.put(FAULT_ABNORMAL_FAN_PANEL, "Abnormal fan panel (EFAn)");
        faultDesc.put(FAULT_AD_CONV_ERROR_1, "A/D conversion fault (CPF02)");
        faultDesc.put(FAULT_PWM_FEEDBACK, "PWM feedback fault (CPF03)");
        faultDesc.put(FAULT_EEPROM_DATA_ERROR, "EEPROM data error (CPF06)");
        faultDesc.put(FAULT_TERM_BOARD_CONN_ERR, "Terminal board connection error (CPF07)");
        faultDesc.put(FAULT_EEPROM_SERIAL_COMM, "EEPROM serial communication fault (CPF08)");
        faultDesc.put(FAULT_RAM_FAULT, "RAM fault (CPF11)");
        faultDesc.put(FAULT_FLASH_MEM_EXCPTN, "Flash memory exception (CPF12)");
        faultDesc.put(FAULT_WATCHDOG_EXCPTN, "Watchdog exception (CPF13)");
        faultDesc.put(FAULT_CONTROL_CIRCUIT_1, "Control circuit fault (CPF14)");
        faultDesc.put(FAULT_CLOCK_FAULT, "Clock fault (CPF16)");
        faultDesc.put(FAULT_TIMING_FAULT, "Timing fault (CPF17)");
        faultDesc.put(FAULT_CONTROL_CIRCUIT_2, "Control circuit fault (CPF18)");
        faultDesc.put(FAULT_CONTROL_CIRCUIT_3, "Control circuit fault (CPF19)");
        faultDesc.put(FAULT_HW_FAULT_POWERUP, "Hardware fault at power-up (CPF20)");
        faultDesc.put(FAULT_HW_FAULT_COMM_START, "Hardware fault at communication startup (CPF21)");
        faultDesc.put(FAULT_AD_CONV_FAULT, "A/D conversion fault (CPF22)");
        faultDesc.put(FAULT_PWM_FEEDBACK_FAULT, "PWM feedback fault (CPF23)");
        faultDesc.put(FAULT_DRIVE_SIGNAL_FAULT, "Drive unit signal fault (CPF24)");
        faultDesc.put(FAULT_ASIC_BB_CIRC_ERROR, "ASIC BB circuit error (CPF26)");
        faultDesc.put(FAULT_ASIC_PWM_SET_ERROR, "ASIC PWM setting register error (CPF27)");
        faultDesc.put(FAULT_ASIC_PWM_PATT_ERROR, "ASIC PWM pattern error (CPF28)");
        faultDesc.put(FAULT_ASIC_ONDELAY_ERROR, "ASIC on-delay error (CPF29)");
        faultDesc.put(FAULT_ASIC_BBON_ERROR, "ASIC BBON error (CPF30)");
        faultDesc.put(FAULT_ASIC_CODE_ERROR, "ASIC code error (CPF31)");
        faultDesc.put(FAULT_ASIC_STARTUP_ERROR, "ASIC startup error (CPF32)");
        faultDesc.put(FAULT_WATCHDOG_ERROR, "Watchdog error (CPF33)");
        faultDesc.put(FAULT_ASIC_POWER_ERROR, "ASIC power clock error (CPF34)");
        faultDesc.put(FAULT_EXT_AD_CONV_ERROR, "External A/D conversion error (CPF35)");
        faultDesc.put(FAULT_CONTROL_CIRC_ERR_1, "Control circuit error (CPF40)");
        faultDesc.put(FAULT_CONTROL_CIRC_ERR_2, "Control circuit error (CPF41)");
        faultDesc.put(FAULT_CONTROL_CIRC_ERR_3, "Control circuit error (CPF42)");
        faultDesc.put(FAULT_CONTROL_CIRC_ERR_4, "Control circuit error (CPF43)");
        faultDesc.put(FAULT_CONTROL_CIRC_ERR_5, "Control circuit error (CPF44)");
        faultDesc.put(FAULT_CONTROL_CIRC_ERR_6, "Control circuit error (CPF45)");
        faultDesc.put(FAULT_OPTION_COMPAT_ERR_A, "Option compatibility error (oFA00)");
        faultDesc.put(FAULT_OPTION_CONN_ERR_A, "Option not properly connected (oFA01)");
        faultDesc.put(FAULT_DUPL_OPTION_ERR_A, "Option of same type already connected (oFA02)");
        faultDesc.put(FAULT_AD_CONV_ERROR_A, "A/D conversion error (oFA05)");
        faultDesc.put(FAULT_OPTION_RESP_ERR_A, "Option response error (oFA06)");
        faultDesc.put(FAULT_OPTION_RAM_FAULT_A, "Option RAM fault (oFA10)");
        faultDesc.put(FAULT_OPTION_OPMODE_FAULT_A, "Option operation mode fault (SLMOD) (oFA11)");
        faultDesc.put(FAULT_DRIVE_RCV_CRC_ERR_A, "Drive receive CRC error (oFA12)");
        faultDesc.put(FAULT_DRIVE_RCV_FRAME_ERR_A, "Drive reveive frsme error (oFA13)");
        faultDesc.put(FAULT_DRIVE_RCV_ABORT_ERR_A, "Drive receive abort error (oFA14)");
        faultDesc.put(FAULT_OPTION_RCV_CRC_ERR_A, "Option receive CRC error (oFA15)");
        faultDesc.put(FAULT_OPTION_RCV_FRAME_ERR_A, "Option receive frame error (oFA16)");
        faultDesc.put(FAULT_OPTION_RCV_ABORT_ERR_A, "Option receive abort error (oFA17)");
        faultDesc.put(FAULT_COMM_ID_ERR_A, "Communication ID error (oFA30)");
        faultDesc.put(FAULT_MODEL_CODE_ERR_A, "Model code error (oFA31)");
        faultDesc.put(FAULT_CHECKSUM_ERR_A, "Checksum error (oFA32)");
        faultDesc.put(FAULT_OPTION_TIMEOUT_A, "Option response timeout (oFA33)");
        faultDesc.put(FAULT_MODBUS_TIMEOUT_A, "Modbus timeout (oFA34)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT1_A, "Drive response timeout (oFA35)");
        faultDesc.put(FAULT_CI_CHECK_ERR_A, "CI check error (oFA36)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT2_A, "Drive response timeout (oFA37)");
        faultDesc.put(FAULT_CTRL_COMMMAND_ERR_A, "Control command selection error (oFA38)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT3_A, "Drive response timeout (oFA39)");
        faultDesc.put(FAULT_CTRL_RESP_ERR1_A, "Control response selection 1 error (oFA40)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT4_A, "Drive response timeout (oFA41)");
        faultDesc.put(FAULT_CTRL_RESP_ERR2_A, "Control rsponse election 2 error (oFA42)");
        faultDesc.put(FAULT_CTRL_RESP_ERR3_A, "Control response selection error (oFA43)");
        faultDesc.put(FAULT_OPTION_CONN_ERR_B, "Option not properly connected (oFb01)");
        faultDesc.put(FAULT_DUPL_OPTION_ERR_B, "Option of same type already connected (oFb02)");
        faultDesc.put(FAULT_AD_CONV_ERR_B, "A/D conversion error (oFb05)");
        faultDesc.put(FAULT_OPTION_RESP_ERR_B, "Option response error (oFb06)");
        faultDesc.put(FAULT_OPTION_RAM_FAULT_B, "Option RAM fault (oFb10)");
        faultDesc.put(FAULT_OPTION_OPMODE_FAULT_B, "Option operation mode fault (SLMOD) (oFb11)");
        faultDesc.put(FAULT_DRIVE_RCV_CRC_ERR_B, "Drive receive CRC error (oFb12)");
        faultDesc.put(FAULT_DRIVE_RCV_FRAME_ERR_B, "Drive reveive frsme error (oFb13)");
        faultDesc.put(FAULT_DRIVE_RCV_ABORT_ERR_B, "Drive receive abort error (oFb14)");
        faultDesc.put(FAULT_OPTION_RCV_CRC_ERR_B, "Option receive CRC error (oFb15)");
        faultDesc.put(FAULT_OPTION_RCV_FRAME_ERR_B, "Option receive frame error (oFb16)");
        faultDesc.put(FAULT_OPTION_RCV_ABORT_ERR_B, "Option receive abort error (oFb17)");
        faultDesc.put(FAULT_COMM_ID_ERROR_B, "Communication ID error (oFb30)");
        faultDesc.put(FAULT_MODEL_CODE_ERR_B, "Model code error (oFb31)");
        faultDesc.put(FAULT_CHECKSUM_ERR_B, "Checksum error (oFb32)");
        faultDesc.put(FAULT_OPTION_TIMEOUT_B, "Option response timeout (oFb33)");
        faultDesc.put(FAULT_MODBUS_TIMEOUT_B, "Modbus timeout (oFb34)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT1_B, "Drive response timeout (oFb35)");
        faultDesc.put(FAULT_CI_CHECK_ERROR_B, "CI check error (oFb36)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT2_B, "Drive response timeout (oFb35)");
        faultDesc.put(FAULT_CTRL_COMMMAND_ERR_B, "Control command selection error (oFb38)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT3_B, "Drive response timeout (oFb39)");
        faultDesc.put(FAULT_CTRL_RESP_ERR1_B, "Control response selection 1 error (oFb40)");
        faultDesc.put(FAULT_DRIVE_TIMEOUT4_B, "Drive response timeout (oFb41)");
        faultDesc.put(FAULT_CTRL_RESP_ERR2_B, "Control response election 2 error (oFb42)");
        faultDesc.put(FAULT_CTRL_RESP_ERR3_B, "Control response selection error (oFb43)");
        faultDesc.put(FAULT_OPTION_COMPAT_ERR_C, "Option compatibility error (oFC00)");
        faultDesc.put(FAULT_OPTION_CONN_ERR_C, "Option not properly connected (oFC01)");
        faultDesc.put(FAULT_DUPL_OPTION_ERR_C, "Option of same type already connected (oFC02)");
        faultDesc.put(FAULT_AD_CONV_ERR_C, "A/D conversion error (oFC05)");
        faultDesc.put(FAULT_OPTION_RESP_ERR_C, "Option response error (oFC06)");
    }

    private static final Map<Language, Short> languageEnumMap = new HashMap<>();
    private static final Map<Short, Language> languageValueMap = new HashMap<>();
    static {
        languageEnumMap.put(Language.ENGLISH, LANG_ENGLISH);
        for (Map.Entry e : languageEnumMap.entrySet()) {
            languageValueMap.put((Short)e.getValue(), (Language)e.getKey());
        }
    }
    private static final Map<AccessLevel, Short> accessLevelEnumMap = new HashMap<>();
    private static final Map<Short, AccessLevel> accessLevelValueMap = new HashMap<>();
    static {
        accessLevelEnumMap.put(AccessLevel.OPR_ONLY, ACC_LEVEL_OPR_ONLY);
        accessLevelEnumMap.put(AccessLevel.USR_PARM, ACC_LEVEL_USR_PARM);
        accessLevelEnumMap.put(AccessLevel.FULL, ACC_LEVEL_FULL);
        for (Map.Entry e : accessLevelEnumMap.entrySet()) {
            accessLevelValueMap.put((Short)e.getValue(), (AccessLevel)e.getKey());
        }
    }
    private static final Map<ControlMode, Short> controlModeEnumMap = new HashMap<>();
    private static final Map<Short, ControlMode> controlModeValueMap = new HashMap<>();
    static {
        controlModeEnumMap.put(ControlMode.VF, CTRL_MODE_VF);
        controlModeEnumMap.put(ControlMode.VF_PG, CTRL_MODE_VF_PG);
        controlModeEnumMap.put(ControlMode.OLV, CTRL_MODE_OLV);
        controlModeEnumMap.put(ControlMode.CLV, CTRL_MODE_CLV);
        controlModeEnumMap.put(ControlMode.CLV_PM, CTRL_MODE_CLV_PM);
        controlModeEnumMap.put(ControlMode.OLV_PM, CTRL_MODE_OLV_PM);
        controlModeEnumMap.put(ControlMode.AOLV_PM, CTRL_MODE_AOLV_PM);
        for (Map.Entry e : controlModeEnumMap.entrySet()) {
            controlModeValueMap.put((Short)e.getValue(), (ControlMode)e.getKey());
        }
    }
    private static final Map<FreqRefSrc, Short> freqRefSrcEnumMap = new HashMap<>();
    private static final Map<Short, FreqRefSrc> freqRefSrcValueMap = new HashMap<>();
    static {
        freqRefSrcEnumMap.put(FreqRefSrc.OPERATOR, FREQ_REF_OPERATOR);
        freqRefSrcEnumMap.put(FreqRefSrc.ANALOG, FREQ_REF_ANALOG);
        freqRefSrcEnumMap.put(FreqRefSrc.MODBUS, FREQ_REF_MODBUS);
        freqRefSrcEnumMap.put(FreqRefSrc.OPTN_PCB, FREQ_REF_OPTN_PCB);
        freqRefSrcEnumMap.put(FreqRefSrc.PULSE, FREQ_REF_PULSE);
        for (Map.Entry e : freqRefSrcEnumMap.entrySet()) {
            freqRefSrcValueMap.put((Short)e.getValue(), (FreqRefSrc)e.getKey());
        }
    }
    private static final Map<RunCmndSrc, Short> runCmndSrcEnumMap = new HashMap<>();
    private static final Map<Short, RunCmndSrc> runCmndSrcValueMap = new HashMap<>();
    static {
        runCmndSrcEnumMap.put(RunCmndSrc.OPERATOR, RUN_CMND_OPERATOR);
        runCmndSrcEnumMap.put(RunCmndSrc.DIGITAL, RUN_CMND_DIGITAL);
        runCmndSrcEnumMap.put(RunCmndSrc.MODBUS, RUN_CMND_MODBUS);
        runCmndSrcEnumMap.put(RunCmndSrc.OPTN_PCB, RUN_CMND_OPTN_PCB);
        for (Map.Entry e : runCmndSrcEnumMap.entrySet()) {
            runCmndSrcValueMap.put((Short)e.getValue(), (RunCmndSrc)e.getKey());
        }
    }
    private static final Map<StopMethod, Short> stopMethodEnumMap = new HashMap<>();
    private static final Map<Short, StopMethod> stopMethodValueMap = new HashMap<>();
    static {
        stopMethodEnumMap.put(StopMethod.COAST, STOP_METH_COAST);
        stopMethodEnumMap.put(StopMethod.COAST_TMR, STOP_METH_COAST_TM);
        stopMethodEnumMap.put(StopMethod.INJECT, STOP_METH_INJECT);
        stopMethodEnumMap.put(StopMethod.RAMP, STOP_METH_RAMP);
        for (Map.Entry e : stopMethodEnumMap.entrySet()) {
            stopMethodValueMap.put((Short)e.getValue(), (StopMethod)e.getKey());
        }
    }
    private static final Map<RunProg, Short> runProgEnumMap = new HashMap<>();
    private static final Map<Short, RunProg> runProgValueMap = new HashMap<>();
    static {
        runProgEnumMap.put(RunProg.PROG_ONLY, PROG_MODE_ONLY);
        runProgEnumMap.put(RunProg.RUN_PROG, RUN_PROG_OKAY);
        runProgEnumMap.put(RunProg.RUN_ONLY, RUN_MODE_ONLY);
        for (Map.Entry e : runProgEnumMap.entrySet()) {
            runProgValueMap.put((Short)e.getValue(), (RunProg)e.getKey());
        }
    }
    private static final Map<BaudRate, Short> baudRateEnumMap = new HashMap<>();
    private static final Map<Short, BaudRate> baudRateValueMap = new HashMap<>();
    static {
        baudRateEnumMap.put(BaudRate.B1200, BAUD_RATE_1200);
        baudRateEnumMap.put(BaudRate.B2400, BAUD_RATE_2400);
        baudRateEnumMap.put(BaudRate.B4800, BAUD_RATE_4800);
        baudRateEnumMap.put(BaudRate.B9600, BAUD_RATE_9600);
        baudRateEnumMap.put(BaudRate.B19200, BAUD_RATE_19200);
        baudRateEnumMap.put(BaudRate.B38400, BAUD_RATE_38400);
        baudRateEnumMap.put(BaudRate.B57600, BAUD_RATE_57600);
        baudRateEnumMap.put(BaudRate.B76800, BAUD_RATE_76800);
        baudRateEnumMap.put(BaudRate.B115200, BAUD_RATE_115200);
        for (Map.Entry e : baudRateEnumMap.entrySet()) {
            baudRateValueMap.put((Short)e.getValue(), (BaudRate)e.getKey());
        }
    }
    private static final Map<Parity, Short> parityEnumMap = new HashMap<>();
    private static final Map<Short, Parity> parityValueMap = new HashMap<>();
    static {
        parityEnumMap.put(Parity.EVEN, PARITY_EVEN);
        parityEnumMap.put(Parity.ODD, PARITY_ODD);
        parityEnumMap.put(Parity.NONE, PARITY_NONE);
        for (Map.Entry e : parityEnumMap.entrySet()) {
            parityValueMap.put((Short)e.getValue(), (Parity)e.getKey());
        }
    }
    private static final Map<ErrorAction, Short> errorActionEnumMap = new HashMap<>();
    private static final Map<Short, ErrorAction> errorActionValueMap = new HashMap<>();
    static {
        errorActionEnumMap.put(ErrorAction.RAMP_STOP, ERROR_RAMP_STOP);
        errorActionEnumMap.put(ErrorAction.COAST_STOP, ERROR_COAST_STOP);
        errorActionEnumMap.put(ErrorAction.FAST_STOP, ERROR_FAST_STOP);
        errorActionEnumMap.put(ErrorAction.ALARM_ONLY, ERROR_ALARM_ONLY);
        for (Map.Entry e : errorActionEnumMap.entrySet()) {
            errorActionValueMap.put((Short)e.getValue(), (ErrorAction)e.getKey());
        }
    }
    private static final Map<VoltageUnits, Short> voltageUnitsEnumMap = new HashMap<>();
    private static final Map<Short, VoltageUnits> voltageUnitsValueMap = new HashMap<>();
    static {
        voltageUnitsEnumMap.put(VoltageUnits.ONES, VOLT_UNITS_ONES);
        voltageUnitsEnumMap.put(VoltageUnits.TENTHS, VOLT_UNITS_TENTHS);
        for (Map.Entry e : voltageUnitsEnumMap.entrySet()) {
            voltageUnitsValueMap.put((Short)e.getValue(), (VoltageUnits)e.getKey());
        }
    }
    private static final Map<VfSelection, Short> vfSelectionEnumMap = new HashMap<>();
    private static final Map<Short, VfSelection> vfSelectionValueMap = new HashMap<>();
    static {
        vfSelectionEnumMap.put(VfSelection.HZ50, VF_50HZ);
        vfSelectionEnumMap.put(VfSelection.HZ60_SAT, VF_60HZ_SAT);
        vfSelectionEnumMap.put(VfSelection.HZ50_SAT, VF_50HZ_SAT);
        vfSelectionEnumMap.put(VfSelection.HZ72, VF_72HZ);
        vfSelectionEnumMap.put(VfSelection.HZ50_VT1, VF_50HZ_VT1);
        vfSelectionEnumMap.put(VfSelection.HZ50_VT2, VF_50HZ_VT2);
        vfSelectionEnumMap.put(VfSelection.HZ60_VT1, VF_60HZ_VT1);
        vfSelectionEnumMap.put(VfSelection.HZ60_VT2, VF_60HZ_VT2);
        vfSelectionEnumMap.put(VfSelection.HZ50_HST1, VF_50HZ_HST1);
        vfSelectionEnumMap.put(VfSelection.HZ50_HST2, VF_50HZ_HST2);
        vfSelectionEnumMap.put(VfSelection.HZ60_HST1, VF_60HZ_HST1);
        vfSelectionEnumMap.put(VfSelection.HZ60_HST2, VF_60HZ_HST2);
        vfSelectionEnumMap.put(VfSelection.HZ90, VF_90HZ);
        vfSelectionEnumMap.put(VfSelection.HZ120, VF_120HZ);
        vfSelectionEnumMap.put(VfSelection.HZ180, VF_180HZ);
        vfSelectionEnumMap.put(VfSelection.CUSTOM, VF_CUSTOM);
        for (Map.Entry e : vfSelectionEnumMap.entrySet()) {
            vfSelectionValueMap.put((Short)e.getValue(), (VfSelection)e.getKey());
        }
    }
    private static final Map<Short, String> modelNameMap = new HashMap<>();
    static {
        modelNameMap.put(MODEL_4A0002, "4A0002");
        modelNameMap.put(MODEL_4A0004, "4A0004");
        modelNameMap.put(MODEL_4A0005, "4A0005");
        modelNameMap.put(MODEL_4A0007, "4A0007");
        modelNameMap.put(MODEL_4A0009, "4A0009");
    }
    private static final Map<Short, Double> ratedCurrentMap = new HashMap<>();
    static {
        ratedCurrentMap.put(MODEL_4A0002, 2.1);
        ratedCurrentMap.put(MODEL_4A0004, 4.1);
        ratedCurrentMap.put(MODEL_4A0005, 5.4);
        ratedCurrentMap.put(MODEL_4A0007, 6.9);
        ratedCurrentMap.put(MODEL_4A0009, 8.8);
    }

    /**
     *  Private data.
     */
    private static final int
        BAUD_RATE = 115200,
        DEVC_ADDR = 0x1f;
    private static final short
        ZERO = 0,
        ONE  = 1,
        TWO  = 2;
    private static final int
        EXCP_BIT_COUNT = 3,
        EXCP_SET_DATA = 33,
        EXCP_WRITE_MODE = 34,
        EXCP_BUS_UNDERVOLT = 35,
        EXCP_WRITE_PARAM = 36,
        EXCP_WRITE_EEPROM = 37;

    private static final Map<Integer, String> excpMap = new HashMap<>();
    static {
        excpMap.put(EXCP_BIT_COUNT, "bit count error");
        excpMap.put(EXCP_SET_DATA, "illegal data value");
        excpMap.put(EXCP_WRITE_MODE, "illegal write operation");
        excpMap.put(EXCP_BUS_UNDERVOLT, "illegal write while in undervoltage");
        excpMap.put(EXCP_WRITE_PARAM, "illegal write while processing parameters");
        excpMap.put(EXCP_WRITE_EEPROM, "illegal write to EEPROM");
    }
    private static final double
        VOLT_UNITS = 0.1,
        FREQ_UNITS = 0.01,
        FREQ_LIMIT_UNITS = 0.1;

    private final Modbus mbus;
    private short devcAddr;
    private double ratedCurrent;
    private VoltageUnits voltageUnits = VoltageUnits.TENTHS;
    private boolean runCmndFwdRev = true;


    /**
     *  Constructor.
     */
    public A1000()
    {
        mbus = new Modbus();
        mbus.setOptions(Modbus.Option.NO_NET);
        mbus.setDefaultParm(BAUD_RATE);
        mbus.setAddressMode(true);
        mbus.setExceptionMap(excpMap);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @param  address   The slave address
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The data characteristics
     *  @throws  DriverException
     */
    public void open(String device, int address, int baudRate, int dataChar) throws DriverException
    {
        mbus.open(Modbus.ConnType.SERIAL, device, baudRate, dataChar);
        devcAddr = address != 0 ? (short)address : DEVC_ADDR;
        try {
            ratedCurrent = getRatedCurrent();
        }
        catch (DriverException e) {
            mbus.closeSilent();
            throw e;
        }
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @param  address   The slave address
     *  @param  baudRate  The baud rate
     *  @throws  DriverException
     */
    public void open(String device, int address, int baudRate) throws DriverException
    {
        open(device, address, baudRate, 0);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @param  address   The slave address
     *  @throws  DriverException
     */
    public void open(String device, int address) throws DriverException
    {
        open(device, address, BAUD_RATE);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @throws  DriverException
     */
    public void open(String device) throws DriverException
    {
        open(device, DEVC_ADDR);
    }


    /**
     *  Closes the connection to the device.
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException
    {
        mbus.close();
    }


    /**
     *  Reads a set of registers.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read
     *  @return  An array of register values
     *  @throws DriverException
     */
    public short[] readRegisters(short addr, short count) throws DriverException
    {
        return mbus.readRegisters(devcAddr, addr, count);
    }


    /**
     *  Reads a register.
     *
     *  @param  addr   The register address
     *  @return  The read value as a signed integer
     *  @throws DriverException
     */
    public int readRegister(short addr) throws DriverException
    {
        return mbus.readRegisters(devcAddr, addr, ONE)[0];
    }


    /**
     *  Writes a register.
     *
     *  @param  addr   The register address
     *  @param  value  The value to write
     *  @throws DriverException
     */
    public void writeRegister(short addr, short value) throws DriverException
    {
        mbus.writeRegisters(devcAddr, addr, new short[]{value});
    }


    /**
     *  Writes a set of registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write
     *  @throws DriverException
     */
    public void writeRegisters(short addr, short[] value) throws DriverException
    {
        mbus.writeRegisters(devcAddr, addr, value);
    }


    /**
     *  Makes new parameter values available.
     *
     *  @throws  DriverException
     */
    public void enterParameters() throws DriverException
    {
        writeRegister(REG_ENTER_RAM, ZERO);
    }


    /**
     *  Saves parameter values in EEPROM.
     *
     *  @throws  DriverException
     */
    public void saveParameters() throws DriverException
    {
        writeRegister(REG_ENTER_EEPROM, ZERO);
    }


    /**
     *  Sets the node address.
     *
     *  @param  address  The device address (1 - 255)
     *  @throws  DriverException
     */
    public void setNodeAddress(int address) throws DriverException
    {
        writeRegister(REG_NODE_ADDRESS, (short)address);
    }


    /**
     *  Gets the node address.
     *
     *  @return  The device address (1 - 255)
     *  @throws  DriverException
     */
    public int getNodeAddress() throws DriverException
    {
        return readRegister(REG_NODE_ADDRESS);
    }


    /**
     *  Sets the drive baud rate.
     * 
     *  @param rate  The baud rate enum
     *  @throws DriverException 
     */
    public void setBaudRate(BaudRate rate) throws DriverException
    {
        writeRegister(REG_BAUD_RATE, baudRateEnumMap.get(rate));
    }


    /**
     *  Gets the drive baud rate
     * 
     *  @return  The baud rate enum
     *  @throws DriverException 
     */
    public BaudRate getBaudRate() throws DriverException
    {
        return baudRateValueMap.get((short)readRegister(REG_BAUD_RATE));
    }


    /**
     *  Sets the drive parity
     * 
     *  @param parity The parity enum
     *  @throws DriverException 
     */
    public void setParity(Parity parity) throws DriverException
    {
        writeRegister(REG_COMM_PARITY, parityEnumMap.get(parity));
    }


    /**
     *  Gets the drive parity.
     * 
     *  @return The parity enum
     *  @throws DriverException 
     */
    public Parity getParity() throws DriverException
    {
        return parityValueMap.get((short)readRegister(REG_COMM_PARITY));
    }


    /**
     *  Gets the model name
     * 
     *  @return The model name
     *  @throws DriverException 
     */
    public String getModelName() throws DriverException
    {
        return modelNameMap.get((short)readRegister(REG_DRIVE_MODEL));
    }


    /**
     *  Gets the rated current
     * 
     *  @return The rated current
     *  @throws DriverException 
     */
    public double getRatedCurrent() throws DriverException
    {
        return ratedCurrentMap.get((short)readRegister(REG_DRIVE_MODEL));
    }


    /**
     *  Sets control to be local.
     * 
     *  @throws DriverException 
     */
    public void setLocalControl() throws DriverException
    {
        setRunCmndSrc(RunCmndSrc.DIGITAL);
        setFreqRefSrc(FreqRefSrc.ANALOG);
        enterParameters();
    }


    /**
     *  Sets control to be remote.
     * 
     *  @throws DriverException 
     */
    public void setRemoteControl() throws DriverException
    {
        setRunCmndSrc(RunCmndSrc.MODBUS);
        setFreqRefSrc(FreqRefSrc.MODBUS);
        enterParameters();
    }


    /**
     *  Sets the access level
     * 
     *  @param level Access level enum
     *  @throws DriverException 
     */
    public void setAccessLevel(AccessLevel level) throws DriverException
    {
        writeRegister(REG_ACCESS_LEVEL, accessLevelEnumMap.get(level));
    }


    /**
     *  Gets the access level
     * 
     *  @return Access level enum
     *  @throws DriverException 
     */
    public AccessLevel getAccessLevel() throws DriverException
    {
        return accessLevelValueMap.get((short)readRegister(REG_ACCESS_LEVEL));
    }


    /**
     *  Sets the control mode
     * 
     *  @param mode Control mode enum
     *  @throws DriverException 
     */
    public void setControlMode(ControlMode mode) throws DriverException
    {
        writeRegister(REG_CTRL_MODE, controlModeEnumMap.get(mode));
    }


    /**
     *  Gets the control mode
     * 
     *  @return Control mode enum
     *  @throws DriverException 
     */
    public ControlMode getControlMode() throws DriverException
    {
        return controlModeValueMap.get((short)readRegister(REG_CTRL_MODE));
    }


    /**
     *  Sets the frequency reference source
     * 
     *  @param src Frequency reference source enum
     *  @throws DriverException 
     */
    public void setFreqRefSrc(FreqRefSrc src) throws DriverException
    {
        writeRegister(REG_FREQ_REF_SRC, freqRefSrcEnumMap.get(src));
    }


    /**
     *  Gets the frequency reference source
     * 
     *  @return Frequency reference source enum
     *  @throws DriverException 
     */
    public FreqRefSrc getFreqRefSrc() throws DriverException
    {
        return freqRefSrcValueMap.get((short)readRegister(REG_FREQ_REF_SRC));
    }


    /**
     *  Sets the run command source
     * 
     *  @param src Run command source enum
     *  @throws DriverException 
     */
    public void setRunCmndSrc(RunCmndSrc src) throws DriverException
    {
        writeRegister(REG_RUN_CMND_SRC, runCmndSrcEnumMap.get(src));
    }


    /**
     *  Gets the run command source
     * 
     *  @return Run command source enum
     *  @throws DriverException 
     */
    public RunCmndSrc getRunCmndSrc() throws DriverException
    {
        return runCmndSrcValueMap.get((short)readRegister(REG_RUN_CMND_SRC));
    }


    /**
     *  Sets the stop method
     * 
     *  @param method Stop method enum
     *  @throws DriverException 
     */
    public void setStopMethod(StopMethod method) throws DriverException
    {
        writeRegister(REG_STOP_METHOD, stopMethodEnumMap.get(method));
    }


    /**
     *  Gets the stop method
     * 
     *  @return Stop method enum
     *  @throws DriverException 
     */
    public StopMethod getStopMethod() throws DriverException
    {
        return stopMethodValueMap.get((short)readRegister(REG_STOP_METHOD));
    }


    /**
     *  Sets the interaction between running and programming
     * 
     *  @param runProg Interaction enum
     *  @throws DriverException 
     */
    public void setRunProg(RunProg runProg) throws DriverException
    {
        writeRegister(REG_RUN_CMND_PROG, runProgEnumMap.get(runProg));
    }


    /**
     *  Gets the interaction between running and programming
     * 
     *  @return Interaction enum
     *  @throws DriverException 
     */
    public RunProg getRunProg() throws DriverException
    {
        return runProgValueMap.get((short)readRegister(REG_RUN_CMND_PROG));
    }


    /**
     *  Sets the communications error action
     * 
     *  @param action Error action enum
     *  @throws DriverException 
     */
    public void setErrorAction(ErrorAction action) throws DriverException
    {
        writeRegister(REG_ERROR_ACTION, errorActionEnumMap.get(action));
    }


    /**
     *  Gets the communications error action
     * 
     *  @return Error action enum
     *  @throws DriverException 
     */
    public ErrorAction getErrorAction() throws DriverException
    {
        return errorActionValueMap.get((short)readRegister(REG_ERROR_ACTION));
    }


    /**
     *  Sets whether reverse running is permitted
     * 
     *  @param permit Whether reverse permitted
     *  @throws DriverException 
     */
    public void setReversePermitted(boolean permit) throws DriverException
    {
        writeRegister(REG_REV_PERMIT, permit ? REV_ENABLED : REV_DISABLED);
    }


    /**
     *  Gets whether reverse running is permitted
     * 
     *  @return Whether reverse permitted
     *  @throws DriverException 
     */
    public boolean isReversePermitted() throws DriverException
    {
        return readRegister(REG_REV_PERMIT) == REV_ENABLED;
    }


    /**
     *  Sets whether communications faults are detected
     * 
     *  @param detect Whether faults are detected
     *  @throws DriverException 
     */
    public void setFaultDetected(boolean detect) throws DriverException
    {
        writeRegister(REG_COMM_FAULT_DET, detect ? FAULT_DETC_ON : FAULT_DETC_OFF);
    }


    /**
     *  Gets whether communications faults are detected
     * 
     *  @return Whether faults are detected
     *  @throws DriverException 
     */
    public boolean isFaultDetected() throws DriverException
    {
        return readRegister(REG_COMM_FAULT_DET) == FAULT_DETC_ON;
    }


    /**
     *  Sets whether enter command is required for parameter application
     * 
     *  @param require Whether enter required
     *  @throws DriverException 
     */
    public void setEnterRequired(boolean require) throws DriverException
    {
        writeRegister(REG_ENTER_FUNC, require ? ENTER_REQUIRED : ENTER_NOT_REQUIRED);
    }


    /**
     *  Gets whether enter command is required for parameter application
     * 
     *  @return Whether enter required
     *  @throws DriverException 
     */
    public boolean isEnterRequired() throws DriverException
    {
        return readRegister(REG_ENTER_FUNC) == ENTER_REQUIRED;
    }


    /**
     *  Sets whether run command is forward/reverse style
     * 
     *  @param fwdRev Whether forward/reverse style
     *  @throws DriverException 
     */
    public void setRunCmndFwdRev(boolean fwdRev) throws DriverException
    {
        writeRegister(REG_RUN_CMND_TYPE, fwdRev ? RUN_CMND_FWD_REV : RUN_CMND_OPER_DIRN);
        runCmndFwdRev = fwdRev;
    }


    /**
     *  Gets whether run command is forward/reverse style
     * 
     *  @return Whether forward/reverse style 
     *  @throws DriverException 
     */
    public boolean isRunCmndFwdRev() throws DriverException
    {
        return readRegister(REG_RUN_CMND_TYPE) == RUN_CMND_FWD_REV;
    }


    /**
     *  Sets the voltage units
     * 
     *  @param units The voltage units enum
     *  @throws DriverException 
     */
    public void setVoltageUnits(VoltageUnits units) throws DriverException
    {
        writeRegister(REG_VOLTAGE_UNITS, voltageUnitsEnumMap.get(units));
        voltageUnits = units;
    }


    /**
     *  Gets the voltage units
     * 
     *  @return The voltage units enum
     *  @throws DriverException 
     */
    public VoltageUnits getVoltageUnits() throws DriverException
    {
        return voltageUnitsValueMap.get((short)readRegister(REG_VOLTAGE_UNITS));
    }


    /**
     *  Sets the input voltage
     * 
     *  @param voltage The input voltage
     *  @throws DriverException 
     */
    public void setInputVoltage(double voltage) throws DriverException
    {
        writeRegister(REG_INPUT_VOLTAGE, (short)voltage);
    }


    /**
     *  Gets the input voltage
     * 
     *  @return The input voltage
     *  @throws DriverException 
     */
    public double getInputVoltage() throws DriverException
    {
        return readRegister(REG_INPUT_VOLTAGE);
    }


    /**
     *  Sets the V/f profile
     * 
     *  @param level The V/f profile enum
     *  @throws DriverException 
     */
    public void setVfSelection(VfSelection level) throws DriverException
    {
        writeRegister(REG_VF_SELECTION, vfSelectionEnumMap.get(level));
    }


    /**
     *  Gets the V/f profile
     * 
     *  @return The V/f profile enum
     *  @throws DriverException 
     */
    public VfSelection getVfSelection() throws DriverException
    {
        return vfSelectionValueMap.get((short)readRegister(REG_VF_SELECTION));
    }


    /**
     *  Sets the maximum allowed frequency
     * 
     *  @param freqPct The frequency percentage to set
     *  @throws DriverException 
     */
    public void setMaxFrequency(double freqPct) throws DriverException
    {
        writeRegister(REG_FREQ_REF_MAX, (short)(freqPct / FREQ_LIMIT_UNITS));
    }


    /**
     *  Gets the maximum allowed frequency
     * 
     *  @return The maximum frequency percentage
     *  @throws DriverException 
     */
    public double getMaxFrequency() throws DriverException
    {
        return FREQ_LIMIT_UNITS * readRegister(REG_FREQ_REF_MAX);
    }


    /**
     *  Sets the minimum allowed frequency
     * 
     *  @param freqPct The frequency percentage to set
     *  @throws DriverException 
     */
    public void setMinFrequency(double freqPct) throws DriverException
    {
        writeRegister(REG_FREQ_REF_MIN, (short)(freqPct / FREQ_LIMIT_UNITS));
    }


    /**
     *  Gets the minimum allowed frequency
     * 
     *  @return The minimum frequency percentage
     *  @throws DriverException 
     */
    public double getMinFrequency() throws DriverException
    {
        return FREQ_LIMIT_UNITS * readRegister(REG_FREQ_REF_MIN);
    }


    /**
     *  Sets the frequency
     * 
     *  @param freq The frequency to set
     *  @throws DriverException 
     */
    public void setFrequency(double freq) throws DriverException
    {
        writeRegister(REG_FREQ_REFC, (short)(freq / FREQ_UNITS));
    }


    /**
     *  Gets the set frequency
     * 
     *  @return The set drive frequency
     *  @throws DriverException 
     */
    public double getFrequency() throws DriverException
    {
        return FREQ_UNITS * readRegister(REG_FREQ_REFC);
    }


    /**
     *  Gets the applied frequency
     * 
     *  @return The applied drive frequency
     *  @throws DriverException 
     */
    public double getFrequencyApp() throws DriverException
    {
        return FREQ_UNITS * readRegister(REG_FREQ_REFC_IN);
    }


    /**
     *  Reads the frequency
     * 
     *  @return The drive frequency
     *  @throws DriverException 
     */
    public double readFrequency() throws DriverException
    {
        return FREQ_UNITS * readRegister(REG_OUTPUT_FREQ);
    }


    /**
     *  Reads the voltage
     * 
     *  @return The drive voltage
     *  @throws DriverException 
     */
    public double readVoltage() throws DriverException
    {
        return VOLT_UNITS * readRegister(REG_VOLTAGE_REFC);
    }


    /**
     *  Reads the current
     * 
     *  @return The drive current
     *  @throws DriverException 
     */
    public double readCurrent() throws DriverException
    {
        return ratedCurrent / 8192 * readRegister(REG_OUTPUT_CURR);
    }


    /**
     *  Reads the bus voltage
     * 
     *  @return The bus voltage
     *  @throws DriverException 
     */
    public double readBusVoltage() throws DriverException
    {
        return readRegister(REG_BUS_VOLTAGE);
    }


    /**
     *  Reads the heatsink temperature
     * 
     *  @return The heatsink temperature (C)
     *  @throws DriverException 
     */
    public double readTemperature() throws DriverException
    {
        return readRegister(REG_HEATSINK_TEMP);
    }


    /**
     *  Gets the cumulative operation time
     * 
     *  @return The operation time (hours)
     *  @throws DriverException 
     */
    public int getOperationTime() throws DriverException
    {
        return readRegister(REG_OPERATION_TIME);
    }


    /**
     *  Sets the cumulative operation time
     * 
     *  @param time The time (hrs)
     *  @throws DriverException 
     */
    public void setOperationTime(int time) throws DriverException
    {
        writeRegister(REG_OPER_TIME_SET, (short)(time / 10));
    }


    /**
     *  Sets whether cumulative operation time measures only time active
     * 
     *  @param isActvTime Whether to use time active
     *  @throws DriverException 
     */
    public void setOpTimeActive(boolean isActvTime) throws DriverException
    {
        writeRegister(REG_OPER_TIME_TYPE, isActvTime ? ONE : ZERO);
    }


    /**
     *  Gets whether cumulative operation time measures only time active
     * 
     *  @return Whether time active is being used
     *  @throws DriverException 
     */
    public boolean isOpTimeActive() throws DriverException
    {
        return readRegister(REG_OPER_TIME_TYPE) != 0;
    }


    /**
     *  Gets the cumulative fan operation time
     * 
     *  @return The operation time (hours)
     *  @throws DriverException 
     */
    public int getFanOperationTime() throws DriverException
    {
        return readRegister(REG_FAN_OPRN_TIME);
    }


    /**
     *  Sets the cumulative fan operation time
     * 
     *  @param time The time (hrs)
     *  @throws DriverException 
     */
    public void setFanOperationTime(int time) throws DriverException
    {
        writeRegister(REG_FAN_OP_TIME_SET, (short)(time / 10));
    }


    /**
     *  Gets the run command count
     * 
     *  @return The command count
     *  @throws DriverException 
     */
    public int getRunCount() throws DriverException
    {
        return readRegister(REG_RUN_CMND_COUNT);
    }


    /**
     *  Gets the cumulative output kilowatt hours
     * 
     *  @return The number of KWH
     *  @throws DriverException 
     */
    public double getOutputKwh() throws DriverException
    {
        short[] values = readRegisters(REG_KWH_LOWER, TWO);
        return 0.1 * values[0] + 100.0 * values[1];
    }


    /**
     *  Gets the fan expected lifetime fraction used
     * 
     *  @return The used fraction
     *  @throws DriverException 
     */
    public double getFanLifetimeUsed() throws DriverException
    {
        return readRegister(REG_FAN_LIFE_PCT) / 100.0;
    }


    /**
     *  Gets the capacitor expected lifetime fraction used
     * 
     *  @return The used fraction
     *  @throws DriverException 
     */
    public double getCapLifetimeUsed() throws DriverException
    {
        return readRegister(REG_CAP_LIFE_PCT) / 100.0;
    }


    /**
     *  Sets the capacitor expected lifetime fraction used
     * 
     *  @param fract The used fraction
     *  @throws DriverException 
     */
    public void setCapLifetimeUsed(double fract) throws DriverException
    {
        writeRegister(REG_CAP_LT_PCT_SET, (short)(100.0 * fract));
    }


    /**
     *  Gets the soft charge bypass relay expected lifetime fraction used
     * 
     *  @return The used fraction
     *  @throws DriverException 
     */
    public double getScbrLifetimeUsed() throws DriverException
    {
        return readRegister(REG_SCBR_LIFE_PCT) / 100.0;
    }


    /**
     *  Sets the soft charge bypass relay expected lifetime fraction used
     * 
     *  @param fract The used fraction
     *  @throws DriverException 
     */
    public void setScbrLifetimeUsed(double fract) throws DriverException
    {
        writeRegister(REG_SCBR_LT_PCT_SET, (short)(100.0 * fract));
    }


    /**
     *  Gets the IGBT expected lifetime fraction used
     * 
     *  @return The used fraction
     *  @throws DriverException 
     */
    public double getIgbtLifetimeUsed() throws DriverException
    {
        return readRegister(REG_IGBT_LIFE_PCT) / 100.0;
    }


    /**
     *  Sets the IGBT expected lifetime fraction used
     * 
     *  @param fract The used fraction
     *  @throws DriverException 
     */
    public void setIgbtLifetimeUsed(double fract) throws DriverException
    {
        writeRegister(REG_IGBT_LT_PCT_SET, (short)(100.0 * fract));
    }


    /**
     *  Flashes the display LEDs
     * 
     *  @throws DriverException 
     */
    public void flashLeds() throws DriverException
    {
        readRegister(REG_LED_CHECK);
    }


    /**
     *  Gets the drive status
     * 
     *  @return The drive status word
     *  @throws DriverException 
     */
    public int getDriveStatus() throws DriverException
    {
        return readRegister(REG_DRIVE_STATUS) & 0xff;
    }


    /**
     *  Runs the motor forwards
     * 
     *  @throws DriverException 
     */
    public void runForward() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, CMND_FORWARD);
    }


    /**
     *  Runs the motor in reverse
     * 
     *  @throws DriverException 
     */
    public void runReverse() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, (short)(runCmndFwdRev ? CMND_FR_REVERSE : CMND_OD_REVERSE));
    }


    /**
     *  Stops the motor
     * 
     *  @throws DriverException 
     */
    public void stop() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, CMND_STOP);
    }


    /**
     *  Gets the alarm code
     * 
     *  @return The alarm code, or 0 if no alarm
     *  @throws DriverException 
     */
    public int getAlarmCode() throws DriverException
    {
        return readRegister(REG_ALARM_CODE);
    }


    /**
     *  Gets all alarm codes
     * 
     *  @return The array of active alarm codes
     *  @throws DriverException 
     */
    public Set<Integer> getAlarmCodes() throws DriverException
    {
        Set<Integer> alarmSet = new HashSet<>();
        short[] alarmBits = readRegisters(REG_ALARM_2, NUM_ALARM_REGS);
        for (int j = 0; j < NUM_ALARM_REGS; j++) {
            short bits = alarmBits[j];
            for (int k = 0; bits != 0 && k < 16; bits >>>= 1, k++) {
                if ((bits & 1) != 0) {
                    alarmSet.add(16 * j + k + 1);
                }
            }
        }
        return alarmSet;
    }


    /**
     *  Gets the alarm description
     * 
     *  @return The alarm description, or null if no alarm active
     *  @throws DriverException 
     */
    public String getAlarmDesc() throws DriverException
    {
        int code = getAlarmCode();
        return code == ALARM_NONE ? null : getAlarmDesc(code);
    }


    /**
     *  Gets the alarm description from its code
     * 
     *  @param  code  The alarm code
     *  @return The alarm description, or null if no alarm active
     */
    public String getAlarmDesc(int code)
    {
        String text = alarmDesc.get(code);
        if (text == null) {
            text = String.format("Undefined alarm code: 0x%04x", code);
        }
        return text;
    }

    /**
     * Get all possible alarm codes for this driver.
     * 
     * @return The map of integer code and description of all available alarm codes.
     */
    public Map<Integer, String> getAllAvailableAlarmCodes() {
        return alarmDesc;
    }

    /**
     *  Gets the current fault code
     * 
     *  @return The fault code
     *  @throws DriverException 
     */
    public int getCurrFaultCode() throws DriverException
    {
        return readRegister(REG_CURR_FAULT);
    }


    /**
     *  Gets the previous fault code
     * 
     *  @return The fault code
     *  @throws DriverException 
     */
    public int getPrevFaultCode() throws DriverException
    {
        return readRegister(REG_PREV_FAULT);
    }


    /**
     *  Gets the current fault description
     * 
     *  @return The fault description, or null if no current fault
     *  @throws DriverException 
     */
    public String getCurrFaultDesc() throws DriverException
    {
        int code = getCurrFaultCode();
        return code == FAULT_NONE ? null : getFaultDesc(code);
    }


    /**
     *  Gets the previous fault description
     * 
     *  @return The fault description, or null if no previous fault
     *  @throws DriverException 
     */
    public String getPrevFaultDesc() throws DriverException
    {
        int code = getPrevFaultCode();
        return code == FAULT_NONE ? null : getFaultDesc(code);
    }


    /**
     *  Gets the fault description from its code
     * 
     *  @param  code  The fault code
     *  @return The fault description, or null if no fault active
     */
    public String getFaultDesc(int code)
    {
        String text = faultDesc.get(code);
        if (text == null) {
            text = String.format("Undefined fault code: 0x%04x", code);
        }
        return text;
    }


    /**
     *  Resets a fault
     * 
     *  @throws DriverException 
     */
    public void resetFault() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, CMND_RESET_FAULT);
    }

}
