package org.lsst.ccs.drivers.yaskawa;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Yaskawa A1000 series motor drive..
 *
 *  @author  Owen Saxton
 */
public class A1000 {

    /**
     *  Public constants.
     */
    public enum Language { ENGLISH }
    public enum AccessLevel { OPR_ONLY, USR_PARM, FULL }
    public enum ControlMode { VF, VF_PG, OLV, CLV, CLV_PM, OLV_PM, AOLV_PM }
    public enum FreqRefSrc { OPERATOR, ANALOG, MODBUS, OPTN_PCB, PULSE }
    public enum RunCmndSrc { OPERATOR, DIGITAL, MODBUS, OPTN_PCB }
    public enum StopMethod { COAST, COAST_TMR, INJECT, RAMP }
    public enum RunProg { PROG_ONLY, RUN_PROG, RUN_ONLY }
    public enum BaudRate { B1200, B2400, B4800, B9600, B19200, B38400, B57600, B76800, B115200 }
    public enum Parity { EVEN, ODD, NONE }
    public enum ErrorAction { RAMP_STOP, COAST_STOP, FAST_STOP, ALARM_ONLY }
    public enum VoltageUnits { ONES, TENTHS }
    public enum VfSelection { HZ50, HZ60_SAT, HZ50_SAT, HZ72, HZ50_VT1, HZ50_VT2, HZ60_VT1, HZ60_VT2,
                              HZ50_HST1, HZ50_HST2, HZ60_HST1, HZ60_HST2, HZ90, HZ120, HZ180, CUSTOM }


    /**
     *  Private constants.
     */
    private static final short
        REG_COMMAND_MF_IN  = 0x001,
        REG_FREQ_REFC      = 0x002,
        REG_VOLTAGE_GAIN   = 0x003,
        REG_TORQUE_REFC    = 0x004,
        REG_TORQUE_COMP    = 0x005,
        REG_PID_TARGET     = 0x006,
        REG_ANAL_OUT_FM    = 0x007,
        REG_ANAL_OUT_AM    = 0x008,
        REG_DIGITAL_OUT    = 0x009,
        REG_PULSE_OUT      = 0x00a,
        REG_CTRL_SLCTN     = 0x00f,

        REG_ANAL_MONITOR_1 = 0x01b,
        REG_ANAL_MONITOR_2 = 0x01c,
        REG_DIG_OUT_OPTN   = 0x01d,
        REG_DRIVE_STATUS_1 = 0x020,
        REG_FAULT_1        = 0x021,
        REG_LINK_STATUS    = 0x022,
        REG_FREQ_REF_IN    = 0x023,
        REG_OUTPUT_FRQ     = 0x024,
        REG_VOLTAGE_REF    = 0x025,
        REG_OUTPUT_CUR     = 0x026,
        REG_OUTPUT_PWR     = 0x027,
        REG_TORQUE_REF_IN  = 0x028,
        REG_FAULT_2        = 0x029,
        REG_ALARM_1        = 0x02a,
        REG_INPUT_TRM      = 0x02b,
        REG_DRIVE_STATUS_2 = 0x02c,
        REG_OUTPUT_TRM     = 0x02d,
        REG_FREQ_REF_BIAS  = 0x02f,
        REG_DC_BUS_VOLTAGE = 0x031,
        REG_TORQUE_REF_IN2 = 0x032,
        REG_PRODUCT_CODE_1 = 0x034,
        REG_PRODUCT_CODE_2 = 0x035,
        REG_COMM_ERROR     = 0x03d,
        REG_OUT_FREQ_RPM   = 0x03e,
        REG_OUT_FREQ_PCT   = 0x03f,

        REG_FREQ_REFC_IN   = 0x040,  //U1-01
        REG_OUTPUT_FREQ    = 0x041,  //U1-02
        REG_OUTPUT_CURR    = 0x042,  //U1-03
        REG_CTRL_METHOD_IN = 0x043,  //U1-04
        REG_MOTOR_SPEED    = 0x044,  //U1-05
        REG_VOLTAGE_REFC   = 0x045,  //U1-06
        REG_BUS_VOLTAGE    = 0x046,  //U1-07
        REG_OUTPUT_POWER   = 0x047,  //U1-08
        REG_INPUT_TERM     = 0x049,  //U1-10
        REG_OUTPUT_TERM    = 0x04a,  //U1-11
        REG_DRIVE_STATUS   = 0x04b,  //U1-12

        REG_ALARM_CODE     = 0x07f,

        REG_ACC_OP_TIME_HI = 0x098,
        REG_ACC_OP_TIME_LO = 0x099,
        REG_FAN_OP_TIME_HI = 0x09a,
        REG_FAN_OP_TIME_LO = 0x09b,
        REG_RATED_CURRENT  = 0x0ab,
        REG_MOTOR_SPD_RPM  = 0x0ac,
        REG_MOTOR_SPD_PCT  = 0x0ad,
        REG_FREQ_REF_S_RPM = 0x0b5,
        REG_FREQ_REF_S_PCT = 0x0b6,
        REG_FREQ_REF_RPM   = 0x0b7,
        REG_FREQ_REF_PCT   = 0x0b8,
        REG_OP_ERROR_CODE  = 0x0bf,
        REG_FAULT_3        = 0x0c0,
        REG_FAULT_4        = 0x0c1,
        REG_FAULT_5        = 0x0c2,
        REG_FAULT_6        = 0x0c3,
        REG_FAULT_7        = 0x0c4,
        REG_FAULT_8        = 0x0c5,
        REG_FAULT_9        = 0x0c6,
        REG_ALARM_2        = 0x0c8,
        REG_ALARM_3        = 0x0c9,
        REG_ALARM_4        = 0x0ca,
        REG_ALARM_5        = 0x0cb,
        REG_ALARM_6        = 0x0cc,
        REG_ALARM_7        = 0x0cd,
        REG_CPF_CONTENTS_1 = 0x0d0,
        REG_CPF_CONTENTS_2 = 0x0d1,
        REG_CPF_CONTENTS_3 = 0x0d2,

        REG_LANGUAGE       = 0x100,  //A1-00
        LANG_ENGLISH       = 0,  // Default
        LANG_JAPANESE      = 1,
        LANG_GERMAN        = 2,
        LANG_FRENCH        = 3,
        LANG_ITALIAN       = 4,
        LANG_SPANISH       = 5,
        LANG_PORTUGUESE    = 6,
        LANG_CHINESE       = 7,
        LANG_CZECH         = 8,
        LANG_RUSSIAN       = 9,
        LANG_TURKISH       = 10,
        LANG_POLISH        = 11,
        LANG_GREEK         = 12,
        REG_ACCESS_LEVEL   = 0x101,  //A1-01
        ACC_LEVEL_OPR_ONLY = 0,
        ACC_LEVEL_USR_PARM = 1,
        ACC_LEVEL_FULL     = 2,  // Default
        REG_CTRL_MODE      = 0x102,  //A1-02
        CTRL_MODE_VF       = 0,
        CTRL_MODE_VF_PG    = 1,
        CTRL_MODE_OLV      = 2,  // Default
        CTRL_MODE_CLV      = 3,
        CTRL_MODE_OLV_PM   = 5,
        CTRL_MODE_AOLV_PM  = 6,
        CTRL_MODE_CLV_PM   = 7,
        REG_INIT_PARAMS    = 0x103,  //A1-03
        INIT_NONE          = 0,  // Default
        INIT_USER          = 0x1110,
        INIT_2WIRE         = 0x2220,
        INIT_3WIRE         = 0x3330,
        INIT_OPE04_RESET   = 0x5550,
        REG_PW_ENTRY       = 0x104,  //A1-04
        REG_PW_SETTING     = 0x105,  //A1-05

        REG_FREQ_REF_SRC   = 0x180,  //b1-01
        FREQ_REF_OPERATOR  = 0,
        FREQ_REF_ANALOG    = 1,  // Default
        FREQ_REF_MODBUS    = 2,
        FREQ_REF_OPTN_PCB  = 3,
        FREQ_REF_PULSE     = 4,
        REG_RUN_CMND_SRC   = 0x181,  //b1-02
        RUN_CMND_OPERATOR  = 0,
        RUN_CMND_DIGITAL   = 1,  // Default
        RUN_CMND_MODBUS    = 2,
        RUN_CMND_OPTN_PCB  = 3,
        REG_STOP_METHOD    = 0x182,  //b1-03
        STOP_METH_RAMP     = 0,  // Default
        STOP_METH_COAST    = 1,
        STOP_METH_INJECT   = 2,
        STOP_METH_COAST_TM = 3,
        REG_REV_PERMIT     = 0x183,  //b1-04
        REV_ENABLED        = 0,  // Default
        REV_DISABLED       = 1,
        REG_DIG_INPUT_CONF = 0x185,  //b1-06
        DIG_CONF_DISABLED  = 0,
        DIG_CONF_ENABLED   = 1,  // Default
        REG_RUN_CMND_CYCLE = 0x186,  //b1-07
        RUN_CYCLE_ENABLED  = 0,  // Default
        RUN_CYCLE_DISABLED = 1,
        REG_RUN_CMND_PROG  = 0x187,  //b1-08
        PROG_MODE_ONLY     = 0,  // Default
        RUN_PROG_OKAY      = 1,
        RUN_MODE_ONLY      = 2,

        REG_INPUT_VOLTAGE  = 0x300,  //E1-01 (0.1 V)
        REG_VF_SELECTION   = 0x302,  //E1-03
        VF_50HZ            = 0,
        VF_60HZ_SAT        = 1,
        VF_50HZ_SAT        = 2,
        VF_72HZ            = 3,
        VF_50HZ_VT1        = 4,
        VF_50HZ_VT2        = 5,
        VF_60HZ_VT1        = 6,
        VF_60HZ_VT2        = 7,
        VF_50HZ_HST1       = 8,
        VF_50HZ_HST2       = 9,
        VF_60HZ_HST1       = 10,
        VF_60HZ_HST2       = 11,
        VF_90HZ            = 12,
        VF_120HZ           = 13,
        VF_180HZ           = 14,
        VF_CUSTOM          = 15,  // Default

        REG_NODE_ADDRESS   = 0x425,  //H5-01
        REG_BAUD_RATE      = 0x426,  //H5-02
        BAUD_RATE_1200     = 0,
        BAUD_RATE_2400     = 1,
        BAUD_RATE_4800     = 2,
        BAUD_RATE_9600     = 3,  // Default
        BAUD_RATE_19200    = 4,
        BAUD_RATE_38400    = 5,
        BAUD_RATE_57600    = 6,
        BAUD_RATE_76800    = 7,
        BAUD_RATE_115200   = 8,
        REG_COMM_PARITY    = 0x427,  //H5-03
        PARITY_NONE        = 0,  // Default
        PARITY_EVEN        = 1,
        PARITY_ODD         = 2,
        REG_ERROR_ACTION   = 0x428,  //H5-04
        ERROR_RAMP_STOP    = 0,  // Default
        ERROR_COAST_STOP   = 1,
        ERROR_FAST_STOP    = 2,
        ERROR_ALARM_ONLY   = 3,
        REG_COMM_FAULT_DET = 0x429,  //H5-05
        FAULT_DETC_OFF     = 0,
        FAULT_DETC_ON      = 1,  // Default
        REG_XMIT_WAIT_TIME = 0x42a,  //H5-06 (msec)
        REG_RTS_CONTROL    = 0x42b,  //H5-07
        RTS_ALWAYS_ON      = 0,
        RTS_ONLY_ON_SEND   = 1,  // Default
        REG_ERROR_DET_TIME = 0x435,  //H5-09 (0.1 sec)
        REG_VOLTAGE_UNITS  = 0x436,  //H5-10
        VOLT_UNITS_TENTHS  = 0,  // Default
        VOLT_UNITS_ONES    = 1,
        REG_ENTER_FUNC     = 0x43c,  //H5-11
        ENTER_REQUIRED     = 0,  // Default
        ENTER_NOT_REQUIRED = 1,
        REG_RUN_CMND_TYPE  = 0x43d,  //H5-12
        RUN_CMND_FWD_REV   = 0,  // Default
        RUN_CMND_OPER_DIRN = 1,

        REG_DRIVE_MODEL    = 0x508,  //o2-04
        MODEL_4A0002       = 0x92,
        MODEL_4A0004       = 0x93,
        MODEL_4A0005       = 0x94,
        MODEL_4A0007       = 0x95,
        MODEL_4A0009       = 0x96,

        REG_ENTER_EEPROM   = 0x900,
        REG_ENTER_RAM      = 0x910,
        REG_ = 0x0,
        z = 0;

    private static final Map<Language, Short> languageEnumMap = new HashMap<>();
    private static final Map<Short, Language> languageValueMap = new HashMap<>();
    static {
        languageEnumMap.put(Language.ENGLISH, LANG_ENGLISH);
        for (Map.Entry e : languageEnumMap.entrySet()) {
            languageValueMap.put((Short)e.getValue(), (Language)e.getKey());
        }
    }
    private static final Map<AccessLevel, Short> accessLevelEnumMap = new HashMap<>();
    private static final Map<Short, AccessLevel> accessLevelValueMap = new HashMap<>();
    static {
        accessLevelEnumMap.put(AccessLevel.OPR_ONLY, ACC_LEVEL_OPR_ONLY);
        accessLevelEnumMap.put(AccessLevel.USR_PARM, ACC_LEVEL_USR_PARM);
        accessLevelEnumMap.put(AccessLevel.FULL, ACC_LEVEL_FULL);
        for (Map.Entry e : accessLevelEnumMap.entrySet()) {
            accessLevelValueMap.put((Short)e.getValue(), (AccessLevel)e.getKey());
        }
    }
    private static final Map<ControlMode, Short> controlModeEnumMap = new HashMap<>();
    private static final Map<Short, ControlMode> controlModeValueMap = new HashMap<>();
    static {
        controlModeEnumMap.put(ControlMode.VF, CTRL_MODE_VF);
        controlModeEnumMap.put(ControlMode.VF_PG, CTRL_MODE_VF_PG);
        controlModeEnumMap.put(ControlMode.OLV, CTRL_MODE_OLV);
        controlModeEnumMap.put(ControlMode.CLV, CTRL_MODE_CLV);
        controlModeEnumMap.put(ControlMode.CLV_PM, CTRL_MODE_CLV_PM);
        controlModeEnumMap.put(ControlMode.OLV_PM, CTRL_MODE_OLV_PM);
        controlModeEnumMap.put(ControlMode.AOLV_PM, CTRL_MODE_AOLV_PM);
        for (Map.Entry e : controlModeEnumMap.entrySet()) {
            controlModeValueMap.put((Short)e.getValue(), (ControlMode)e.getKey());
        }
    }
    private static final Map<FreqRefSrc, Short> freqRefSrcEnumMap = new HashMap<>();
    private static final Map<Short, FreqRefSrc> freqRefSrcValueMap = new HashMap<>();
    static {
        freqRefSrcEnumMap.put(FreqRefSrc.OPERATOR, FREQ_REF_OPERATOR);
        freqRefSrcEnumMap.put(FreqRefSrc.ANALOG, FREQ_REF_ANALOG);
        freqRefSrcEnumMap.put(FreqRefSrc.MODBUS, FREQ_REF_MODBUS);
        freqRefSrcEnumMap.put(FreqRefSrc.OPTN_PCB, FREQ_REF_OPTN_PCB);
        freqRefSrcEnumMap.put(FreqRefSrc.PULSE, FREQ_REF_PULSE);
        for (Map.Entry e : freqRefSrcEnumMap.entrySet()) {
            freqRefSrcValueMap.put((Short)e.getValue(), (FreqRefSrc)e.getKey());
        }
    }
    private static final Map<RunCmndSrc, Short> runCmndSrcEnumMap = new HashMap<>();
    private static final Map<Short, RunCmndSrc> runCmndSrcValueMap = new HashMap<>();
    static {
        runCmndSrcEnumMap.put(RunCmndSrc.OPERATOR, RUN_CMND_OPERATOR);
        runCmndSrcEnumMap.put(RunCmndSrc.DIGITAL, RUN_CMND_DIGITAL);
        runCmndSrcEnumMap.put(RunCmndSrc.MODBUS, RUN_CMND_MODBUS);
        runCmndSrcEnumMap.put(RunCmndSrc.OPTN_PCB, RUN_CMND_OPTN_PCB);
        for (Map.Entry e : runCmndSrcEnumMap.entrySet()) {
            runCmndSrcValueMap.put((Short)e.getValue(), (RunCmndSrc)e.getKey());
        }
    }
    private static final Map<StopMethod, Short> stopMethodEnumMap = new HashMap<>();
    private static final Map<Short, StopMethod> stopMethodValueMap = new HashMap<>();
    static {
        stopMethodEnumMap.put(StopMethod.COAST, STOP_METH_COAST);
        stopMethodEnumMap.put(StopMethod.COAST_TMR, STOP_METH_COAST_TM);
        stopMethodEnumMap.put(StopMethod.INJECT, STOP_METH_INJECT);
        stopMethodEnumMap.put(StopMethod.RAMP, STOP_METH_RAMP);
        for (Map.Entry e : stopMethodEnumMap.entrySet()) {
            stopMethodValueMap.put((Short)e.getValue(), (StopMethod)e.getKey());
        }
    }
    private static final Map<RunProg, Short> runProgEnumMap = new HashMap<>();
    private static final Map<Short, RunProg> runProgValueMap = new HashMap<>();
    static {
        runProgEnumMap.put(RunProg.PROG_ONLY, PROG_MODE_ONLY);
        runProgEnumMap.put(RunProg.RUN_PROG, RUN_PROG_OKAY);
        runProgEnumMap.put(RunProg.RUN_ONLY, RUN_MODE_ONLY);
        for (Map.Entry e : runProgEnumMap.entrySet()) {
            runProgValueMap.put((Short)e.getValue(), (RunProg)e.getKey());
        }
    }
    private static final Map<BaudRate, Short> baudRateEnumMap = new HashMap<>();
    private static final Map<Short, BaudRate> baudRateValueMap = new HashMap<>();
    static {
        baudRateEnumMap.put(BaudRate.B1200, BAUD_RATE_1200);
        baudRateEnumMap.put(BaudRate.B2400, BAUD_RATE_2400);
        baudRateEnumMap.put(BaudRate.B4800, BAUD_RATE_4800);
        baudRateEnumMap.put(BaudRate.B9600, BAUD_RATE_9600);
        baudRateEnumMap.put(BaudRate.B19200, BAUD_RATE_19200);
        baudRateEnumMap.put(BaudRate.B38400, BAUD_RATE_38400);
        baudRateEnumMap.put(BaudRate.B57600, BAUD_RATE_57600);
        baudRateEnumMap.put(BaudRate.B76800, BAUD_RATE_76800);
        baudRateEnumMap.put(BaudRate.B115200, BAUD_RATE_115200);
        for (Map.Entry e : baudRateEnumMap.entrySet()) {
            baudRateValueMap.put((Short)e.getValue(), (BaudRate)e.getKey());
        }
    }
    private static final Map<Parity, Short> parityEnumMap = new HashMap<>();
    private static final Map<Short, Parity> parityValueMap = new HashMap<>();
    static {
        parityEnumMap.put(Parity.EVEN, PARITY_EVEN);
        parityEnumMap.put(Parity.ODD, PARITY_ODD);
        parityEnumMap.put(Parity.NONE, PARITY_NONE);
        for (Map.Entry e : parityEnumMap.entrySet()) {
            parityValueMap.put((Short)e.getValue(), (Parity)e.getKey());
        }
    }
    private static final Map<ErrorAction, Short> errorActionEnumMap = new HashMap<>();
    private static final Map<Short, ErrorAction> errorActionValueMap = new HashMap<>();
    static {
        errorActionEnumMap.put(ErrorAction.RAMP_STOP, ERROR_RAMP_STOP);
        errorActionEnumMap.put(ErrorAction.COAST_STOP, ERROR_COAST_STOP);
        errorActionEnumMap.put(ErrorAction.FAST_STOP, ERROR_FAST_STOP);
        errorActionEnumMap.put(ErrorAction.ALARM_ONLY, ERROR_ALARM_ONLY);
        for (Map.Entry e : errorActionEnumMap.entrySet()) {
            errorActionValueMap.put((Short)e.getValue(), (ErrorAction)e.getKey());
        }
    }
    private static final Map<VoltageUnits, Short> voltageUnitsEnumMap = new HashMap<>();
    private static final Map<Short, VoltageUnits> voltageUnitsValueMap = new HashMap<>();
    static {
        voltageUnitsEnumMap.put(VoltageUnits.ONES, VOLT_UNITS_ONES);
        voltageUnitsEnumMap.put(VoltageUnits.TENTHS, VOLT_UNITS_TENTHS);
        for (Map.Entry e : voltageUnitsEnumMap.entrySet()) {
            voltageUnitsValueMap.put((Short)e.getValue(), (VoltageUnits)e.getKey());
        }
    }
    private static final Map<VfSelection, Short> vfSelectionEnumMap = new HashMap<>();
    private static final Map<Short, VfSelection> vfSelectionValueMap = new HashMap<>();
    static {
        vfSelectionEnumMap.put(VfSelection.HZ50, VF_50HZ);
        vfSelectionEnumMap.put(VfSelection.HZ60_SAT, VF_60HZ_SAT);
        vfSelectionEnumMap.put(VfSelection.HZ50_SAT, VF_50HZ_SAT);
        vfSelectionEnumMap.put(VfSelection.HZ72, VF_72HZ);
        vfSelectionEnumMap.put(VfSelection.HZ50_VT1, VF_50HZ_VT1);
        vfSelectionEnumMap.put(VfSelection.HZ50_VT2, VF_50HZ_VT2);
        vfSelectionEnumMap.put(VfSelection.HZ60_VT1, VF_60HZ_VT1);
        vfSelectionEnumMap.put(VfSelection.HZ60_VT2, VF_60HZ_VT2);
        vfSelectionEnumMap.put(VfSelection.HZ50_HST1, VF_50HZ_HST1);
        vfSelectionEnumMap.put(VfSelection.HZ50_HST2, VF_50HZ_HST2);
        vfSelectionEnumMap.put(VfSelection.HZ60_HST1, VF_60HZ_HST1);
        vfSelectionEnumMap.put(VfSelection.HZ60_HST2, VF_60HZ_HST2);
        vfSelectionEnumMap.put(VfSelection.HZ90, VF_90HZ);
        vfSelectionEnumMap.put(VfSelection.HZ120, VF_120HZ);
        vfSelectionEnumMap.put(VfSelection.HZ180, VF_180HZ);
        vfSelectionEnumMap.put(VfSelection.CUSTOM, VF_CUSTOM);
        for (Map.Entry e : vfSelectionEnumMap.entrySet()) {
            vfSelectionValueMap.put((Short)e.getValue(), (VfSelection)e.getKey());
        }
    }
    private static final Map<Short, String> modelNameMap = new HashMap<>();
    static {
        modelNameMap.put(MODEL_4A0002, "4A0002");
        modelNameMap.put(MODEL_4A0004, "4A0004");
        modelNameMap.put(MODEL_4A0005, "4A0005");
        modelNameMap.put(MODEL_4A0007, "4A0007");
        modelNameMap.put(MODEL_4A0009, "4A0009");
    }
    private static final Map<Short, Double> ratedCurrentMap = new HashMap<>();
    static {
        ratedCurrentMap.put(MODEL_4A0002, 2.1);
        ratedCurrentMap.put(MODEL_4A0004, 4.1);
        ratedCurrentMap.put(MODEL_4A0005, 5.4);
        ratedCurrentMap.put(MODEL_4A0007, 6.9);
        ratedCurrentMap.put(MODEL_4A0009, 8.8);
    }

    /**
     *  Private data.
     */
    private static final int
        BAUD_RATE = 115200,
        DEVC_ADDR = 0x1f;
    private static final short
        ZERO = 0,
        ONE  = 1,
        TWO  = 2;
    private static final int
        EXCP_BIT_COUNT = 3,
        EXCP_SET_DATA = 33,
        EXCP_WRITE_MODE = 34,
        EXCP_BUS_UNDERVOLT = 35,
        EXCP_WRITE_PARAM = 36,
        EXCP_WRITE_EEPROM = 37;

    private static final Map<Integer, String> excpMap = new HashMap<>();
    static {
        excpMap.put(EXCP_BIT_COUNT, "bit count error");
        excpMap.put(EXCP_SET_DATA, "illegal data value");
        excpMap.put(EXCP_WRITE_MODE, "illegal write operation");
        excpMap.put(EXCP_BUS_UNDERVOLT, "illegal write while in undervoltage");
        excpMap.put(EXCP_WRITE_PARAM, "illegal write while processing parameters");
        excpMap.put(EXCP_WRITE_EEPROM, "illegal write to EEPROM");
    }
    private static final double
        VOLT_UNITS = 0.1,
        FREQ_UNITS = 0.01;

    private final Modbus mbus;
    private short devcAddr;
    private double ratedCurrent;
    private VoltageUnits voltageUnits = VoltageUnits.TENTHS;
    private boolean runCmndFwdRev = true;


    /**
     *  Constructor.
     */
    public A1000()
    {
        mbus = new Modbus();
        mbus.setOptions(Modbus.Option.NO_NET);
        mbus.setDefaultParm(BAUD_RATE);
        mbus.setAddressMode(true);
        mbus.setExceptionMap(excpMap);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @param  address   The slave address
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The data characteristics
     *  @throws  DriverException
     */
    public void open(String device, int address, int baudRate, int dataChar) throws DriverException
    {
        mbus.open(Modbus.ConnType.SERIAL, device, baudRate, dataChar);
        devcAddr = address != 0 ? (short)address : DEVC_ADDR;
        try {
            ratedCurrent = getRatedCurrent();
        }
        catch (DriverException e) {
            mbus.closeSilent();
            throw e;
        }
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @param  address   The slave address
     *  @param  baudRate  The baud rate
     *  @throws  DriverException
     */
    public void open(String device, int address, int baudRate) throws DriverException
    {
        open(device, address, baudRate, 0);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @param  address   The slave address
     *  @throws  DriverException
     */
    public void open(String device, int address) throws DriverException
    {
        open(device, address, BAUD_RATE);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  device    The serial device name
     *  @throws  DriverException
     */
    public void open(String device) throws DriverException
    {
        open(device, DEVC_ADDR);
    }


    /**
     *  Closes the connection to the device.
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException
    {
        mbus.close();
    }


    /**
     *  Reads a set of registers.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read
     *  @return  An array of register values
     *  @throws DriverException
     */
    public short[] readRegisters(short addr, short count) throws DriverException
    {
        return mbus.readRegisters(devcAddr, addr, count);
    }


    /**
     *  Reads a register.
     *
     *  @param  addr   The register address
     *  @return  The read value as a signed integer
     *  @throws DriverException
     */
    public int readRegister(short addr) throws DriverException
    {
        return mbus.readRegisters(devcAddr, addr, ONE)[0];
    }


    /**
     *  Writes a register.
     *
     *  @param  addr   The register address
     *  @param  value  The value to write
     *  @throws DriverException
     */
    public void writeRegister(short addr, short value) throws DriverException
    {
        mbus.writeRegisters(devcAddr, addr, new short[]{value});
    }


    /**
     *  Writes a set of registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write
     *  @throws DriverException
     */
    public void writeRegisters(short addr, short[] value) throws DriverException
    {
        mbus.writeRegisters(devcAddr, addr, value);
    }


    /**
     *  Makes new parameter values available.
     *
     *  @throws  DriverException
     */
    public void enterParameters() throws DriverException
    {
        writeRegister(REG_ENTER_RAM, ZERO);
    }


    /**
     *  Saves parameter values in EEPROM.
     *
     *  @throws  DriverException
     */
    public void saveParameters() throws DriverException
    {
        writeRegister(REG_ENTER_EEPROM, ZERO);
    }


    /**
     *  Sets the node address.
     *
     *  @param  address  The device address (1 - 255)
     *  @throws  DriverException
     */
    public void setNodeAddress(int address) throws DriverException
    {
        writeRegister(REG_NODE_ADDRESS, (short)address);
    }


    /**
     *  Gets the node address.
     *
     *  @return  The device address (1 - 255)
     *  @throws  DriverException
     */
    public int getNodeAddress() throws DriverException
    {
        return readRegister(REG_NODE_ADDRESS);
    }


    /**
     *  Sets the drive baud rate.
     * 
     *  @param rate
     *  @throws DriverException 
     */
    public void setBaudRate(BaudRate rate) throws DriverException
    {
        writeRegister(REG_BAUD_RATE, baudRateEnumMap.get(rate));
    }


    /**
     *  Gets the drive baud rate
     * 
     *  @return  The baud rate
     *  @throws DriverException 
     */
    public BaudRate getBaudRate() throws DriverException
    {
        return baudRateValueMap.get((short)readRegister(REG_BAUD_RATE));
    }


    /**
     *  Sets the drive parity
     * 
     *  @param parity
     *  @throws DriverException 
     */
    public void setParity(Parity parity) throws DriverException
    {
        writeRegister(REG_COMM_PARITY, parityEnumMap.get(parity));
    }


    /**
     *  Gets the drive parity.
     * 
     *  @return
     *  @throws DriverException 
     */
    public Parity getParity() throws DriverException
    {
        return parityValueMap.get((short)readRegister(REG_COMM_PARITY));
    }


    /**
     *  Gets the model name
     * 
     *  @return
     *  @throws DriverException 
     */
    public String getModelName() throws DriverException
    {
        return modelNameMap.get((short)readRegister(REG_DRIVE_MODEL));
    }


    /**
     *  Gets the rated current
     * 
     *  @return
     *  @throws DriverException 
     */
    public double getRatedCurrent() throws DriverException
    {
        return ratedCurrentMap.get((short)readRegister(REG_DRIVE_MODEL));
    }


    /**
     *  Sets control to be local.
     * 
     *  @throws DriverException 
     */
    public void setLocalControl() throws DriverException
    {
        setRunCmndSrc(RunCmndSrc.DIGITAL);
        setFreqRefSrc(FreqRefSrc.ANALOG);
        enterParameters();
    }


    /**
     *  Sets control to be remote.
     * 
     *  @throws DriverException 
     */
    public void setRemoteControl() throws DriverException
    {
        setRunCmndSrc(RunCmndSrc.MODBUS);
        setFreqRefSrc(FreqRefSrc.MODBUS);
        enterParameters();
    }


    /**
     *  Sets the access level
     * 
     *  @param level
     *  @throws DriverException 
     */
    public void setAccessLevel(AccessLevel level) throws DriverException
    {
        writeRegister(REG_ACCESS_LEVEL, accessLevelEnumMap.get(level));
    }


    /**
     *  Gets the access level
     * 
     *  @return
     *  @throws DriverException 
     */
    public AccessLevel getAccessLevel() throws DriverException
    {
        return accessLevelValueMap.get((short)readRegister(REG_ACCESS_LEVEL));
    }


    /**
     *  Sets the control mode
     * 
     *  @param mode
     *  @throws DriverException 
     */
    public void setControlMode(ControlMode mode) throws DriverException
    {
        writeRegister(REG_CTRL_MODE, controlModeEnumMap.get(mode));
    }


    /**
     *  Gets the control mode
     * 
     *  @return
     *  @throws DriverException 
     */
    public ControlMode getControlMode() throws DriverException
    {
        return controlModeValueMap.get((short)readRegister(REG_CTRL_MODE));
    }


    /**
     *  Sets the frequency reference source
     * 
     *  @param src
     *  @throws DriverException 
     */
    public void setFreqRefSrc(FreqRefSrc src) throws DriverException
    {
        writeRegister(REG_FREQ_REF_SRC, freqRefSrcEnumMap.get(src));
    }


    /**
     *  Gets the frequency reference source
     * 
     *  @return
     *  @throws DriverException 
     */
    public FreqRefSrc getFreqRefSrc() throws DriverException
    {
        return freqRefSrcValueMap.get((short)readRegister(REG_FREQ_REF_SRC));
    }


    /**
     *  Sets the run command source
     * 
     *  @param src
     *  @throws DriverException 
     */
    public void setRunCmndSrc(RunCmndSrc src) throws DriverException
    {
        writeRegister(REG_RUN_CMND_SRC, runCmndSrcEnumMap.get(src));
    }


    /**
     *  Gets the run command source
     * 
     *  @return
     *  @throws DriverException 
     */
    public RunCmndSrc getRunCmndSrc() throws DriverException
    {
        return runCmndSrcValueMap.get((short)readRegister(REG_RUN_CMND_SRC));
    }


    /**
     *  Sets the stop method
     * 
     *  @param method
     *  @throws DriverException 
     */
    public void setStopMethod(StopMethod method) throws DriverException
    {
        writeRegister(REG_STOP_METHOD, stopMethodEnumMap.get(method));
    }


    /**
     *  Gets the stop method
     * 
     *  @return
     *  @throws DriverException 
     */
    public StopMethod getStopMethod() throws DriverException
    {
        return stopMethodValueMap.get((short)readRegister(REG_STOP_METHOD));
    }


    /**
     *  Sets the interaction between running and programming
     * 
     *  @param runProg
     *  @throws DriverException 
     */
    public void setRunProg(RunProg runProg) throws DriverException
    {
        writeRegister(REG_RUN_CMND_PROG, runProgEnumMap.get(runProg));
    }


    /**
     *  Gets the interaction between running and programming
     * 
     *  @return
     *  @throws DriverException 
     */
    public RunProg getRunProg() throws DriverException
    {
        return runProgValueMap.get((short)readRegister(REG_RUN_CMND_PROG));
    }


    /**
     *  Sets the communications error action
     * 
     *  @param action
     *  @throws DriverException 
     */
    public void setErrorAction(ErrorAction action) throws DriverException
    {
        writeRegister(REG_ERROR_ACTION, errorActionEnumMap.get(action));
    }


    /**
     *  Gets the communications error action
     * 
     *  @return
     *  @throws DriverException 
     */
    public ErrorAction getErrorAction() throws DriverException
    {
        return errorActionValueMap.get((short)readRegister(REG_ERROR_ACTION));
    }


    /**
     *  Sets whether reverse running is permitted
     * 
     *  @param permit
     *  @throws DriverException 
     */
    public void setReversePermitted(boolean permit) throws DriverException
    {
        writeRegister(REG_REV_PERMIT, permit ? REV_ENABLED : REV_DISABLED);
    }


    /**
     *  Gets whether reverse running is permitted
     * 
     *  @return
     *  @throws DriverException 
     */
    public boolean isReversePermitted() throws DriverException
    {
        return readRegister(REG_REV_PERMIT) == REV_ENABLED;
    }


    /**
     *  Sets whether communications faults are detected
     * 
     *  @param detect
     *  @throws DriverException 
     */
    public void setFaultDetected(boolean detect) throws DriverException
    {
        writeRegister(REG_COMM_FAULT_DET, detect ? FAULT_DETC_ON : FAULT_DETC_OFF);
    }


    /**
     *  Gets whether communications faults are detected
     * 
     *  @return
     *  @throws DriverException 
     */
    public boolean isFaultDetected() throws DriverException
    {
        return readRegister(REG_COMM_FAULT_DET) == FAULT_DETC_ON;
    }


    /**
     *  Sets whether enter command is required for parameter application
     * 
     *  @param require
     *  @throws DriverException 
     */
    public void setEnterRequired(boolean require) throws DriverException
    {
        writeRegister(REG_ENTER_FUNC, require ? ENTER_REQUIRED : ENTER_NOT_REQUIRED);
    }


    /**
     *  Gets whether enter command is required for parameter application
     * 
     *  @return
     *  @throws DriverException 
     */
    public boolean isEnterRequired() throws DriverException
    {
        return readRegister(REG_ENTER_FUNC) == ENTER_REQUIRED;
    }


    /**
     *  Sets whether run command is forward/reverse style
     * 
     *  @param fwdRev
     *  @throws DriverException 
     */
    public void setRunCmndFwdRev(boolean fwdRev) throws DriverException
    {
        writeRegister(REG_RUN_CMND_TYPE, fwdRev ? RUN_CMND_FWD_REV : RUN_CMND_OPER_DIRN);
        runCmndFwdRev = fwdRev;
    }


    /**
     *  Gets whether run command is forward/reverse style
     * 
     *  @return
     *  @throws DriverException 
     */
    public boolean isRunCmndFwdRev() throws DriverException
    {
        return readRegister(REG_RUN_CMND_TYPE) == RUN_CMND_FWD_REV;
    }


    /**
     *  Sets the voltage units
     * 
     *  @param units
     *  @throws DriverException 
     */
    public void setVoltageUnits(VoltageUnits units) throws DriverException
    {
        writeRegister(REG_VOLTAGE_UNITS, voltageUnitsEnumMap.get(units));
        voltageUnits = units;
    }


    /**
     *  Gets the voltage units
     * 
     *  @return
     *  @throws DriverException 
     */
    public VoltageUnits getVoltageUnits() throws DriverException
    {
        return voltageUnitsValueMap.get((short)readRegister(REG_VOLTAGE_UNITS));
    }


    /**
     *  Sets the input voltage
     * 
     *  @param voltage
     *  @throws DriverException 
     */
    public void setInputVoltage(double voltage) throws DriverException
    {
        writeRegister(REG_INPUT_VOLTAGE, (short)voltage);
    }


    /**
     *  Gets the input voltage
     * 
     *  @return
     *  @throws DriverException 
     */
    public double getInputVoltage() throws DriverException
    {
        return readRegister(REG_INPUT_VOLTAGE);
    }


    /**
     *  Sets the V/f profile
     * 
     *  @param level
     *  @throws DriverException 
     */
    public void setVfSelection(VfSelection level) throws DriverException
    {
        writeRegister(REG_VF_SELECTION, vfSelectionEnumMap.get(level));
    }


    /**
     *  Gets the V/f profile
     * 
     *  @return
     *  @throws DriverException 
     */
    public VfSelection getVfSelection() throws DriverException
    {
        return vfSelectionValueMap.get((short)readRegister(REG_VF_SELECTION));
    }


    /**
     *  Sets the frequency
     * 
     *  @param freq
     *  @throws DriverException 
     */
    public void setFrequency(double freq) throws DriverException
    {
        writeRegister(REG_FREQ_REFC, (short)(freq / FREQ_UNITS));
    }


    /**
     *  Gets the set frequency
     * 
     *  @return
     *  @throws DriverException 
     */
    public double getFrequency() throws DriverException
    {
        return FREQ_UNITS * readRegister(REG_FREQ_REFC);
    }


    /**
     *  Gets the applied frequency
     * 
     *  @return
     *  @throws DriverException 
     */
    public double getFrequencyApp() throws DriverException
    {
        return FREQ_UNITS * readRegister(REG_FREQ_REFC_IN);
    }


    /**
     *  Reads the frequency
     * 
     *  @return
     *  @throws DriverException 
     */
    public double readFrequency() throws DriverException
    {
        return FREQ_UNITS * readRegister(REG_OUTPUT_FREQ);
    }


    /**
     *  Reads the voltage
     * 
     *  @return
     *  @throws DriverException 
     */
    public double readVoltage() throws DriverException
    {
        return VOLT_UNITS * readRegister(REG_VOLTAGE_REFC);
    }


    /**
     *  Reads the current
     * 
     *  @return
     *  @throws DriverException 
     */
    public double readCurrent() throws DriverException
    {
        return ratedCurrent / 8192 * readRegister(REG_OUTPUT_CURR);
    }


    /**
     *  Reads the bus voltage
     * 
     *  @return
     *  @throws DriverException 
     */
    public double readBusVoltage() throws DriverException
    {
        return readRegister(REG_BUS_VOLTAGE);
    }


    /**
     *  Runs the motor forwards
     * 
     *  @throws DriverException 
     */
    public void runForward() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, (short)(runCmndFwdRev ? 1 : 3));
    }


    /**
     *  Runs the motor in reverse
     * 
     *  @throws DriverException 
     */
    public void runReverse() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, (short)(runCmndFwdRev ? 2 : 1));
    }


    /**
     *  Stops the motor
     * 
     *  @throws DriverException 
     */
    public void stop() throws DriverException
    {
        writeRegister(REG_COMMAND_MF_IN, ZERO);
    }

}
