package org.lsst.ccs.drivers.xed;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * General access routines for the XED device 
 * 
 * @author Homer Neal
 */
public class XED extends Ascii {

    /*
     * Private constants *
     */
    private final int timeout = 5000;
    private boolean isconnected = false ;

    /**
     * Constructor.
     */
    public XED()
    {
        setOptions(Option.NO_NET);
    }

    /**
     * Opens a connection.
     *
     * Overrides the main 4-argument Ascii open method.
     * 
     * @param connType   The connection type
     * @param serialName The device name of the serial port
     * @param baud       The baud rate
     * @param dataChar   The serial data characteristics
     * @throws DriverException
     */
    @Override
    public void open(ConnType connType, String serialName, int baud, int dataChar) throws DriverException
    {
        super.open(connType, serialName, baud, 0);
        init();
        setTimeout(timeout);
        try {
            reset();
            String response = null;
            for (int i=0;i<7;i++) {
                response = readXED();
                System.out.println(i+" : "+response);
            }
//            if (getIdent().contains("XED")) {
            if (getIdent().contains(":")) {
                isconnected = true;
            } else {
                System.out.println("XED: connected to port but handshake failed!"); 
            }
        } catch (Exception ex) {
            System.out.println("Failed to open connection to XED device!");
            isconnected = false;
        }
    
    }

    /**
     * Opens a serial connection.
     *
     * @param device The device name of the serial port
     * @param baud The baud rate
     * @throws DriverException
     */
    @Deprecated
    public void open(String device, int baud) throws DriverException
    {
        openSerial(device, baud);
    }
    
    /**
     * Opens an FTDI connection.
     * 
     * @param device The device name of the serial port
     * @param baud The baud rate
     * @throws DriverException
    */
    @Deprecated
    public void openftdi(String device, int baud) throws DriverException
    {
        openFtdi(device, baud);
    }

    /**
     *  Initializes device.
     */
    public void init()
    {
       setCommandTerm(Terminator.CRLF);
    }    
    
    /** reset to factory defaults **/
    public void reset() throws DriverException {
        writeXED("h");
    }

     /** Gets the XED identity. **/
    public String getIdent() throws DriverException {
        return readXED(" ") ;
    }

    public void extendFe55() throws DriverException {
         if (isconnected) writeXED("e");
    }

    public void retractFe55() throws DriverException {
         if (isconnected) writeXED("r");
    }

    /**
     * Writes a command.
     *
     * @param command The command to write, excluding terminator
     * @throws DriverException *
     */
    public synchronized void writeXED(String command) throws DriverException {
       write(command);
    }

    /**
     * Reads a response.
     *
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readXED() throws DriverException {
        return(read());
    }

    /**
     * Reads a response after writing a command.
     *
     * @param command The command to write, excluding terminator
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readXED(String command) throws DriverException {
        writeXED(command);
        return readXED();
    }

}
