package org.lsst.ccs.drivers.wattsup;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the WattsUp power meter driver
 *
 *  @author Owen Saxton
 */
public class TestWattsUp implements WattsUp.Listener {

    /*
     *  On/off enumeration
     */
    public enum OnOff {
        ON, OFF;
    }

    /*
     *  Memory-full action enumeration
     */
    public enum MemFullAction {

        SUSPEND(WattsUp.MEM_OPT_SUSPEND),
        OVERWRITE(WattsUp.MEM_OPT_OVERWRITE),
        CONDENSE(WattsUp.MEM_OPT_CONDENSE);

        int value;

        MemFullAction(int value) {
            this.value = value;
        }

        int getValue() {
            return value;
        }

    }

    /*
     *  Logging type enumeration
     */
    public enum LogType {
        INTERNAL, EXTERNAL;
    }

    /*
     *  Currency enumeration
     */
    public enum Currency {
        DOLLAR, EURO;
    }

    /*
     *  Data type enumeration
     */
    public enum DataType {
        DETAIL, SUMMARY;
    }

    /*
     *  Private constants
     */
    private final static Map<Integer, String> memWords = new HashMap<>();
    static {
        memWords.put(WattsUp.MEM_OPT_SUSPEND,   "suspend");
        memWords.put(WattsUp.MEM_OPT_OVERWRITE, "overwrite");
        memWords.put(WattsUp.MEM_OPT_CONDENSE,  "condense");
    }

    private final static Map<Integer, String> logstWords = new HashMap<>();
    static {
        logstWords.put(WattsUp.LOG_STA_SUSPENDED, "suspended");
        logstWords.put(WattsUp.LOG_STA_INTERNAL,  "internal");
        logstWords.put(WattsUp.LOG_STA_EXTERNAL,  "external");
    }

    /*
     *  Private fields
     */
    private final static PrintStream out = System.out;
    private final WattsUp wtu = new WattsUp();
    private int fieldMask, dataCount;
    private boolean dataSumm, listening;


    /**
     *  Opens connection to a meter.
     *
     *  @param  ident  The device name
     *  @throws  DriverException
     */
    @Command(description="Open connection to meter")
    public void open(@Argument(description="Device identification") String ident) throws DriverException
    {
        wtu.openSerial(ident);
        fieldMask = wtu.getLoggedFields();
    }


    /**
     *  Closes meter connection.
     *
     *  @throws  DriverException
     */
    @Command(description="Close connection to meter")
    public void close() throws DriverException
    {
        wtu.close();
    }


    /**
     *  Turns the listener on or off.
     *
     *  @param  action  Action to take: on or off
     */
    @Command(description="Enable or disable the listener")
    public void listen(@Argument(description="Action to take")
                       OnOff action)
    {
        listening = (action == OnOff.ON);
        if (listening) {
            wtu.addListener(this);
        }
        else {
            wtu.removeListener();
        }
    }


    /**
     *  Sets the mask of logged fields.
     *
     *  @param  mask  Mask of fields to be logged
     *  @throws  DriverException
     */
    @Command(name="fldset", description="Set the mask of logged fields")
    public void fldSet(@Argument(name="mask", description="Mask to set")
                       int mask) throws DriverException
    {
        fieldMask = mask;
        out.println("Record limit = " + wtu.setLoggedFields(fieldMask));
    }


    /**
     *  Shows the mask of logged fields.
     *
     *  @throws  DriverException
     */
    @Command(name="fldshow", description="Show the mask of logged fields")
    public void fldShow() throws DriverException
    {
        out.format("Value = %08x\n", wtu.getLoggedFields());
    }


    /**
     *  Shows calibration data.
     *
     *  @throws  DriverException
     */
    @Command(name="calshow", description="Show the calibration data")
    public void calShow() throws DriverException
    {
        showIntegers(wtu.getCalibrationData());
    }


    /**
     *  Shows header strings.
     *
     *  @throws  DriverException
     */
    @Command(name="hdrshow", description="Show the header strings")
    public void hdrShow() throws DriverException
    {
        showStrings(wtu.getHeaderRecord());
    }


    /**
     *  Shows the record limit.
     *
     *  @throws  DriverException
     */
    @Command(name="limshow", description="Show the record limit")
    public void limShow() throws DriverException
    {
        out.println("Record limit = " + wtu.getRecordLimit());
    }


    /**
     *  Sets the memory-full action.
     *
     *  @param  action  The memory-full action: suspend, overwrite or condense
     *  @throws  DriverException
     */
    @Command(name="memset", description="Set the memory-full action")
    public void memSet(@Argument(name="action", description="Action to take")
                       MemFullAction action) throws DriverException
    {
        wtu.setFullOption(action.getValue());
    }


    /**
     *  Shows the memory-full action.
     *
     *  @throws  DriverException
     */
    @Command(name="memshow", description="Show the memory-full action")
    public void memShow() throws DriverException
    {
        int option = wtu.getFullOption();
        out.format("Memory full option = %s (%s)\n",
                   option, memWords.get(option));
    }


    /**
     *  Sets the logging interval.
     *
     *  @param  intvl  The logging interval (secs)
     *  @throws  DriverException
     */
    @Command(name="intset", description="Set the logging interval")
    public void intSet(@Argument(name="intvl", description="Logging interval")
                       int intvl) throws DriverException
    {
        wtu.setLoggingInterval(intvl);
    }


    /**
     *  Sets the logging type and interval.
     *
     *  @param  type   The logging type: internal or external
     *  @param  intvl  The logging interval (secs)
     *  @throws  DriverException
     */
    @Command(name="logset", description="Set the logging type and interval")
    public void logSet(@Argument(name="type", description="Logging type")
                       LogType type,
                       @Argument(name="intvl", description="Logging Interval")
                       int intvl) throws DriverException
    {
        if (type == LogType.INTERNAL) {
            intvl = wtu.setInternalLogging(intvl);
            out.println("Resulting interval = " + intvl);
        }
        else {
            dataSumm = false;
            wtu.setExternalLogging(intvl);
            if (!listening) {
                out.println("Turn listener on for data display");
            }
        }
    }


    /**
     *  Shows the logging interval and state.
     *
     *  @throws  DriverException
     */
    @Command(name="logshow", description="Show the logging interval & state")
    public void logShow() throws DriverException
    {
        int state = wtu.getLoggingState();
        out.format("Logging interval = %s, logging state = %s (%s)\n",
                   wtu.getLoggingInterval(), state, logstWords.get(state));
    }


    /**
     *  Sets the user parameters.
     *
     *  @param  rate    The electricity rate: currency units / KWH
     *  @param  thresh  The duty cycle threshold
     *  @param  curr    The currency: dollar or euro
     *  @throws  DriverException
     */
    @Command(name="parmset", description="Set the user parameters")
    public void parmSet(@Argument(name="rate", description="Electricity rate")
                        double rate,
                        @Argument(name="thresh", description="Duty cycle threshold")
                        int thresh,
                        @Argument(name="curr", description="Currency")
                        Currency curr) throws DriverException
    {
        wtu.setUserParameters(rate, thresh, curr == Currency.EURO);
    }


    /**
     *  Sets the user parameters.
     *
     *  @param  rate    The electricity rate: currency units / KWH
     *  @param  thresh  The duty cycle threshold
     *  @throws  DriverException
     */
    @Command(name="parmset", description="Set the user parameters")
    public void parmSet(@Argument(name="rate", description="Electricity rate")
                        double rate,
                        @Argument(name="thresh", description="Duty cycle threshold")
                        int thresh) throws DriverException
    {
        parmSet(rate, thresh, Currency.DOLLAR);
    }


    /**
     *  Shows the user parameters.
     *
     *  @throws  DriverException
     */
    @Command(name="parmshow", description="Show the user parameters")
    public void parmShow() throws DriverException
    {
        out.format("Rate = %.3f, threshold = %s, euro = %s\n",
                   wtu.getUserRate(), wtu.getUserThreshold(), wtu.getUserEuro());
    }


    /**
     *  Shows the version strings.
     *
     *  @throws  DriverException
     */
    @Command(name="vershow", description="Show the version strings")
    public void verShow() throws DriverException
    {
        showStrings(wtu.getVersionData());
    }


    /**
     *  Resets memory.
     *
     *  @throws  DriverException
     */
    @Command(name="reset", description="Reset memory")
    public void reset() throws DriverException
    {
        wtu.resetMemory();
    }


    /**
     *  Restarts the meter.
     *
     *  @throws  DriverException
     */
    @Command(name="restart", description="Restart the meter")
    public void restart() throws DriverException
    {
        wtu.restart();
    }


    /**
     *  Shows data.
     *
     *  @param  type  The type of display: detail or summary
     *  @throws  DriverException
     */
    @Command(name="datashow", description="Show data")
    public void dataShow(@Argument(name="type", description="Display type")
                         DataType type) throws DriverException
    {
        dataSumm = (type == DataType.SUMMARY);
        dataCount = 0;
        if (!listening) {
            out.println("Turn listener on for data display");
        }
        int interval = wtu.getLoggedData();
        out.println("Record count = " + dataCount
                      + ", logging interval = " + interval);
        dataSumm = false;
    }


    /**
     *  Handles powered state change
     */
    @Override
    public void setPowered(boolean on)
    {
        out.println("Device powered " + (on ? "on" : "off"));
    }


    /**
     *  Handles error-induced port closure
     */
    @Override
    public void setClosed()
    {
        out.println("Device connection closed");
    }


    /**
     *  Processes logged data
     */
    private static final byte[] flags = {0x7c, 0x2f, 0x2d, 0x5c};

    @Override
    public void processData(double[] data)
    {
        dataCount++;

        if (dataSumm) {
            if ((dataCount & 0x07) == 0) {
                byte[] flag = {flags[(dataCount >> 3) & 3], 0x08};
                out.write(flag, 0, 2);
            }
        }
        else {
            showDoubles(data, fieldMask);
        }
    }


    /**
     *  Shows an array of strings
     */
    private void showStrings(String[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (String val : value) {
            String sValue = " " + val;
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


    /**
     *  Shows an array of integers
     */
    private void showIntegers(int[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int val : value) {
            String sValue = " " + val;
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


    /**
     *  Shows an array of doubles, selected by a mask
     */
    private void showDoubles(double[] value, int mask)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            if ((mask & (1 << j)) == 0) continue;
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }

}
