package org.lsst.ccs.drivers.wattsup;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 *  Program to enable direct talking to a WattsUp? meter.
 *
 *  @author Owen Saxton
 */
public class TalkWattsUp {

    private final static PrintStream out = System.out;
    private final Thread readW = new Thread(new Reader());
    private final Ascii asc = new Ascii();
    private int nRead = 0, nByte = 0;
    private boolean open;


    /**
     *  Inner class to implement meter reading thread.
     */
    private class Reader implements Runnable {

        byte[] data = new byte[1024];

        @Override
        public void run()
        {
            while (true) {
                int leng = 0;
                try {
                    leng = asc.readBytes(data, 0);
                }
                catch (DriverTimeoutException e) {
                    continue;
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                    }
                    break;
                }
                out.write(data, 0, leng);
                nRead++;
                nByte += leng;
            }
        }

    }


    /**
     *  Main program.
     *
     *  @param  args  Command-line arguments
     *  @throws  Exception
     */
    public static void main(String[] args) throws Exception
    {
        try {
            String serial = args.length >= 1 ? args[0] : null;
            new TalkWattsUp().run(serial);
        }
        catch (NumberFormatException e) {
            out.println("Invalid device index");
        }
        System.exit(0);
    }


    /**
     *  Sends commands to the meter.
     *
     *  Loops reading and processing each new typed command line.
     *
     *  @param  serial  The USB serial number of the device
     *  @throws  Exception
     */
    public void run(String serial) throws Exception
    {
        asc.openSerial(serial, 115200, 0);
        open = true;

        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.equals("")) {
                out.format("#reads: %s; #bytes: %s\n", nRead, nByte);
            }
            else {
                asc.writeBytes(line.getBytes());
            }
        }

        open = false;
        asc.close();
    }

}
