package org.lsst.ccs.drivers.wattsup;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ftdi.Ftdi;

/**
 **********************************************************
 *
 *  Program to enable direct talking to a WattsUp? meter.
 *
 *  @author Owen Saxton
 *
 **********************************************************
 */
public class TalkWattsUp {

    private final static PrintStream out = System.out;
    private final Thread readW = new Thread(new Reader());
    private final Ftdi ftd = new Ftdi();
    private int nRead = 0, nByte = 0;
    private boolean open;


   /**
    *  Inner class to implement meter reading thread.
    */
    private class Reader implements Runnable {

        byte[] data = new byte[1024];

        @Override
        public void run()
        {
            while (true) {
                int leng = 0;
                try {
                    leng = ftd.getQueueStatus();
                    if (leng == 0) {
                        ftd.awaitEvent(0);
                        leng = ftd.getQueueStatus();
                    }
                    if (leng > data.length) {
                        leng = data.length;
                    }
                    if (!open) break;
                    leng = ftd.read(data, 0, leng);
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                    }
                    break;
                }
                out.write(data, 0, leng);
                nRead++;
                nByte += leng;
            }
        }

    }


   /**
    *  Main program.
    *
    *  @param  args  Command-line arguments
    *
    *  @throws  Exception
    */
    public static void main(String[] args) throws Exception
    {
        try {
            int index = args.length >= 1 ? Integer.decode(args[0]) : 0;
            String serial = args.length >= 2 ? args[1] : null;
            (new TalkWattsUp()).run(index, serial);
        }
        catch (NumberFormatException e) {
            out.println("Invalid device index");
        }
        System.exit(0);
    }


   /**
    *  Sends commands to the meter.
    *
    *  Loops reading and processing each new typed command line.
    *
    *  @param  index   The index of the device
    *
    *  @param  serial  The USB serial number of the device
    *
    *  @throws  Exception
    */
    public void run(int index, String serial) throws Exception
    {
        ftd.open(index, serial);
        ftd.setBaudrate(115200);
        ftd.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                   Ftdi.PARITY_NONE);
        ftd.enableEvents(Ftdi.EVENT_RXCHAR);
        open = true;

        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.equals("")) {
                out.format("#reads: %s; #bytes: %s\n", nRead, nByte);
            }
            else {
                ftd.write(line.getBytes());
            }
        }

        open = false;
        ftd.close();
    }

}
