package org.lsst.ccs.drivers.wattsup;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ftdi.Ftdi;

/**
 ******************************************************************************
 **
 **  Controls a WattsUp? meter
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class WattsUp {

    /**
     **  Public constants
     */
    /** Data field - watts */
    public final static int FLD_WATTS         = 0;

    /** Data field - volts */
    public final static int FLD_VOLTS         = 1;

    /** Data field - amps */
    public final static int FLD_AMPS          = 2;

    /** Data field - watt-hours over collection period */
    public final static int FLD_WATT_HOURS    = 3;

    /** Data field - cost over collection period */
    public final static int FLD_COST          = 4;

    /** Data field - watt-hours per month */
    public final static int FLD_WH_MONTH      = 5;

    /** Data field - cost per month */
    public final static int FLD_COST_MONTH    = 6;

    /** Data field - maximum watts during interval */
    public final static int FLD_WATTS_MAX     = 7;

    /** Data field - maximum volts during interval */
    public final static int FLD_VOLTS_MAX     = 8;

    /** Data field - maximum amps during interval */
    public final static int FLD_AMPS_MAX      = 9;

    /** Data field - minimum watts during interval */
    public final static int FLD_WATTS_MIN     = 10;

    /** Data field - minimum volts during interval */
    public final static int FLD_VOLTS_MIN     = 11;

    /** Data field - minimum amps during interval */
    public final static int FLD_AMPS_MIN      = 12;

    /** Data field - power factor (watts / volt-amps) */
    public final static int FLD_POWER_FACTOR  = 13;

    /** Data field - duty cycle (fraction of time over power threshold) */
    public final static int FLD_DUTY_CYCLE    = 14;

    /** Data field - power cycle (number of times power was cycled) */
    public final static int FLD_POWER_CYCLE   = 15;

    /** Data field - line frequency */
    public final static int FLD_FREQUENCY     = 16;

    /** Data field - volt-amps (product of volts and amps) */
    public final static int FLD_VOLT_AMPS     = 17;

    /** Number of data fields */
    public final static int N_FIELDS          = 18;


    /** Number of calibration fields */
    public final static int N_CAL_FIELDS      = 48;


    /** Memory-full option - suspend logging */
    public final static int MEM_OPT_SUSPEND   = 0;

    /** Memory-full option - overwrite oldest data */
    public final static int MEM_OPT_OVERWRITE = 1;

    /** Memory-full option - condense (sparsify) the data */
    public final static int MEM_OPT_CONDENSE  = 2;


    /** Logging status - suspended (memory full) */
    public final static int LOG_STA_SUSPENDED = 0;

    /** Logging status - active internal */
    public final static int LOG_STA_INTERNAL  = 1;

    /** Logging status - external */
    public final static int LOG_STA_EXTERNAL  = 2;


    /** Version field - model */
    public final static int VER_FLD_MODEL     = 0;

    /** Version field - memory size */
    public final static int VER_FLD_MEMORY    = 1;

    /** Version field - major hardware version */
    public final static int VER_FLD_HWMAJOR   = 2;

    /** Version field - minor hardware version*/
    public final static int VER_FLD_HW_MINOR  = 3;

    /** Version field - major firmware version */
    public final static int VER_FLD_FWMAJOR   = 4;

    /** Version field - minor firmware version */
    public final static int VER_FLD_FWMINOR   = 5;

    /** Version field - firmware time stamp (YYYYMMDDHHMM) */
    public final static int VER_FLD_FWTSTAMP  = 6;

    /** Version field - checksum */
    public final static int VER_FLD_CHECKSUM  = 7;

    /** Number of version fields */
    public final static int N_VER_FIELDS      = 8;

    /**
     **  Private constants
     */
    private final static int
        BAUD_RATE       = 115200,
        READ_TIMEOUT    = 2000,
        READ_INTERVAL   = 1000,
        CHECK_INTERVAL  = 1000;

    /**
     **  Private fields
     */
    private final CheckPower checkPower;
    private final Ftdi ftd = new Ftdi();
    private final BlockingQueue replyQ = new ArrayBlockingQueue(10),
                                closeQ = new ArrayBlockingQueue(1);
    private boolean open, powered;
    private Reader reader;
    private int nDataSent;
    private Listener listener;


   /**
    ***************************************************************************
    **
    **  Implements the meter-reading thread.
    **
    ***************************************************************************
    */
    private class Reader extends Thread {

        private final byte HASH = 0x23, SEMIC = 0x3b, DATA_ID = 0x64;
        private int buffCurr, recdStart = -1, recdEnd = -1, recdCurr;
        private Exception excp;

        @Override
        public void run()
        {
            byte[] buff = new byte[1024];
            while (true) {

                // Wait until some data arrives and read it
                try {
                    int leng = ftd.getQueueStatus();
                    if (leng == 0) {
                        int event = 0;
                        while (open && event == 0) {
                            event = ftd.awaitEvent(READ_INTERVAL);
                        }
                        if (!open) break;
                        leng = ftd.getQueueStatus();
                    }
                    if (leng > buff.length - buffCurr) {
                        leng = buff.length - buffCurr;
                    }
                    leng = ftd.read(buff, buffCurr, leng);
                    buffCurr += leng;
                }
                catch (DriverException e) {
                    excp = e;
                    break;
                }

                // Loop until all the data has been processed
                while (recdCurr < buffCurr) {

                    // Look for record start (#) and end (;) markers
                    while (recdStart < 0 && recdCurr < buffCurr) {
                        if (buff[recdCurr++] == HASH) {
                            recdStart = recdCurr;
                        }
                    }
                    while (recdEnd < 0 && recdCurr < buffCurr) {
                        if (buff[recdCurr++] == SEMIC) {
                            recdEnd = recdCurr - 1;
                        }
                    }

                    // If no record found in buffer, we're done for now
                    if (recdStart < 0 || recdEnd < 0) break;

                    // Get the record ID
                    byte recdId = buff[recdStart];

                    // If it's not logged data or there's a listener, process it
                    if (recdId != DATA_ID || listener != null) {

                        // Form a string from the data
                        String record = new String(buff, recdStart,
                                                  recdEnd - recdStart);

                        // If it's logged data, pass it to the listener
                        if (recdId == DATA_ID) {
                            sendListener(record);
                        }

                        // Otherwise queue it to the receiver
                        else {
                            replyQ.offer(record);
                        }
                    }

                    // Set to look for next record
                    recdStart = -1;
                    recdEnd = -1;
                }

                // If start of new record not found, reset to buffer start
                if (recdStart < 0) {
                    buffCurr = 0;
                    recdCurr = 0;
                }

                // Otherwise copy partial record to buffer start and reset
                else {
                    int leng = buffCurr - recdStart;
                    System.arraycopy(buff, recdStart, buff, 0, leng);
                    buffCurr = leng;
                    recdStart = 0;
                    recdCurr = leng;
                }
            }

            // Error or close call: close the connection
            try {
                ftd.close();
            }
            catch (DriverException e) {
                if (excp == null) {
                    excp = e;
                }
            }

            // If error, queue the exception back to the caller and report
            //   closure to listener
            if (open) {
                open = false;
                powered = false;
                replyQ.clear();
                replyQ.offer(new DriverException(excp.getMessage()));
                if (listener != null) {
                    listener.setClosed();
                }
            }

            // If close request, queue response back to the caller
            else {
                if (excp == null) {
                    closeQ.offer(new Object());
                }
                else {
                    closeQ.offer(new DriverException(excp.getMessage()));
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Implements a thread to periodically check for read timeouts.
    **
    ***************************************************************************
    */
    private class CheckPower extends Thread {

        @Override
        public void run()
        {
            while (true) {
                if (open) {
                    try {
                        getRecordLimit();
                        if (!powered) {
                            powered = true;
                            if (listener != null) {
                                listener.setPowered(powered);
                            }
                        }
                    }
                    catch (DriverException e) {
                        if (e instanceof DriverTimeoutException) {
                            if (powered) {
                                powered = false;
                                if (listener != null) {
                                    listener.setPowered(powered);
                                }
                            }
                        }
                    }
                }
                try {
                    Thread.sleep(CHECK_INTERVAL);
                }
                catch (InterruptedException e) {
                    break;
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Defines an event listener.
    **
    ***************************************************************************
    */
    public interface Listener {

       /**
        ***********************************************************************
        **
        **  Processes logged data.
        **
        **  @param  data  An 18-element double array containing the data.
        **
        ***********************************************************************
        */
        public void processData(double[] data);

       /**
        ***********************************************************************
        **
        **  Handles power on/off transitions.
        **
        **  @param  on  Whether the power is on
        **
        ***********************************************************************
        */
        public void setPowered(boolean on);

       /**
        ***********************************************************************
        **
        **  Handles error-induced device closure.
        **
        ***********************************************************************
        */
        public void setClosed();

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public WattsUp()
    {
        checkPower = new CheckPower();
        checkPower.setName("WattsUp CheckPower");
        checkPower.setDaemon(true);
        checkPower.start();
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the meter.
    **
    **  @param  index   The zero-based index of the WattsUp meter within the
    **                  list of FTDI devices selected by the serial argument
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void open(int index, String serial) throws DriverException
    {
        open(null, index, serial);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the meter.
    **
    **  @param  node    The name of the node from which the meter is being
    **                  served, or null for the local node
    **
    **  @param  index   The zero-based index of the WattsUp meter within the
    **                  list of FTDI devices selected by the serial argument
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void open(String node, int index, String serial)
        throws DriverException
    {
        if (open) {
            throw new DriverException("Device already open");
        }
        ftd.open(node, index, serial);
        ftd.setBaudrate(BAUD_RATE);
        ftd.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                   Ftdi.PARITY_NONE);
        ftd.setTimeouts(READ_INTERVAL, 0);
        ftd.enableEvents(Ftdi.EVENT_RXCHAR);
        open = true;
        nDataSent = 0;
        replyQ.clear();
        closeQ.clear();
        reader = new Reader();
        reader.setName("WattsUp Reader");
        reader.setDaemon(true);
        reader.start();
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the meter.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void close() throws DriverException
    {
        checkOpen();
        open = false;
        powered = false;
        Object excp = null;
        try {
            excp = closeQ.take();
        }
        catch (InterruptedException e) {
        }
        if (excp instanceof DriverException) {
            throw (DriverException)excp;
        }
    }


   /**
    ***************************************************************************
    **
    **  Adds an event listener.
    **
    **  @param  listener  The event listener object, containing the
    **                    {@code processData}, {@code setPowered} and
    **                    {@code setClosed} methods.
    **
    ***************************************************************************
    */
    public void addListener(Listener listener)
    {
        this.listener = listener;
    }


   /**
    ***************************************************************************
    **
    **  Removes the event listener.
    **
    ***************************************************************************
    */
    public void removeListener()
    {
        listener = null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the bit mask of fields being logged.
    **
    **  @return  A bit mask where each set bit represents a field being
    **           logged.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getLoggedFields() throws DriverException
    {
        checkOpen();
        send("#C,R,0;");
        String[] reply = receive("c", N_FIELDS);
        int mask = 0;
        for (int j = 0; j < N_FIELDS; j++) {
            if (reply[j].contains("1")) {
                mask |= (1 << j);
            }
        }

        return mask;
    }


   /**
    ***************************************************************************
    **
    **  Sets the fields to be logged.
    **
    **  @param  mask  A bit mask where each set bit represents a field to be
    **                logged.
    **
    **  @return  The number of data records that can be logged internally.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int setLoggedFields(int mask) throws DriverException
    {
        checkOpen();
        String cmnd = "#C,W," + N_FIELDS;
        for (int j = 0; j < N_FIELDS; j++) {
            cmnd += "," + ((mask & (1 << j)) != 0 ? "1" : "0");
        }
        cmnd += ';';
        send(cmnd);

        return Integer.valueOf(receive("n", 1)[0]);
    }


   /**
    ***************************************************************************
    **
    **  Gets the calibration data.
    **
    **  @return  A 48-element integer array containing the calibration data.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int[] getCalibrationData() throws DriverException
    {
        checkOpen();
        send("#F,R,0;");
        String[] reply = receive("f", N_CAL_FIELDS);
        int[] values = new int[N_CAL_FIELDS];
        for (int j = 0; j < N_CAL_FIELDS; j++) {
            values[j] = (int)(long)Long.valueOf(reply[j]);
        }

        return values;
    }


   /**
    ***************************************************************************
    **
    **  Gets the header record.
    **
    **  @return  An 18-element string array containing short names for the
    **           defined fields.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized String[] getHeaderRecord() throws DriverException
    {
        checkOpen();
        send("#H,R,0;");

        return receive("h", N_FIELDS);
    }


   /**
    ***************************************************************************
    **
    **  Gets all the logged data.
    **
    **  All the logged data currently stored in the meter is read out and
    **  passed to the data listener's {@code processData} method, one record
    **  at a time.  This may take many seconds, depending on the amount of
    **  logged data.
    **
    **  @return  The current logging interval (seconds).
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getLoggedData() throws DriverException
    {
        checkOpen();
        send("#D,R,0;");
        receive("n", 3);
        int count = nDataSent;
        while (true) {
            try {
                return Integer.valueOf(receive("l", 2)[1]);
            }
            catch (DriverException e) {
                if (nDataSent == count) {
                    throw e;
                }
            }
            count = nDataSent;
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets internal logging.
    **
    **  Data logging is set to be internal, where each logged data record is
    **  stored in the meter's internal memory, for subsequent retrieval via
    **  {@code getLoggedData}.
    **
    **  @param  interval  The logging interval to set.  This is ignored if
    **                    the memory-full option is set to 2 (sparsify).
    **
    **  @return  The actual current logging interval.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int setInternalLogging(int interval)
        throws DriverException
    {
        checkOpen();
        send("#L,W,3,_,I," + interval + ";");

        return Integer.valueOf(receive("s", 3)[1]);
    }


   /**
    ***************************************************************************
    **
    **  Sets external logging.
    **
    **  Data logging is set to be external, wherein a data record is sent at
    **  the end of each logging interval and then passed to the {@code process}
    **  method of the data listener object.
    **
    **  @param  interval  The logging interval to set.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void setExternalLogging(int interval) throws DriverException
    {
        checkOpen();
        send("#L,W,3,_,E," + interval + ";");
    }


   /**
    ***************************************************************************
    **
    **  Gets the record count limit.
    **
    **  @return  The maximum number of data records than can be stored
    **           internally.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getRecordLimit() throws DriverException
    {
        checkOpen();
        send("#N,R,0;");

        return Integer.valueOf(receive("n", 1)[0]);
    }


   /**
    ***************************************************************************
    **
    **  Gets the memory-full handling option.
    **
    **  @return  The memory-full handling option, encoded as follows: 0 =
    **           pause; 1 = wrap; 2 = sparsify.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getFullOption() throws DriverException
    {
        checkOpen();
        send("#O,R,0;");

        return Integer.valueOf(receive("o", 1)[0]);
    }


   /**
    ***************************************************************************
    **
    **  Sets the memory-full handling option.
    **
    **  @param  option  The memory-full option to set, encoded as follows:
    **                  0 = pause; 1 = wrap; 2 = sparsify by deleting every
    **                  other record and doubling the logging interval.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void setFullOption(int option) throws DriverException
    {
        checkOpen();
        send("#O,W,1," + option + ";");
    }


   /**
    ***************************************************************************
    **
    **  Resets (clears) data memory.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void resetMemory() throws DriverException
    {
        checkOpen();
        send("#R,W,0;");
    }


   /**
    ***************************************************************************
    **
    **  Gets the logging interval.
    **
    **  @return  The current logging interval in effect.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getLoggingInterval() throws DriverException
    {
        checkOpen();
        send("#S,R,0;");

        return Integer.valueOf(receive("s", 3)[1]);
    }


   /**
    ***************************************************************************
    **
    **  Gets the logging state.
    **
    **  @return  The current logging state, encoded as follows: 0 = paused
    **           internal; 1 = active internal; 2 = external.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getLoggingState() throws DriverException
    {
        checkOpen();
        send("#S,R,0;");

        return Integer.valueOf(receive("s", 3)[2]);
    }


   /**
    ***************************************************************************
    **
    **  Sets the logging interval.
    **
    **  @param  interval  The logging interval to set.  This is ignored for
    **                    internal logging with the memory-full option set to
    **                    2 (sparsify).
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void setLoggingInterval(int interval) throws DriverException
    {
        checkOpen();
        send("#S,W,2,_," + interval + ";");
    }


   /**
    ***************************************************************************
    **
    **  Gets the user-set electricity rate.
    **
    **  @return  The user-set electricity rate in currency units (dollars or
    **           euros) per KWH.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized double getUserRate() throws DriverException
    {
        checkOpen();
        send("#U,R,0;");

        return Double.valueOf(receive("u", 3)[0]) / 1000;
    }


   /**
    ***************************************************************************
    **
    **  Gets the user-set duty cycle threshold.
    **
    **  @return  The user-set power threshold (watts) used for calculating
    **           the duty cycle.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized int getUserThreshold() throws DriverException
    {
        checkOpen();
        send("#U,R,0;");

        return Integer.valueOf(receive("u", 3)[1]);
    }


   /**
    ***************************************************************************
    **
    **  Gets the user-set euro flag.
    **
    **  @return  The user-set euro flag, true if the currency is euros and
    **           false if dollars.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized boolean getUserEuro() throws DriverException
    {
        checkOpen();
        send("#U,R,0;");

        return !receive("u", 3)[2].equals("0");
    }


   /**
    ***************************************************************************
    **
    **  Sets the user parameters.
    **
    **  @param  rate       The electricity rate in currency units (dollars or
    **                     euros) per KWH.
    **
    **  @param  threshold  The power threshold (watts) used for calculating
    **                     the duty cycle.
    **
    **  @param  euro       The euro flag, true if the currency is euros and
    **                     false if dollars.  This is used to display the
    **                     correct currency symbol on the meter.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void setUserParameters(double rate, int threshold, boolean euro)
        throws DriverException
    {
        checkOpen();
        send("#U,W,3," + (int)(1000 * rate + 0.5) + "," + threshold + ","
               + (euro ? "1" : "0") + ";");
    }


   /**
    ***************************************************************************
    **
    **  Gets the version information.
    **
    **  @return  An 8-element string array containing version information.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public synchronized String[] getVersionData() throws DriverException
    {
        checkOpen();
        send("#V,R,0;");

        return receive("v", N_VER_FIELDS);
    }


   /**
    ***************************************************************************
    **
    **  Restarts the meter.
    **
    **  This has the same effect as power cycling the meter.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    public void restart() throws DriverException
    {
        checkOpen();
        send("#V,W,0;");
    }


   /**
    ***************************************************************************
    **
    **  Sends a command to the meter.
    **
    **  The supplied command string is converted to a byte array and sent to
    **  the meter.
    **
    **  @param  cmnd  The command string to send.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void send(String cmnd) throws DriverException
    {
        ftd.write(cmnd.getBytes());
    }


   /**
    ***************************************************************************
    **
    **  Waits for a response from the meter and checks for correctness.
    **
    **  The timeout period is the default of two seconds.
    **
    **  @param  id     The ID (the first field) expected in the response.
    **
    **  @param  nFlds  The number of data fields expected in the response.
    **
    **  @return  A string obtained from all the bytes of the response between
    **           the beginning (#) and end (;) markers.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private String[] receive(String id, int nFlds) throws DriverException
    {
        return receive(id, nFlds, READ_TIMEOUT);
    }


   /**
    ***************************************************************************
    **
    **  Waits for a response from the meter and checks for correctness.
    **
    **  @param  id       The ID (the first field) expected in the response.
    **
    **  @param  nFlds    The number of data fields expected in the response.
    **
    **  @param  timeout  The timeout period in milliseconds.
    **
    **  @return  A string obtained from all the bytes of the response between
    **           the beginning (#) and end (;) markers.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private String[] receive(String id, int nFlds, int timeout)
        throws DriverException
    {
        while (true) {
            Object reply = null;
            try {
                reply = replyQ.poll(timeout, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
            }
            if (reply == null) {
                throw new DriverTimeoutException("Read timed out");
            }
            if (reply instanceof DriverException) {
                throw (DriverException)reply;
            }
            String[] head = ((String)reply).split(",", 4);
            if (head.length >= 4 && head[0].equals(id)
                  && Integer.valueOf(head[2]) == nFlds) {
                return (head.length < 4) ? null : head[3].split(",");
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Converts and sends logged data to the listener.
    **
    **  The reply string, which consists of a set of comma-delimited fields,
    **  is split into a string array, and the data fields of this are
    **  converted into a array of doubles which is passed to the data listener.
    **
    **  @param  reply  The reply string received from the meter.
    **
    ***************************************************************************
    */
    private void sendListener(String reply)
    {
        nDataSent++;
        if (listener == null) return;

        String[] fields = reply.split(",");
        double[] values = new double[N_FIELDS];

        for (int j = 0; j < N_FIELDS; j++) {
            String field = fields[j + 3];
            if (field.equals("_")) {
                values[j] = 0;
            }
            else {
                int temp = Integer.valueOf(field);
                if (j == FLD_AMPS || j == FLD_AMPS_MAX || j == FLD_AMPS_MIN
                      || j == FLD_COST || j == FLD_COST_MONTH) {
                    values[j] = temp / 1000.0;
                }
                else if (j == FLD_WH_MONTH || j == FLD_POWER_CYCLE) {
                    values[j] = temp;
                }
                else if (j == FLD_POWER_FACTOR || j == FLD_DUTY_CYCLE) {
                    values[j] = temp / 100.0;
                }
                else {
                    values[j] = temp / 10.0;
                }
            }
        }
        listener.processData(values);
    }


   /**
    ***************************************************************************
    **
    **  Checks that the meter is open.
    **
    **  @throws DriverException
    **
    ***************************************************************************
    */
    private void checkOpen() throws DriverException
    {
        if (!open) {
            throw new DriverException("Device not open");
        }
    }

}
