package org.lsst.ccs.drivers.wattsup;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to test a WattsUp power meter
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestWattsUp implements CmndProcess.Dispatch, WattsUp.Listener {

    /*
    **  Command codes
    */
    private final static int
        CMD_OPEN        = 0,
        CMD_CLOSE       = 1,
        CMD_FLDSHOW     = 2,
        CMD_CALSHOW     = 3,
        CMD_HDRSHOW     = 4,
        CMD_LIMSHOW     = 5,
        CMD_MEMSHOW     = 6,
        CMD_LOGSHOW     = 7,
        CMD_PARMSHOW    = 8,
        CMD_VERSHOW     = 9,
        CMD_FLDSET      = 10,
        CMD_LOGSET      = 11,
        CMD_MEMSET      = 12,
        CMD_RESET       = 13,
        CMD_INTSET      = 14,
        CMD_PARMSET     = 15,
        CMD_RESTART     = 16,
        CMD_DATASHOW    = 17,
        CMD_LISTEN      = 18,
        NUM_CMDS        = 19;

    /*
    **  Command help
    */
    private final static String[] helpOpen = {
        "Open meter connection",
        "open [<index>] [<serial>] [<node>]",
        "index   The index of the meter in the list of matching FTDI devices",
        "        (default 0)",
        "serial  The serial number fragment to be matched (default none)",
        "node    The node where the meter is attached (default local)",
    };

    private final static String[] helpClose = {
        "Close meter connection",
        "close",
    };

    private final static String[] helpListen = {
        "Turn the event listener on or off",
        "listen <action>",
        "action  The action to perform: 'on' or 'off'",
    };

    private final static String[] helpFldset = {
        "Set the fields to be logged",
        "fldset <mask>",
        "mask  Bit mask of the fields to be logged",
    };

    private final static String[] helpFldshow = {
        "Display the fields being logged",
        "fldshow",
    };

    private final static String[] helpCalshow = {
        "Display calibration data",
        "calshow",
    };

    private final static String[] helpHdrshow = {
        "Display header record",
        "hdrshow",
    };

    private final static String[] helpLimshow = {
        "Display the record limit",
        "limshow",
    };

    private final static String[] helpMemset = {
        "Set the memory-full option",
        "memset <option>",
        "option  The memory-full option: 'suspend', 'overwrite' or 'condense'",
    };

    private final static String[] helpMemshow = {
        "Display the memory-full option",
        "memshow",
    };

    private final static String[] helpIntset = {
        "Set the logging interval",
        "intset <interval>",
        "interval  The logging interval (secs)",
    };

    private final static String[] helpLogset = {
        "Set the logging type and interval",
        "logset <type> <interval>",
        "type      The logging type: 'internal' or 'external'",
        "interval  The logging interval (secs)",
    };

    private final static String[] helpLogshow = {
        "Display the logging state",
        "logshow",
    };

    private final static String[] helpParmset = {
        "Set the user parameters",
        "parmset <rate> <thresh> [<curr>]",
        "rate    The electricity rate (currency units / KWH)",
        "thresh  The duty cycle threshold (watts)",
        "curr    The currency: 'dollar' (default) or 'euro'",
    };

    private final static String[] helpParmshow = {
        "Display the user parameters",
        "parmshow",
    };

    private final static String[] helpVershow = {
        "Display the version data",
        "vershow",
    };

    private final static String[] helpReset = {
        "Reset (clear) the logging memory",
        "reset",
    };

    private final static String[] helpRestart = {
        "Restart the meter",
        "restart",
    };

    private final static String[] helpDatashow = {
        "Read and display all the stored data",
        "datashow [<level>]",
        "level    The display level: 'detail' (default) or 'summary' (show only",
        "         the record count)",
    };

    /*
    **  Command keyword definitions
    */
    private final static CmndProcess.Lookup onoffWords;
    static {
        onoffWords = new CmndProcess.Lookup(2);
        onoffWords.add("on",  1);
        onoffWords.add("off", 0);
    }

    private final static CmndProcess.Lookup memWords;
    static {
        memWords = new CmndProcess.Lookup(3);
        memWords.add("suspend",   WattsUp.MEM_OPT_SUSPEND);
        memWords.add("overwrite", WattsUp.MEM_OPT_OVERWRITE);
        memWords.add("condense",  WattsUp.MEM_OPT_CONDENSE);
    }

    private final static CmndProcess.Lookup logWords;
    static {
        logWords = new CmndProcess.Lookup(2);
        logWords.add("external", 1);
        logWords.add("internal", 0);
    }

    private final static CmndProcess.Lookup logshWords;
    static {
        logshWords = new CmndProcess.Lookup(3);
        logshWords.add("suspended", WattsUp.LOG_STA_SUSPENDED);
        logshWords.add("internal",  WattsUp.LOG_STA_INTERNAL);
        logshWords.add("external",  WattsUp.LOG_STA_EXTERNAL);
    }

    private final static CmndProcess.Lookup currWords;
    static {
        currWords = new CmndProcess.Lookup(2);
        currWords.add("euro",   1);
        currWords.add("dollar", 0);
    }

    private final static CmndProcess.Lookup dataWords;
    static {
        dataWords = new CmndProcess.Lookup(2);
        dataWords.add("summary", 1);
        dataWords.add("detail",  0);
    }

    /*
    **  Command definition table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("open",     CMD_OPEN,     helpOpen,     "iss");
        cmnd.add("close",    CMD_CLOSE,    helpClose,    "");
        cmnd.add("listen",   CMD_LISTEN,   helpListen,   "K",   onoffWords);
        cmnd.add("fldset",   CMD_FLDSET,   helpFldset,   "I");
        cmnd.add("fldshow",  CMD_FLDSHOW,  helpFldshow,  "");
        cmnd.add("calshow",  CMD_CALSHOW,  helpCalshow,  "");
        cmnd.add("hdrshow",  CMD_HDRSHOW,  helpHdrshow,  "");
        cmnd.add("limshow",  CMD_LIMSHOW,  helpLimshow,  "");
        cmnd.add("memset",   CMD_MEMSET,   helpMemset,   "K",   memWords);
        cmnd.add("memshow",  CMD_MEMSHOW,  helpMemshow,  "");
        cmnd.add("intset",   CMD_INTSET,   helpIntset,   "I");
        cmnd.add("logset",   CMD_LOGSET,   helpLogset,   "KI",  logWords);
        cmnd.add("logshow",  CMD_LOGSHOW,  helpLogshow,  "");
        cmnd.add("parmset",  CMD_PARMSET,  helpParmset,  "FIk", currWords);
        cmnd.add("parmshow", CMD_PARMSHOW, helpParmshow, "");
        cmnd.add("vershow",  CMD_VERSHOW,  helpVershow,  "");
        cmnd.add("reset",    CMD_RESET,    helpReset,    "");
        cmnd.add("restart",  CMD_RESTART,  helpRestart,  "");
        cmnd.add("datashow", CMD_DATASHOW, helpDatashow, "k",   dataWords);
    }

    /*
    **  Private fields
    */
    private final static PrintStream out = System.out;
    private final CmndProcess proc = new CmndProcess();
    private WattsUp wtu;
    private int fieldMask, dataCount;
    private boolean devOpen, dataSumm, listening;


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestWattsUp()).run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    **  <p>
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            wtu = new WattsUp();
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null || !proc.process(line)) break;
            }
            if (devOpen) {
                wtu.close();
            }
        }
        catch (Exception e) {
            out.println(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_OPEN:
                procOpen(found, args); break;
            case CMD_CLOSE:
                procClose(found, args); break;
            case CMD_LISTEN:
                procListen(found, args); break;
            case CMD_FLDSET:
                procFldset(found, args); break;
            case CMD_FLDSHOW:
                procFldshow(found, args); break;
            case CMD_CALSHOW:
                procCalshow(found, args); break;
            case CMD_HDRSHOW:
                procHdrshow(found, args); break;
            case CMD_LIMSHOW:
                procLimshow(found, args); break;
            case CMD_MEMSET:
                procMemset(found, args); break;
            case CMD_MEMSHOW:
                procMemshow(found, args); break;
            case CMD_INTSET:
                procIntset(found, args); break;
            case CMD_LOGSET:
                procLogset(found, args); break;
            case CMD_LOGSHOW:
                procLogshow(found, args); break;
            case CMD_PARMSET:
                procParmset(found, args); break;
            case CMD_PARMSHOW:
                procParmshow(found, args); break;
            case CMD_VERSHOW:
                procVershow(found, args); break;
            case CMD_RESET:
                procReset(found, args); break;
            case CMD_RESTART:
                procRestart(found, args); break;
            case CMD_DATASHOW:
                procDatashow(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (WattsUpException e) {
            out.println(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the OPEN command
    **
    ***************************************************************************
    */
    private void procOpen(int found, Object[] args) throws WattsUpException
    {
        int index = (found & 0x01) != 0 ? (Integer)args[0] : 0;
        String serial = (found & 0x02) != 0 ? (String)args[1] : null;
        String node = (found & 0x04) != 0 ? (String)args[2] : null;
        wtu.open(node, index, serial);
        devOpen = true;
        fieldMask = wtu.getLoggedFields();
    }


   /**
    ***************************************************************************
    **
    **  Processes the CLOSE command
    **
    ***************************************************************************
    */
    private void procClose(int found, Object[] args) throws WattsUpException
    {
        wtu.close();
        devOpen = false;
    }


   /**
    ***************************************************************************
    **
    **  Processes the LISTEN command
    **
    ***************************************************************************
    */
    private void procListen(int found, Object[] args)
    {
        listening = ((Integer)args[0] != 0);
        if (listening) {
            wtu.addListener(this);
        }
        else {
            wtu.removeListener();
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the FLDSET command
    **
    ***************************************************************************
    */
    private void procFldset(int found, Object[] args) throws WattsUpException
    {
        fieldMask = (Integer)args[0];
        int limit = wtu.setLoggedFields(fieldMask);
        out.println("Record limit = " + limit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the FLDSHOW command
    **
    ***************************************************************************
    */
    private void procFldshow(int found, Object[] args) throws WattsUpException
    {
        out.format("Value = %08x\n", wtu.getLoggedFields());
    }


   /**
    ***************************************************************************
    **
    **  Processes the CALSHOW command
    **
    ***************************************************************************
    */
    private void procCalshow(int found, Object[] args) throws WattsUpException
    {
        showIntegers(wtu.getCalibrationData());
    }


   /**
    ***************************************************************************
    **
    **  Processes the HDRSHOW command
    **
    ***************************************************************************
    */
    private void procHdrshow(int found, Object[] args) throws WattsUpException
    {
        showStrings(wtu.getHeaderRecord());
    }


   /**
    ***************************************************************************
    **
    **  Processes the LIMSHOW command
    **
    ***************************************************************************
    */
    private void procLimshow(int found, Object[] args) throws WattsUpException
    {
        out.println("Record limit = " + wtu.getRecordLimit());
    }


   /**
    ***************************************************************************
    **
    **  Processes the MEMSET command
    **
    ***************************************************************************
    */
    private void procMemset(int found, Object[] args) throws WattsUpException
    {
        wtu.setFullOption((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the MEMSHOW command
    **
    ***************************************************************************
    */
    private void procMemshow(int found, Object[] args) throws WattsUpException
    {
        int option = wtu.getFullOption();
        out.format("Memory full option = %s (%s)\n",
                   option, memWords.decode(option));
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTSET command
    **
    ***************************************************************************
    */
    private void procIntset(int found, Object[] args) throws WattsUpException
    {
        wtu.setLoggingInterval((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOGSET command
    **
    ***************************************************************************
    */
    private void procLogset(int found, Object[] args) throws WattsUpException
    {
        if ((Integer)args[0] == 0) {
            int intvl = wtu.setInternalLogging((Integer)args[1]);
            out.println("Resulting interval = " + intvl);
        }
        else {
            dataSumm = false;
            wtu.setExternalLogging((Integer)args[1]);
            if (!listening) {
                out.println("Turn listener on for data display");
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOGSHOW command
    **
    ***************************************************************************
    */
    private void procLogshow(int found, Object[] args) throws WattsUpException
    {
        int state = wtu.getLoggingState();
        out.format("Logging interval = %s, logging state = %s (%s)\n",
                   wtu.getLoggingInterval(), state, logshWords.decode(state));
    }


   /**
    ***************************************************************************
    **
    **  Processes the PARMSET command
    **
    ***************************************************************************
    */
    private void procParmset(int found, Object[] args) throws WattsUpException
    {
        boolean euro = (found & 0x04) != 0 && (Integer)args[2] != 0;
        wtu.setUserParameters((Float)args[0], (Integer)args[1], euro);
    }


   /**
    ***************************************************************************
    **
    **  Processes the PARMSHOW command
    **
    ***************************************************************************
    */
    private void procParmshow(int found, Object[] args) throws WattsUpException
    {
        out.format("Rate = %.3f, threshold = %s, euro = %s\n",
                   wtu.getUserRate(), wtu.getUserThreshold(),
                   wtu.getUserEuro());
    }


   /**
    ***************************************************************************
    **
    **  Processes the VERSHOW command
    **
    ***************************************************************************
    */
    private void procVershow(int found, Object[] args) throws WattsUpException
    {
        showStrings(wtu.getVersionData());
    }


   /**
    ***************************************************************************
    **
    **  Processes the RESET command
    **
    ***************************************************************************
    */
    private void procReset(int found, Object[] args) throws WattsUpException
    {
        wtu.resetMemory();
    }


   /**
    ***************************************************************************
    **
    **  Processes the RESTART command
    **
    ***************************************************************************
    */
    private void procRestart(int found, Object[] args) throws WattsUpException
    {
        wtu.restart();
    }


   /**
    ***************************************************************************
    **
    **  Processes the DATASHOW command
    **
    ***************************************************************************
    */
    private void procDatashow(int found, Object[] args) throws WattsUpException
    {
        dataSumm = (found & 0x01) != 0 && (Integer)args[0] != 0;
        dataCount = 0;
        if (!listening) {
            out.println("Turn listener on for data display");
        }
        int interval = wtu.getLoggedData();
        out.println("Record count = " + dataCount
                      + ", logging interval = " + interval);
        dataSumm = false;
    }


   /**
    ***************************************************************************
    **
    **  Handles powered state change
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        out.println("Device powered " + (on ? "on" : "off"));
    }


   /**
    ***************************************************************************
    **
    **  Handles error-induced port closure
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        out.println("Device connection closed");
        devOpen = false;
    }


   /**
    ***************************************************************************
    **
    **  Processes logged data
    **
    ***************************************************************************
    */
    private static final byte[] flags = {0x7c, 0x2f, 0x2d, 0x5c};

    @Override
    public void processData(float[] data)
    {
        dataCount++;

        if (dataSumm) {
            if ((dataCount & 0x07) == 0) {
                byte[] flag = {flags[(dataCount >> 3) & 3], 0x08};
                out.write(flag, 0, 2);
            }
        }
        else {
            showFloats(data, fieldMask);
        }
    }


   /**
    ***************************************************************************
    **
    **  Shows an array of strings
    **
    ***************************************************************************
    */
    private void showStrings(String[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Shows an array of integers
    **
    ***************************************************************************
    */
    private void showIntegers(int[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Shows an array of floats, selected by a mask
    **
    ***************************************************************************
    */
    private void showFloats(float[] value, int mask)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            if ((mask & (1 << j)) == 0) continue;
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }

}
