package org.lsst.ccs.drivers.wattsup;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.lsst.ccs.drivers.ftdi.Ftdi;
import org.lsst.ccs.drivers.ftdi.FtdiException;

/**
 ***************************************************************************
 **
 **  Program to enable direct talking to a WattsUp? meter
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TalkWatts {

    private final static PrintStream out = System.out;
    private final Thread readW = new Thread(new Reader());
    private Ftdi ftd = new Ftdi();
    private int nRead = 0, nByte = 0;
    private boolean open;


   /**
    ***************************************************************************
    **
    **  Inner class to implement meter reading thread
    **
    ***************************************************************************
    */
    private class Reader implements Runnable {

        byte[] data = new byte[1024];

        @Override
        public void run()
        {
            while (true) {
                int leng;
                try {
                    leng = ftd.read(data, 0, 1);
                    int nread = ftd.getQueueStatus();
                    if (nread > data.length - leng) {
                        nread = data.length - leng;
                    }
                    if (!open) break;
                    leng += ftd.read(data, leng, nread);
                }
                catch (FtdiException e) {
                    if (open) {
                        out.println(e);
                    }
                    break;
                }
                out.write(data, 0, leng);
                nRead++;
                nByte += leng;
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args) throws Exception
    {
        try {
            int index = args.length >= 1 ? Integer.decode(args[0]) : 0;
            String serial = args.length >= 2 ? args[1] : null;
            (new TalkWatts()).run(index, serial);
        }
        catch (NumberFormatException e) {
            out.println("Invalid device index");
        }
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Sends commands to the meter
    **
    **  <p>
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run(int index, String serial) throws Exception
    {
        ftd.open(index, serial);
        ftd.setBaudrate(115200);
        ftd.setDataCharacteristics(Ftdi.DATABITS_8, Ftdi.STOPBITS_1,
                                   Ftdi.PARITY_NONE);
        open = true;

        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.equals("")) {
                out.format("#reads: %s; #bytes: %s\n", nRead, nByte);
            }
            else {
                ftd.write(line.getBytes());
            }
        }

        open = false;
        ftd.close();
    }

}
