package org.lsst.ccs.drivers.wattsup;

import gnu.io.CommPortIdentifier;
import gnu.io.NoSuchPortException;
import gnu.io.PortInUseException;
import gnu.io.SerialPort;
import gnu.io.SerialPortEvent;
import gnu.io.SerialPortEventListener;
import gnu.io.UnsupportedCommOperationException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import jline.ConsoleReader;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to enable direct talking to a WattsUp? meter
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TalkWatts {

    private final String dfltName = "/dev/ttyUSB0";

    private final Output out = new ConsOut();
    private final ConsoleReader readc = new ConsoleReader();
    private final CommPortIdentifier portId;
    private final SerialPort port;
    private final InputStream winp;
    private final OutputStream wout;
    private final Thread readw = new Thread(new Reader());
    private final Listener listenw = new Listener();
    private boolean gotData = false;
    private byte[] data = new byte[1024];
    private int nRead = 0, nByte = 0;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TalkWatts(String portName) throws Exception
    {
        if (portName == null) portName = dfltName;
        portId = CommPortIdentifier.getPortIdentifier(portName);
        port = (SerialPort)portId.open("TalkWatts", 1000);
        port.setSerialPortParams(115200, SerialPort.DATABITS_8,
                                 SerialPort.STOPBITS_1, SerialPort.PARITY_NONE);
        winp = port.getInputStream();
        wout = port.getOutputStream();
        port.addEventListener(listenw);
        port.notifyOnDataAvailable(true);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args) throws Exception
    {
        try {
            (new TalkWatts(args.length > 0 ? args[0] : null)).run();
        }
        catch (IOException e) {
            System.out.println(e);
        }
        catch (NoSuchPortException e) {
            System.out.println(e);
        }
        catch (PortInUseException e) {
            System.out.println(e);
        }
        catch (UnsupportedCommOperationException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Send commands to the meter
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run() throws IOException
    {
        readw.start();

        while (true) {
            String line = readc.readLine("");
            if (line == null) break;
            if (line.equals(""))
                out.format("#reads: %s; #bytes: %s\n", nRead, nByte);
            wout.write(line.getBytes());
        }

        port.close();
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement meter reading thread
    **
    ***************************************************************************
    */
    private class Reader implements Runnable {

        @Override
        public void run()
        {
            while (true) {
                int leng;
                try {
                    Thread.sleep(1000);
                }
                catch (InterruptedException e) {
                }
                if (!gotData) continue;
                gotData = false;
                try {
                    leng = winp.read(data);
                }
                catch (IOException e) {
                    out.println(e);
                    break;
                }
                if (leng <= 0) break;
                out.write(data, 0, leng);
                nRead++;
                nByte += leng;
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement serial port listener
    **
    ***************************************************************************
    */
    private class Listener implements SerialPortEventListener {

        @Override
        public void serialEvent(SerialPortEvent event)
        {
            if (event.getEventType() == SerialPortEvent.DATA_AVAILABLE) {
                gotData = true;
                readw.interrupt();
            }
        }

    }

}
