package org.lsst.ccs.drivers.wattsup;

import gnu.io.SerialPort;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test a WattsUp power meter
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestWattsUp implements CmndProcess.Dispatch, WattsUp.Listener {

    /*
    **  Command codes
    */
    private final static int
        CMD_OPEN        = 0,
        CMD_CLOSE       = 1,
        CMD_FLDSHOW     = 2,
        CMD_CALSHOW     = 3,
        CMD_HDRSHOW     = 4,
        CMD_LIMSHOW     = 5,
        CMD_MEMSHOW     = 6,
        CMD_LOGSHOW     = 7,
        CMD_PARMSHOW    = 8,
        CMD_VERSHOW     = 9,
        CMD_FLDSET      = 10,
        CMD_LOGSET      = 11,
        CMD_MEMSET      = 12,
        CMD_RESET       = 13,
        CMD_INTSET      = 14,
        CMD_PARMSET     = 15,
        CMD_RESTART     = 16,
        CMD_DATASHOW    = 17,
        CMD_PORTSHOW    = 18,
        NUM_CMDS        = 19;

    /*
    **  Command help
    */
    private final static String[] helpOpen = {
        "Open device connection",
        "open [<device>]",
        "device  The name of the serial port to connect to",
    };

    private final static String[] helpClose = {
        "Close device connection",
        "close",
    };

    private final static String[] helpFldset = {
        "Set the fields to be logged",
        "fldset <mask>",
        "mask  Bit mask of the fields to be logged",
    };

    private final static String[] helpFldshow = {
        "Display the fields being logged",
        "fldshow",
    };

    private final static String[] helpCalshow = {
        "Display calibration data",
        "calshow",
    };

    private final static String[] helpHdrshow = {
        "Display header record",
        "hdrshow",
    };

    private final static String[] helpLimshow = {
        "Display the record limit",
        "limshow",
    };

    private final static String[] helpMemset = {
        "Set the memory-full option",
        "memset <option>",
        "option  The memory-full option",
    };

    private final static String[] helpMemshow = {
        "Display the memory full option",
        "memshow",
    };

    private final static String[] helpLogset = {
        "Set the logging type and interval",
        "logset <type> <interval>",
        "type      The logging type: i = internal, e = external",
        "interval  The logging interval (secs)",
    };

    private final static String[] helpIntset = {
        "Set the logging interval",
        "intset <interval>",
        "interval  The logging interval (secs)",
    };

    private final static String[] helpLogshow = {
        "Display the logging state",
        "logshow",
    };

    private final static String[] helpParmset = {
        "Set the user parameters",
        "parmset <rate> <thresh> [<euro>]",
        "rate    The electricity rate (currency units / KWH)",
        "thresh  The duty cycle threshold (watts)",
        "euro    If non-zero, currency is euros, not dollars",
    };

    private final static String[] helpParmshow = {
        "Display the user parameters",
        "parmshow",
    };

    private final static String[] helpVershow = {
        "Display the version data",
        "vershow",
    };

    private final static String[] helpReset = {
        "Reset (clear) the logging memory",
        "reset",
    };

    private final static String[] helpRestart = {
        "Restart the meter",
        "restart",
    };

    private final static String[] helpDatashow = {
        "Read and display all the stored data",
        "datashow <summary>",
        "summary  If present and non-zero, show only the record count",
    };

    private final static String[] helpPortshow = {
        "Show the state of the serial port being used",
        "portshow",
    };

    /*
    **  Command definition table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("open",     CMD_OPEN,     helpOpen,     "s");
        cmnd.add("close",    CMD_CLOSE,    helpClose,    "");
        cmnd.add("fldshow",  CMD_FLDSHOW,  helpFldshow,  "");
        cmnd.add("calshow",  CMD_CALSHOW,  helpCalshow,  "");
        cmnd.add("hdrshow",  CMD_HDRSHOW,  helpHdrshow,  "");
        cmnd.add("limshow",  CMD_LIMSHOW,  helpLimshow,  "");
        cmnd.add("memshow",  CMD_MEMSHOW,  helpMemshow,  "");
        cmnd.add("logshow",  CMD_LOGSHOW,  helpLogshow,  "");
        cmnd.add("parmshow", CMD_PARMSHOW, helpParmshow, "");
        cmnd.add("vershow",  CMD_VERSHOW,  helpVershow,  "");
        cmnd.add("fldset",   CMD_FLDSET,   helpFldset,   "I");
        cmnd.add("logset",   CMD_LOGSET,   helpLogset,   "SI");
        cmnd.add("memset",   CMD_MEMSET,   helpMemset,   "I");
        cmnd.add("reset",    CMD_RESET,    helpReset,    "");
        cmnd.add("intset",   CMD_INTSET,   helpIntset,   "I");
        cmnd.add("parmset",  CMD_PARMSET,  helpParmset,  "FIi");
        cmnd.add("restart",  CMD_RESTART,  helpRestart,  "");
        cmnd.add("datashow", CMD_DATASHOW, helpDatashow, "i");
        cmnd.add("portshow", CMD_PORTSHOW, helpPortshow, "");
    }

    /*
    **  Private fields
    */
    private final Output out = new ConsOut();
    private final CmndProcess proc = new CmndProcess();
    private WattsUp wtu;
    private int fieldMask, dataCount;
    private boolean devOpen, dataSumm;


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestWattsUp()).run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    **  <p>
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            wtu = new WattsUp();
            wtu.addListener(this);
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null) break;
                if (!proc.process(line)) break;
            }
        }
        catch (Exception e) {
            out.println(e);
        }

        if (devOpen) wtu.close();
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        switch (code) {
        case CMD_OPEN:
            procOpen(found, args); break;
        case CMD_CLOSE:
            procClose(found, args); break;
        case CMD_FLDSET:
            procFldset(found, args); break;
        case CMD_FLDSHOW:
            procFldshow(found, args); break;
        case CMD_CALSHOW:
            procCalshow(found, args); break;
        case CMD_HDRSHOW:
            procHdrshow(found, args); break;
        case CMD_LIMSHOW:
            procLimshow(found, args); break;
        case CMD_MEMSET:
            procMemset(found, args); break;
        case CMD_MEMSHOW:
            procMemshow(found, args); break;
        case CMD_LOGSET:
            procLogset(found, args); break;
        case CMD_INTSET:
            procIntset(found, args); break;
        case CMD_LOGSHOW:
            procLogshow(found, args); break;
        case CMD_PARMSET:
            procParmset(found, args); break;
        case CMD_PARMSHOW:
            procParmshow(found, args); break;
        case CMD_VERSHOW:
            procVershow(found, args); break;
        case CMD_RESET:
            procReset(found, args); break;
        case CMD_RESTART:
            procRestart(found, args); break;
        case CMD_DATASHOW:
            procDatashow(found, args); break;
        case CMD_PORTSHOW:
            procPortshow(found, args); break;
        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the OPEN command
    **
    ***************************************************************************
    */
    private void procOpen(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            wtu.open((String)args[0]);
        else
            wtu.open();
        devOpen = true;
        fieldMask = wtu.getLoggedFields();
    }


   /**
    ***************************************************************************
    **
    **  Processes the CLOSE command
    **
    ***************************************************************************
    */
    private void procClose(int found, Object[] args)
    {
        wtu.close();
        devOpen = false;
    }


   /**
    ***************************************************************************
    **
    **  Processes the FLDSET command
    **
    ***************************************************************************
    */
    private void procFldset(int found, Object[] args)
    {
        fieldMask = (Integer)args[0];
        int limit = wtu.setLoggedFields(fieldMask);
        out.println("Record limit = " + limit);
    }


   /**
    ***************************************************************************
    **
    **  Processes the FLDSHOW command
    **
    ***************************************************************************
    */
    private void procFldshow(int found, Object[] args)
    {
        out.format("Value = %08x\n", wtu.getLoggedFields());
    }


   /**
    ***************************************************************************
    **
    **  Processes the CALSHOW command
    **
    ***************************************************************************
    */
    private void procCalshow(int found, Object[] args)
    {
        showIntegers(wtu.getCalibrationData());
    }


   /**
    ***************************************************************************
    **
    **  Processes the HDRSHOW command
    **
    ***************************************************************************
    */
    private void procHdrshow(int found, Object[] args)
    {
        showStrings(wtu.getHeaderRecord());
    }


   /**
    ***************************************************************************
    **
    **  Processes the LIMSHOW command
    **
    ***************************************************************************
    */
    private void procLimshow(int found, Object[] args)
    {
        out.println("Record limit = " + wtu.getRecordLimit());
    }


   /**
    ***************************************************************************
    **
    **  Processes the MEMSET command
    **
    ***************************************************************************
    */
    private void procMemset(int found, Object[] args)
    {
        wtu.setFullOption((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the MEMSHOW command
    **
    ***************************************************************************
    */
    private void procMemshow(int found, Object[] args)
    {
        out.println("Memory full option = " + wtu.getFullOption());
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOGSET command
    **
    ***************************************************************************
    */
    private void procLogset(int found, Object[] args)
    {
        String type = (String)args[0];
        if (type.equalsIgnoreCase("i")) {
            int intvl = wtu.setInternalLogging((Integer)args[1]);
            out.println("Resulting interval = " + intvl);
        }
        else if (type.equalsIgnoreCase("e")) {
            dataSumm = false;
            wtu.setExternalLogging((Integer)args[1]);
        }
        else
            out.println("Invalid logging type");
    }


   /**
    ***************************************************************************
    **
    **  Processes the INTSET command
    **
    ***************************************************************************
    */
    private void procIntset(int found, Object[] args)
    {
        wtu.setLoggingInterval((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOGSHOW command
    **
    ***************************************************************************
    */
    private void procLogshow(int found, Object[] args)
    {
        out.println("Logging interval = " + wtu.getLoggingInterval()
                      + ", logging state = " + wtu.getLoggingState());
    }


   /**
    ***************************************************************************
    **
    **  Processes the PARMSET command
    **
    ***************************************************************************
    */
    private void procParmset(int found, Object[] args)
    {
        boolean euro = (found & 0x04) != 0 && (Integer)args[2] != 0;
        wtu.setUserParameters((Float)args[0], (Integer)args[1], euro);
    }


   /**
    ***************************************************************************
    **
    **  Processes the PARMSHOW command
    **
    ***************************************************************************
    */
    private void procParmshow(int found, Object[] args)
    {
        out.format("Rate = %.3f, threshold = %s, euro = %s\n",
                   wtu.getUserRate(), wtu.getUserThreshold(),
                   wtu.getUserEuro());
    }


   /**
    ***************************************************************************
    **
    **  Processes the VERSHOW command
    **
    ***************************************************************************
    */
    private void procVershow(int found, Object[] args)
    {
        showStrings(wtu.getVersionData());
    }


   /**
    ***************************************************************************
    **
    **  Processes the RESET command
    **
    ***************************************************************************
    */
    private void procReset(int found, Object[] args)
    {
        wtu.resetMemory();
    }


   /**
    ***************************************************************************
    **
    **  Processes the RESTART command
    **
    ***************************************************************************
    */
    private void procRestart(int found, Object[] args)
    {
        wtu.restart();
    }


   /**
    ***************************************************************************
    **
    **  Processes the DATASHOW command
    **
    ***************************************************************************
    */
    private void procDatashow(int found, Object[] args)
    {
        dataSumm = (found & 0x01) != 0 && (Integer)args[0] != 0;
        dataCount = 0;
        int interval = wtu.getLoggedData();
        out.println("Record count = " + dataCount
                      + ", logging interval = " + interval);
        dataSumm = false;
    }


   /**
    ***************************************************************************
    **
    **  Processes the PORTSHOW command
    **
    ***************************************************************************
    */
    private void procPortshow(int found, Object[] args)
    {
        SerialPort p = wtu.getPort();
        if (p == null) {
            out.println("No connection open");
            return;
        }
        out.format("CD = %s, CTS = %s, DSR = %s, DTR = %s, RI = %s, "
                     + "RTS = %s\n",
                   p.isCD() ? "on" : "off", p.isCTS() ? "on" : "off",
                   p.isDSR() ? "on" : "off", p.isDTR() ? "on" : "off",
                   p.isRI() ? "on" : "off", p.isRTS() ? "on" : "off");
    }


   /**
    ***************************************************************************
    **
    **  Handles powered state change
    **
    ***************************************************************************
    */
    @Override
    public void setPowered(boolean on)
    {
        out.println("Device powered " + (on ? "on" : "off"));
    }


   /**
    ***************************************************************************
    **
    **  Handles error-induced port closure
    **
    ***************************************************************************
    */
    @Override
    public void setClosed()
    {
        out.println("Device connection closed");
        devOpen = false;
    }


   /**
    ***************************************************************************
    **
    **  Processes logged data
    **
    ***************************************************************************
    */
    private static final byte[] flags = {0x7c, 0x2f, 0x2d, 0x5c};

    @Override
    public void processData(float[] data)
    {
        dataCount++;

        if (dataSumm) {
            if ((dataCount & 0x07) == 0) {
                byte[] flag = {flags[(dataCount >> 3) & 3], 0x08};
                out.write(flag, 0, 2);
            }
        }
        else
            showFloats(data, fieldMask);
    }


   /**
    ***************************************************************************
    **
    **  Shows an array of strings
    **
    ***************************************************************************
    */
    private void showStrings(String[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Shows an array of integers
    **
    ***************************************************************************
    */
    private void showIntegers(int[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Shows an array of floats, selected by a mask
    **
    ***************************************************************************
    */
    private void showFloats(float[] value, int mask)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            if ((mask & (1 << j)) == 0) continue;
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }

}
