package org.lsst.ccs.drivers.wattsup;

import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;
import java.util.Scanner;
import java.io.IOException;
import jline.ConsoleReader;

/**
 ***************************************************************************
 **
 **  Program to test a WattsUp power meter
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestWattsUp implements CmndProc.Dispatch, WattsUp.DataListener {

    private final static int
        CMD_OPEN        = 0,
        CMD_CLOSE       = 1,
        CMD_FLDSHOW     = 2,
        CMD_CALSHOW     = 3,
        CMD_HDRSHOW     = 4,
        CMD_LIMSHOW     = 5,
        CMD_MEMSHOW     = 6,
        CMD_LOGSHOW     = 7,
        CMD_PARMSHOW    = 8,
        CMD_VERSHOW     = 9,
        CMD_FLDSET      = 10,
        CMD_LOGSET      = 11,
        CMD_MEMSET      = 12,
        CMD_RESET       = 13,
        CMD_INTSET      = 14,
        CMD_PARMSET     = 15,
        CMD_RESTART     = 16,
        CMD_DATASHOW    = 17,
        NUM_CMDS        = 18;

    private final static String[] helpOpen = {
        "Open device connection",
        "open [<device>]",
        "device  The name of the serial port to connect to",
    };

    private final static String[] helpClose = {
        "Close device connection",
        "close",
    };

    private final static String[] helpFldset = {
        "Set the fields to be logged",
        "fldset <mask>",
        "mask  Bit mask of the fields to be logged",
    };

    private final static String[] helpFldshow = {
        "Display the fields being logged",
        "fldshow",
    };

    private final static String[] helpCalshow = {
        "Display calibration data",
        "calshow",
    };

    private final static String[] helpHdrshow = {
        "Display header record",
        "hdrshow",
    };

    private final static String[] helpLimshow = {
        "Display the record limit",
        "limshow",
    };

    private final static String[] helpMemset = {
        "Set the memory-full option",
        "memset <option>",
        "option  The memory-full option",
    };

    private final static String[] helpMemshow = {
        "Display the memory full option",
        "memshow",
    };

    private final static String[] helpLogset = {
        "Set the logging type and interval",
        "logset <type> <interval>",
        "type      The logging type: i = internal, e = external",
        "interval  The logging interval (secs)",
    };

    private final static String[] helpIntset = {
        "Set the logging interval",
        "intset <interval>",
        "interval  The logging interval (secs)",
    };

    private final static String[] helpLogshow = {
        "Display the logging state",
        "logshow",
    };

    private final static String[] helpParmset = {
        "Set the user parameters",
        "parmset <rate> <thresh> [<euro>]",
        "rate    The electricity rate (currency units / KWH)",
        "thresh  The duty cycle threshold (watts)",
        "euro    If non-zero, currency is euros, not dollars",
    };

    private final static String[] helpParmshow = {
        "Display the user parameters",
        "parmshow",
    };

    private final static String[] helpVershow = {
        "Display the version data",
        "vershow",
    };

    private final static String[] helpReset = {
        "Reset (clear) the logging memory",
        "reset",
    };

    private final static String[] helpRestart = {
        "Restart the meter",
        "restart",
    };

    private final static String[] helpDatashow = {
        "Read and display all the stored data",
        "datashow <summary>",
        "summary  If present and non-zero, show only the record count",
    };


    private final Output out = new ConsOut();
    private final CmndProc.Lookup devNames;
    private final ConsoleReader reader = new ConsoleReader();
    private final CmndProc proc = new CmndProc();
    private final WattsUp wtu = new WattsUp();
    private int fieldMask, dataCount;
    private boolean devOpen, dataSumm;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestWattsUp() throws IOException 
    {
        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("open",        CMD_OPEN,        helpOpen);
        cmnd.add("close",       CMD_CLOSE,       helpClose);
        cmnd.add("fldshow",     CMD_FLDSHOW,     helpFldshow);
        cmnd.add("calshow",     CMD_CALSHOW,     helpCalshow);
        cmnd.add("hdrshow",     CMD_HDRSHOW,     helpHdrshow);
        cmnd.add("limshow",     CMD_LIMSHOW,     helpLimshow);
        cmnd.add("memshow",     CMD_MEMSHOW,     helpMemshow);
        cmnd.add("logshow",     CMD_LOGSHOW,     helpLogshow);
        cmnd.add("parmshow",    CMD_PARMSHOW,    helpParmshow);
        cmnd.add("vershow",     CMD_VERSHOW,     helpVershow);
        cmnd.add("fldset",      CMD_FLDSET,      helpFldset);
        cmnd.add("logset",      CMD_LOGSET,      helpLogset);
        cmnd.add("memset",      CMD_MEMSET,      helpMemset);
        cmnd.add("reset",       CMD_RESET,       helpReset);
        cmnd.add("intset",      CMD_INTSET,      helpIntset);
        cmnd.add("parmset",     CMD_PARMSET,     helpParmset);
        cmnd.add("restart",     CMD_RESTART,     helpRestart);
        cmnd.add("datashow",    CMD_DATASHOW,    helpDatashow);

        proc.add(this, cmnd);

        devNames = new CmndProc.Lookup(1);
        devNames.add("tc-ai", 0);

        wtu.addDataListener(this);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        TestWattsUp twtu = null;

        try {
            twtu = new TestWattsUp();
            twtu.run();
        }
        catch (IOException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run() throws IOException
    {
        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            try {
                if (!proc.process(line)) break;
            }
            catch (WattsUpException e) {
                out.println(e);
            }
        }
        if (devOpen) wtu.close();
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (code) {

        case CMD_OPEN:
            if ((found = CmndProc.scanArgs(scan, "s", args)) >= 0) {
                if ((found & 0x01) != 0)
                    wtu.open((String)args[0]);
                else
                    wtu.open();
                devOpen = true;
                fieldMask = wtu.getLoggedFields();
            }
            break;
        
        case CMD_CLOSE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                wtu.close();
                devOpen = false;
            }
            break;

        case CMD_FLDSET:
            if ((found = CmndProc.scanArgs(scan, "I", args)) >= 0) {
                fieldMask = (Integer)args[0];
                int limit = wtu.setLoggedFields(fieldMask);
                out.println("Record limit = " + limit);
            }
            break;

        case CMD_FLDSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.format("Value = %08x\n", wtu.getLoggedFields());
            }
            break;

        case CMD_CALSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                showIntegers(wtu.getCalibrationData());
            break;

        case CMD_HDRSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                showStrings(wtu.getHeaderRecord());
            break;

        case CMD_LIMSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                out.println("Record limit = " + wtu.getRecordLimit());
            break;

        case CMD_MEMSET:
            if ((found = CmndProc.scanArgs(scan, "I", args)) >= 0)
                wtu.setFullOption((Integer)args[0]);
            break;

        case CMD_MEMSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                out.println("Memory full option = " + wtu.getFullOption());
            break;

        case CMD_LOGSET:
            if ((found = CmndProc.scanArgs(scan, "SI", args)) >= 0) {
                String type = (String)args[0];
                if (type.equalsIgnoreCase("i")) {
                    int intvl = wtu.setInternalLogging((Integer)args[1]);
                    out.println("Resulting interval = " + intvl);
                }
                else if (type.equalsIgnoreCase("e")) {
                    dataSumm = false;
                    wtu.setExternalLogging((Integer)args[1]);
                }
                else
                    out.println("Invalid logging type");
            }
            break;

        case CMD_INTSET:
            if ((found = CmndProc.scanArgs(scan, "I", args)) >= 0)
                wtu.setLoggingInterval((Integer)args[0]);
            break;

        case CMD_LOGSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                out.println("Logging interval = " + wtu.getLoggingInterval()
                              + ", logging state = " + wtu.getLoggingState());
            break;

        case CMD_PARMSET:
            if ((found = CmndProc.scanArgs(scan, "FIi", args)) >= 0) {
                boolean euro = (found & 0x04) != 0 && (Integer)args[2] != 0;
                wtu.setUserParameters((Float)args[0], (Integer)args[1], euro);
            }
            break;

        case CMD_PARMSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                out.format("Rate = %.3f, threshold = %s, euro = %s\n",
                           wtu.getUserRate(), wtu.getUserThreshold(),
                           wtu.getUserEuro());
            break;

        case CMD_VERSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                showStrings(wtu.getVersionData());
            break;

        case CMD_RESET:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                wtu.resetMemory();
            break;

        case CMD_RESTART:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                wtu.restart();
            break;

        case CMD_DATASHOW:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                dataSumm = (found & 0x01) != 0 && (Integer)args[0] != 0;
                dataCount = 0;
                int interval = wtu.getLoggedData();
                out.println("Record count = " + dataCount
                              + ", logging interval = " + interval);
                dataSumm = false;
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Routine to process logged data
    **
    ***************************************************************************
    */
    private static final byte[] flags = {0x7c, 0x2f, 0x2d, 0x5c};

    @Override
    public void process(float[] data)
    {
        dataCount++;

        if (dataSumm) {
            if ((dataCount & 0x07) == 0) {
                byte[] flag = {flags[(dataCount >> 3) & 3], 0x08};
                out.write(flag, 0, 2);
            }
        }
        else
            showFloats(data, fieldMask);
    }


   /**
    ***************************************************************************
    **
    **  Show an array of strings
    **
    ***************************************************************************
    */
    private void showStrings(String[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Show an array of integers
    **
    ***************************************************************************
    */
    private void showIntegers(int[] value)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Show an array of floats, selected by a mask
    **
    ***************************************************************************
    */
    private void showFloats(float[] value, int mask)
    {
        out.print("Values =");
        int posn = 8;
        for (int j = 0; j < value.length; j++) {
            if ((mask & (1 << j)) == 0) continue;
            String sValue = " " + value[j];
            posn += sValue.length();
            if (posn > 80) {
                out.print("\n        ");
                posn = 8 + sValue.length();
            }
            out.print(sValue);
        }
        out.println();
    }

}
