package org.lsst.ccs.drivers.usb;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the UsbLib methods
 *
 *  @author Owen Saxton
 */
public class TestUsbLib {

    /*
     *  Lookup tables
     */
    private final static Map<Integer, String> speedMap = new HashMap<>();
    static {
        speedMap.put(UsbLib.SPEED_UNKNOWN, "Unknown");
        speedMap.put(UsbLib.SPEED_LOW,     "Low");
        speedMap.put(UsbLib.SPEED_FULL,    "Full");
        speedMap.put(UsbLib.SPEED_HIGH,    "High");
        speedMap.put(UsbLib.SPEED_SUPER,   "Super");
    }

    private final static Map<Integer, String> devcMap = new HashMap<>();
    static {
        devcMap.put(UsbLib.CLASS_INTERFACE,   "See interface");
        devcMap.put(UsbLib.CLASS_AUDIO,       "Audio");
        devcMap.put(UsbLib.CLASS_COMM,        "Communications");
        devcMap.put(UsbLib.CLASS_HID,         "Human intface");
        devcMap.put(UsbLib.CLASS_PHYSICAL,    "Physical");
        devcMap.put(UsbLib.CLASS_IMAGE,       "Still image");
        devcMap.put(UsbLib.CLASS_PRINTER,     "Printer");
        devcMap.put(UsbLib.CLASS_HUB,         "Hub");
        devcMap.put(UsbLib.CLASS_MASS_STORE,  "Mass storage");
        devcMap.put(UsbLib.CLASS_DATA,        "Data");
        devcMap.put(UsbLib.CLASS_SMART_CARD,  "Smart card");
        devcMap.put(UsbLib.CLASS_CONTENT_SEC, "Content secty");
        devcMap.put(UsbLib.CLASS_VIDEO,       "Video");
        devcMap.put(UsbLib.CLASS_HEALTHCARE,  "Healthcare");
        devcMap.put(UsbLib.CLASS_DIAGNOSTIC,  "Diagnostic");
        devcMap.put(UsbLib.CLASS_WIRELESS,    "Wireless");
        devcMap.put(UsbLib.CLASS_MISC,        "Miscellaneous");
        devcMap.put(UsbLib.CLASS_APP_SPEC,    "App specific");
        devcMap.put(UsbLib.CLASS_VENDOR_SPEC, "Vendor spec");
    }

    private final static Map<Integer, String> cfgaMap = new HashMap<>();
    static {
        cfgaMap.put(UsbLib.CFG_ATTR_SELF_POWER,  "Self-powered");
        cfgaMap.put(UsbLib.CFG_ATTR_REMOTE_WAKE, "Remote wakeup");
    }

    private final static Map<Integer, String> epaMap = new HashMap<>();
    static {
        epaMap.put(UsbLib.EP_ATTR_CONTROL,     "Control");
        epaMap.put(UsbLib.EP_ATTR_ISOCHRONOUS, "Isochronous");
        epaMap.put(UsbLib.EP_ATTR_BULK,        "Bulk");
        epaMap.put(UsbLib.EP_ATTR_INTERRUPT,   "Interrupt");
    }

    /*
     *  Private fields
     */
    private final static PrintStream out = System.out;
    private UsbDevice devc;


    /**
     *  Connects to a USB device.
     *
     *  @param  vendId  The vendor ID, or -1 for all
     *  @param  devId   The device ID, or -1 for all
     *  @param  serial  The serial number, or empty for all
     *  @param  index   The index in the list of matching devices
     *  @throws  DriverException
     */
    @Command(name = "connect", description = "Connect to a USB device")
    public void connect(@Argument(description = "The vendor ID, or -1 for all")
                        int vendId,
                        @Argument(description = "The device ID, or -1 for all")
                        int devId,
                        @Argument(description = "The serial number, or empty for all")
                        String serial,
                        @Argument(description="Index in list of matching devices")
                        int index) throws DriverException
    {
        if (devc != null) {
            out.println("Device already connected");
        }
        else {
            devc = UsbLib.getDevice(vendId, devId,
                                    serial.isEmpty() ? null : serial, index);
            if (devc == null) {
                out.println("Device not found");
            }
        }
    }


    /**
     *  Connects to a USB device.
     *
     *  @param  index   The index in the list of all devices
     *  @throws  DriverException
     */
    @Command(name = "connect", description = "Connect to a USB device")
    public void connect(@Argument(description="Index in list of all devices")
                        int index) throws DriverException
    {
        connect(-1, -1, null, index);
    }


    /**
     *  Disconnects from a USB device.
     */
    @Command(name = "disconnect", description = "Disconnect from a USB device")
    public void disconnect()
    {
        if (!isConnected()) return;
        devc.close();
        devc = null;
    }


    /**
     *  Reads from the control endpoint.
     *
     *  @param  type     The request type
     *  @param  request  The request code
     *  @param  value    The value
     *  @param  index    The index
     *  @param  leng     The number of bytes to read
     *  @param  timeout  The timeout (ms)
     *  @throws  DriverException
     */
    @Command(name = "readcontrol", description = "Read from the control endpoint")
    public void readControl(@Argument(description="Request type")
                            int type,
                            @Argument(description="Request code")
                            int request,
                            @Argument(description="Value")
                            int value,
                            @Argument(description="Index")
                            int index,
                            @Argument(description="No. bytes to read")
                            int leng,
                            @Argument(description="Timeout (ms)")
                            int timeout) throws DriverException
    {
        if (!isConnected()) return;
        byte[] data = new byte[leng];
        int nRead = devc.controlRead(type, request, value, index, data, 0, leng, timeout);
        dispData(data, nRead);
    }


    /**
     *  Writes to the control endpoint.
     *
     *  @param  type     The request type
     *  @param  request  The request code
     *  @param  value    The value
     *  @param  index    The index
     *  @param  timeout  The timeout (ms)
     *  @param  data0    The first byte to write
     *  @param  data     The remaining bytes to write
     *  @throws  DriverException
     */
    @Command(name = "writecontrol", description = "Write to the control endpoint")
    public void writeControl(@Argument(description="Request type")
                             int type,
                             @Argument(description="Request code")
                             int request,
                             @Argument(description="Value")
                             int value,
                             @Argument(description="Index")
                             int index,
                             @Argument(description="Timeout (ms)")
                             int timeout,
                             @Argument(description="First byte of data to write")
                             int data0,
                             @Argument(description="The rest of the data to write")
                             int... data) throws DriverException
    {
        if (!isConnected()) return;
        byte[] wData = new byte[data.length + 1];
        wData[0] = (byte)data0;
        for (int j = 0; j < data.length; j++) {
            wData[j + 1] = (byte)data[j];
        }
        int nWrite = devc.controlWrite(type, request, value, index, wData, 0, wData.length, timeout);
        out.println(nWrite + " bytes written");
    }


    /**
     *  Reads bulk data.
     *
     *  @param  endp     The endpoint address
     *  @param  leng     The number of bytes to read
     *  @param  timeout  The timeout (ms)
     *  @throws  DriverException
     */
    @Command(name = "readbulk", description = "Read bulk data")
    public void readBulk(@Argument(description="Endpoint address")
                         int endp,
                         @Argument(description="No. bytes to read")
                         int leng,
                         @Argument(description="Timeout (ms)")
                         int timeout) throws DriverException
    {
        if (!isConnected()) return;
        byte[] data = new byte[leng];
        int nRead = devc.bulkRead(endp, data, 0, leng, timeout);
        dispData(data, nRead);
    }


    /**
     *  Writes bulk data.
     *
     *  @param  endp     The endpoint address
     *  @param  timeout  The timeout (ms)
     *  @param  data0    The first byte to write
     *  @param  data     The remaining bytes to write
     *  @throws  DriverException
     */
    @Command(name = "writebulk", description = "Write bulk data")
    public void writeBulk(@Argument(description="Endpoint address")
                          int endp,
                          @Argument(description="Timeout (ms)")
                          int timeout,
                          @Argument(description="First byte of data to write")
                          int data0,
                          @Argument(description="The rest of the data to write")
                          int... data) throws DriverException
    {
        if (!isConnected()) return;
        byte[] wData = new byte[data.length + 1];
        wData[0] = (byte)data0;
        for (int j = 0; j < data.length; j++) {
            wData[j + 1] = (byte)data[j];
        }
        int nWrite = devc.bulkWrite(endp, wData, 0, wData.length, timeout);
        out.println(nWrite + " bytes written");
    }


    /**
     *  Claims an interface.
     *
     *  @param  ifcnum  The interface number
     *  @param  force   Whether to force the claim
     *  @throws  DriverException
     */
    @Command(name = "claim", description = "Claim an interface")
    public void claim(@Argument(description="Interface number")
                      int ifcnum,
                      @Argument(description="Force the claim")
                      boolean force) throws DriverException
    {
        if (!isConnected()) return;
        devc.claimInterface(ifcnum, force);
    }


    /**
     *  Releases an interface.
     *
     *  @param  ifcnum  The interface number
     *  @throws  DriverException
     */
    @Command(name = "release", description = "Release an interface")
    public void release(@Argument(description="Interface number")
                        int ifcnum) throws DriverException
    {
        if (!isConnected()) return;
        devc.releaseInterface(ifcnum);
    }


    /**
     *  Sets the active configuration.
     *
     *  @param  cfgnum  The configuration number
     *  @throws  DriverException
     */
    @Command(name = "setconfig", description = "Sets the active configuration")
    public void setConfig(@Argument(description="Configuration number")
                          int cfgnum) throws DriverException
    {
        if (!isConnected()) return;
        devc.setActiveConfiguration(cfgnum);
    }


    /**
     *  Sets the alternate setting (interface).
     *
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @throws  DriverException
     */
    @Command(name = "setinterface", description = "Sets the interface")
    public void setInterface(@Argument(description="Interface number")
                             int ifcnum,
                             @Argument(description="Alternate setting number")
                             int altnum) throws DriverException
    {
        if (!isConnected()) return;
        devc.setAltSetting(ifcnum, altnum);
    }


    /**
     *  Shows all the devices on the bus.
     *
     *  @throws  DriverException
     */
    @Command(name = "showbus", description = "Show all the devices on the bus")
    public void showBus() throws DriverException
    {
        List<UsbDevice> devs = UsbLib.getDevices(-1, -1, null);
        if (!devs.isEmpty()) {
            for (int j = 0; j < devs.size(); j++) {
                UsbDevice dev = devs.get(j);
                if (j > 0) {
                    out.println();
                }
                out.println("Device " + j + ":");
                dispDevice(dev);
            }
        }
        else {
            out.println("No devices found");
        }
    }


    /**
     *  Shows an arbitrary device.
     *
     *  @param  devnum  The index in the list of all devices
     *  @throws  DriverException
     */
    @Command(name = "showdevice", description = "Show a device")
    public void showDevice(@Argument(description="Device number")
                           int devnum) throws DriverException
    {
        UsbDevice dev = getDevice(devnum);
        if (dev != null) {
            dispDevice(dev);
        }
    }


    /**
     *  Shows the connected device.
     *
     *  @throws  DriverException
     */
    @Command(name = "showdevice", description = "Show the connected device")
    public void showDevice() throws DriverException
    {
        showDevice(-1);
    }


    /**
     *  Shows a configuration in an arbitrary device.
     *
     *  @param  cfgnum  The configuration number
     *  @param  devnum  The index in the list of all devices
     *  @throws  DriverException
     */
    @Command(name = "showconfig", description = "Show a configuration")
    public void showConfig(@Argument(description="Configuration number")
                           int cfgnum,
                           @Argument(description="Device number")
                           int devnum) throws DriverException
    {
        UsbDevice dev = getDevice(devnum);
        if (dev == null) return;
        UsbConfiguration cfg = getConfiguration(dev, cfgnum);
        if (cfg == null) return;
        dispConfiguration(cfg);
    }


    /**
     *  Shows a configuration in the connected device.
     *
     *  @param  cfgnum  The configuration number
     *  @throws  DriverException
     */
    @Command(name = "showconfig", description = "Show a configuration")
    public void showConfig(@Argument(description="Configuration number")
                           int cfgnum) throws DriverException
    {
        showConfig(cfgnum, -1);
    }


    /**
     *  Shows the active configuration in the connected device.
     *
     *  @throws  DriverException
     */
    @Command(name = "showconfig", description = "Show the active configuration")
    public void showConfig() throws DriverException
    {
        showConfig(-1, -1);
    }


    /**
     *  Shows an arbitrary interface in an arbitrary device.
     *
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @param  cfgnum  The configuration number
     *  @param  devnum  The index in the list of all devices
     *  @throws  DriverException
     */
    @Command(name = "showinterface", description = "Show an interface")
    public void showIntfce(@Argument(description="Interface number")
                           int ifcnum,
                           @Argument(description="Alternate setting number")
                           int altnum,
                           @Argument(description="Configuration number")
                           int cfgnum,
                           @Argument(description="Device number")
                           int devnum) throws DriverException
    {
        UsbDevice dev = getDevice(devnum);
        if (dev == null) return;
        UsbConfiguration cfg = getConfiguration(dev, cfgnum);
        if (cfg == null) return;
        UsbInterface ifc = getInterface(cfg, ifcnum, altnum);
        if (ifc == null) return;
        dispInterface(ifc);
    }


    /**
     *  Shows an arbitrary interface in the connected device.
     *
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @param  cfgnum  The configuration number
     *  @throws  DriverException
     */
    @Command(name = "showinterface", description = "Show an interface")
    public void showIntfce(@Argument(description="Interface number")
                           int ifcnum,
                           @Argument(description="Alternate setting number")
                           int altnum,
                           @Argument(description="Configuration number")
                           int cfgnum) throws DriverException
    {
        showIntfce(ifcnum, altnum, cfgnum, -1);
    }


    /**
     *  Shows an interface in the active configuration of the connected device.
     *
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @throws  DriverException
     */
    @Command(name = "showinterface", description = "Show an interface")
    public void showIntfce(@Argument(description="Interface number")
                           int ifcnum,
                           @Argument(description="Alternate setting number")
                           int altnum) throws DriverException
    {
        showIntfce(ifcnum, altnum, -1, -1);
    }


    /**
     *  Shows the interface with the current setting in the active
     *  configuration of the connected device.
     *
     *  @param  ifcnum  The interface number
     *  @throws  DriverException
     */
    @Command(name = "showinterface", description = "Show an interface")
    public void showIntfce(@Argument(description="Interface number")
                           int ifcnum) throws DriverException
    {
        showIntfce(ifcnum, -1, -1, -1);
    }


    /**
     *  Shows an arbitrary endpoint in an arbitrary device.
     *
     *  @param  endp    The endpoint address
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @param  cfgnum  The configuration number
     *  @param  devnum  The index in the list of all devices
     *  @throws  DriverException
     */
    @Command(name = "showendpoint", description = "Show an endpoint")
    public void showEndpt(@Argument(description="Endpoint address")
                          int endp,
                          @Argument(description="Interface number")
                          int ifcnum,
                          @Argument(description="Alternate setting number")
                          int altnum,
                          @Argument(description="Configuration number")
                          int cfgnum,
                          @Argument(description="Device number")
                          int devnum) throws DriverException
    {
        UsbDevice dev = getDevice(devnum);
        if (dev == null) return;
        UsbConfiguration cfg = getConfiguration(dev, cfgnum);
        if (cfg == null) return;
        UsbInterface ifc = getInterface(cfg, ifcnum, altnum);
        if (ifc == null) return;
        UsbEndpoint ep = ifc.getEndpoint(endp);
        if (ep == null) {
            out.println("Endpoint not found");
            return;
        }
        dispEndpoint(ep);
    }


    /**
     *  Shows an arbitrary endpoint in the connected device.
     *
     *  @param  endp    The endpoint address
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @param  cfgnum  The configuration number
     *  @throws  DriverException
     */
    @Command(name = "showendpoint", description = "Show an endpoint")
    public void showEndpt(@Argument(description="Endpoint address")
                          int endp,
                          @Argument(description="Interface number")
                          int ifcnum,
                          @Argument(description="Alternate setting number")
                          int altnum,
                          @Argument(description="Configuration number")
                          int cfgnum) throws DriverException
    {
        showEndpt(endp, ifcnum, altnum, cfgnum, -1);
    }


    /**
     *  Shows an endpoint in the active configuration of the connected device.
     *
     *  @param  endp    The endpoint address
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @throws  DriverException
     */
    @Command(name = "showendpoint", description = "Show an endpoint")
    public void showEndpt(@Argument(description="Endpoint address")
                          int endp,
                          @Argument(description="Interface number")
                          int ifcnum,
                          @Argument(description="Alternate setting number")
                          int altnum) throws DriverException
    {
        showEndpt(endp, ifcnum, altnum, -1, -1);
    }


    /**
     *  Shows an endpoint in the current setting of the active configuration of
     *  the connected device.
     *
     *  @param  endp    The endpoint address
     *  @param  ifcnum  The interface number
     *  @throws  DriverException
     */
    @Command(name = "showendpoint", description = "Show an endpoint")
    public void showEndpt(@Argument(description="Endpoint address")
                          int endp,
                          @Argument(description="Interface number")
                          int ifcnum) throws DriverException
    {
        showEndpt(endp, ifcnum, -1, -1, -1);
    }


    /**
     *  Gets a device using its index in the system.
     */
    private UsbDevice getDevice(int index) throws DriverException
    {
        if (index < 0) {
            isConnected();
            return devc;
        }
        List<UsbDevice> devs = UsbLib.getDevices(-1, -1, null);
        if (index < devs.size()) {
            return devs.get(index);
        }
        else {
            out.println("Device not found");
            return null;
        }
    }


    /**
     *  Gets a configuration
     */
    private static UsbConfiguration getConfiguration(UsbDevice devc, int cfgnum) throws DriverException
    {
        UsbConfiguration cfg;
        if (cfgnum < 0) {
            cfg = devc.getActiveConfiguration();
            if (cfg == null) {
                out.println("No configuration active");
            }
        }
        else {
            cfg = devc.getConfigurationByValue(cfgnum);
            if (cfg == null) {
                out.println("Configuration not found");
            }
        }

        return cfg;
    }


    /**
     *  Gets an interface
     */
    private static UsbInterface getInterface(UsbConfiguration cfg, int ifcnum, int altnum) throws DriverException
    {
        UsbInterface ifc = cfg.getInterface(ifcnum);
        if (ifc != null) {
            if (altnum < 0) {
                ifc = ifc.getActiveSetting();
            }
            else {
                ifc = ifc.getSetting(altnum);
            }
        }
        if (ifc == null) {
            out.println("Interface not found");
        }

        return ifc;
    }


    /**
     *  Displays read data
     */
    private static void dispData(byte[] data, int count)
    {
        String cnt = count + " bytes read";
        String pad = String.format("%24s", "").substring(0, cnt.length() + 1);
        out.print(cnt);
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j == 0) {
                    out.print(":");
                }
                else {
                    out.println();
                    out.print(pad);
                }
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


    /**
     *  Generates write data from hexadecimal string arguments
     */
    private static byte[] genData(int found, Object[] args)
    {
        int count = 0, mask = found;
        List<byte[]> list = new ArrayList<>();
        for (int j = 0; found != 0; found >>= 1, j++) {
            if ((found & 1) == 0) continue;
            String hex = (String)args[j];
            int leng = hex.length();
            if ((leng & 1) != 0) break;
            byte[] sData = new byte[leng / 2];
            try {
                for (int k = 0; k < leng; k += 2) {
                    String num = hex.substring(k, k + 2);
                    sData[k / 2] = (byte)Integer.parseInt(num, 16);
                }
            }
            catch (NumberFormatException e) {
                break;
            }
            count += leng / 2;
            list.add(sData);
            mask ^= (1 << j);
        }
        if (mask != 0) {
            out.println("Invalid hexadecimal string");
            return null;
        }
        byte[] data = new byte[count];
        int l = 0;
        for (byte[] sData : list) {
            for (byte datum : sData) {
                data[l++] = datum;
            }
        }

        return data;
    }


    /**
     *  Displays device information
     */
    private static void dispDevice(UsbDevice dev) throws DriverException
    {
        out.format("Bus number          : %03d\n", dev.getBusNumber());
        out.format("Device address      : %03d\n", dev.getAddress());
        UsbDeviceDescriptor desc = dev.getDescriptor();
        out.format("Vendor ID (name)    : 0x%04x (%s)\n", desc.idVendor(), dev.getManufacturer());
        out.format("Product ID (name)   : 0x%04x (%s)\n", desc.idProduct(), dev.getProduct());
        out.println("Serial number       : " + dev.getSerialNumber());
        out.println("Device speed        : " + speedMap.get(dev.getSpeed()));
        int cls = desc.bDeviceClass();
        out.format("Device class        : %s (%s)\n", cls, devcMap.get(cls));
        out.println("Device subclass     : " + desc.bDeviceSubClass());
        out.println("Device protocol     : " + desc.bDeviceProtocol());
        out.format("USB release         : %04x\n", desc.bcdUSB());
        out.println("Number of configs   : " + desc.bNumConfigurations());
        UsbConfiguration cfg = dev.getActiveConfiguration();
        out.println("Active config number: " + cfg.getDescriptor().bConfigurationValue());
        out.println("Maxm ctl packet size: " + desc.bMaxPacketSize0());
    }


    /**
     *  Displays configuration information.
     */
    private static void dispConfiguration(UsbConfiguration cfg)
    {
        UsbConfigurationDescriptor desc = cfg.getDescriptor();
        out.format("ConfigNum (name) : %s (%s)\n",
                   desc.bConfigurationValue(), cfg.getString());
        int attr = desc.bmAttributes();
        StringBuilder sAttr = new StringBuilder();
        for (int j = 0, mask = 1; j < 8; j++, mask <<= 1) {
            if ((attr & mask) != 0) {
                if (sAttr.length() != 0) {
                    sAttr.append(", ");
                }
                String name = cfgaMap.get(mask);
                sAttr.append(name != null ? name : "??");
            }
        }
        out.format("Attributes       : 0x%02x (%s)\n", attr, sAttr);
        out.format("Maximum power    : %s mA\n", 2 * desc.bMaxPower());
        out.println("Number interfaces: " + desc.bNumInterfaces());
    }


    /**
     *  Displays interface information.
     */
    private static void dispInterface(UsbInterface ifc)
    {
        UsbInterfaceDescriptor desc = ifc.getDescriptor();
        out.format("Interface no. (name): %s (%s)\n",
                   desc.bInterfaceNumber(), ifc.getString());
        out.println("Setting number      : " + desc.bAlternateSetting());
        out.println("Active setting no.  : " + ifc.getActiveSettingNumber());
        out.println("Number of settings  : " + ifc.getNumSettings());
        int cls = desc.bInterfaceClass();
        out.format("Interface class     : %s (%s)\n", cls, devcMap.get(cls));
        out.println("Interface subclass  : " + desc.bInterfaceSubClass());
        out.println("Interface protocol  : " + desc.bInterfaceProtocol());
        out.println("Number of endpoints : " + desc.bNumEndpoints());
        out.print("Endpoint addressess :");
        List<UsbEndpoint> eps = ifc.getEndpoints();
        for (UsbEndpoint ep : eps) {
            UsbEndpointDescriptor epDesc = ep.getDescriptor();
            out.format(" 0x%02x", epDesc.bEndpointAddress());
        }
        out.println();
    }


    /**
     *  Displays endpoint information.
     */
    private static void dispEndpoint(UsbEndpoint ep)
    {
        UsbEndpointDescriptor desc = ep.getDescriptor();
        out.format("Address          : 0x%02x\n", desc.bEndpointAddress());
        int attr = desc.bmAttributes();
        out.format("Attributes       : 0x%02x (%s)\n",
                   attr, epaMap.get(attr & 0x03));
        out.format("Interval         : %s msec\n", desc.bInterval());
        out.println("Maxm packet size : " + desc.wMaxPacketSize());
    }


    /**
     *  Checks whether there is a device connected.
     */
    private boolean isConnected()
    {
        if (devc == null) {
            out.println("No device connected");
            return false;
        }
        return true;
    }

}
