package org.lsst.ccs.drivers.usb;

/**
 *  The USB device descriptor.
 *
 *  @author Owen Saxton
 */
public class UsbDeviceDescriptor extends UsbDescriptor {

    private final short bcdUSB;
    private final byte bDeviceClass;
    private final byte bDeviceSubClass;
    private final byte bDeviceProtocol;
    private final byte bMaxPacketSize0;
    private final short idVendor;
    private final short idProduct;
    private final short bcdDevice;
    private final byte iManufacturer;
    private final byte iProduct;
    private final byte iSerialNumber;
    private final byte bNumConfigurations;


   /**
     *  Constructor.
     *
     *  @param  bLength             Descriptor length
     *  @param  bDescriptorType     Descriptor type
     *  @param  bcdUSB              USB release number (BCD encoded)
     *  @param  bDeviceClass        Device class
     *  @param  bDeviceSubClass     Device subclass
     *  @param  bDeviceProtocol     Device protocol
     *  @param  bMaxPacketSize0     Maximum packet size for control endpoint
     *  @param  idVendor            Vendor ID
     *  @param  idProduct           Product ID
     *  @param  bcdDevice           Device release number (BCD encoded)
     *  @param  iManufacturer       Manufacturer string descriptor index
     *  @param  iProduct            Product string descriptor index
     *  @param  iSerialNumber       Serial number string descriptor index
     *  @param  bNumConfigurations  Number of configurations
     */
    public UsbDeviceDescriptor(byte bLength, byte bDescriptorType,
                               short bcdUSB, byte bDeviceClass,
                               byte bDeviceSubClass, byte bDeviceProtocol,
                               byte bMaxPacketSize0, short idVendor,
                               short idProduct, short bcdDevice,
                               byte iManufacturer, byte iProduct,
                               byte iSerialNumber, byte bNumConfigurations)
    {
        super(bLength, bDescriptorType);
        this.bcdUSB = bcdUSB;
        this.bDeviceClass = bDeviceClass;
        this.bDeviceSubClass = bDeviceSubClass;
        this.bDeviceProtocol = bDeviceProtocol;
        this.bMaxPacketSize0 = bMaxPacketSize0;
        this.idVendor = idVendor;
        this.idProduct = idProduct;
        this.bcdDevice = bcdDevice;
        this.iManufacturer = iManufacturer;
        this.iProduct = iProduct;
        this.iSerialNumber = iSerialNumber;
        this.bNumConfigurations = bNumConfigurations;
    }


   /**
     *  Gets the supported USB version.
     *
     *  @return  The supported USB version
     */
    public int bcdUSB()
    {
        return bcdUSB & 0xffff;
    }


   /**
     *  Gets the device class.
     *
     *  @return  The device class
     */
    public int bDeviceClass()
    {
        return bDeviceClass & 0xff;
    }


   /**
     *  Gets the device subclass.
     *
     *  @return  The device subclass
     */
    public int bDeviceSubClass()
    {
        return bDeviceSubClass & 0xff;
    }


   /**
     *  Gets the device protocol.
     *
     *  @return  The device protocol
     */
    public int bDeviceProtocol()
    {
        return bDeviceProtocol & 0xff;
    }


   /**
     *  Gets the maximum packet size for the control endpoint.
     *
     *  @return  The maximum packet size
     */
    public int bMaxPacketSize0()
    {
        return bMaxPacketSize0 & 0xff;
    }


   /**
     *  Gets the vendor ID.
     *
     *  @return  The vendor ID
     */
    public int idVendor()
    {
        return idVendor & 0xffff;
    }


   /**
     *  Gets the product ID.
     *
     *  @return  The product ID
     */
    public int idProduct()
    {
        return idProduct & 0xffff;
    }


   /**
     *  Gets the device type.
     *
     *  @return  The device type
     */
    public int bcdDevice()
    {
        return bcdDevice & 0xffff;
    }


   /**
     *  Gets the manufacturer name index.
     *
     *  @return  The manufacturer name index
     */
    public int iManufacturer()
    {
        return iManufacturer & 0xff;
    }


   /**
     *  Gets the product name index.
     *
     *  @return  The product name index
     */
    public int iProduct()
    {
        return iProduct & 0xff;
    }


   /**
     *  Gets the serial number index.
     *
     *  @return  The serial number index
     */
    public int iSerialNumber()
    {
        return iSerialNumber & 0xff;
    }


   /**
     *  Gets the number of configurations.
     *
     *  @return  The number of configurations
     */
    public int bNumConfigurations()
    {
        return bNumConfigurations & 0xff;
    }

}
