package org.lsst.ccs.drivers.usb;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Operations for a USB device
 *
 *  @author Owen Saxton
 */
public class UsbDevice {
    
    private final long devc;       // libusb device
    private long devh;             // libusb device handle
    private final UsbDeviceDescriptor desc;
    private final Map<Integer, Integer> altMap = new HashMap<>();


    /**
     *  Constructor.
     */
    UsbDevice(long devc, long devh, UsbDeviceDescriptor desc)
        throws DriverException
    {
        this.devc = devc;
        this.devh = devh;
        this.desc = desc;
    }


    /**
     *  Gets the device descriptor.
     *
     *  @return  The device descriptor
     */
    public UsbDeviceDescriptor getDescriptor()
    {
        return desc;
    }


    /**
     *  Gets the bus number.
     *
     *  @return  The bus number
     */
    public int getBusNumber()
    {
        return UsbLib.getBusNumber(devc);
    }


    /**
     *  Gets the port number.
     *
     *  @return  The port number
     */
    public int getPortNumber()
    {
        return UsbLib.getPortNumber(devc);
    }


    /**
     *  Gets the device address.
     *
     *  @return  The device address
     */
    public int getAddress()
    {
        return UsbLib.getDeviceAddress(devc);
    }


    /**
     *  Gets the negotiated device speed.
     *
     *  @return  The device's speed
     */
    public int getSpeed()
    {
        return UsbLib.getDeviceSpeed(devc);
    }


    /**
     *  Gets the list of available configurations.
     *
     *  @return  The list of configurations
     *  @throws  DriverException
     */
    public List<UsbConfiguration> getConfigurations() throws DriverException
    {
        List<UsbConfiguration> cfgs = new ArrayList<>();
        for (int j = 0; j < desc.bNumConfigurations(); j++) {
            long cfgd = UsbLib.getConfigDescriptor(devc, j);
            cfgs.add(UsbLib.buildConfiguration(cfgd, this));
            UsbLib.freeConfigDescriptor(cfgd);
        }

        return cfgs;
    }


    /**
     *  Gets the active configuration.
     *
     *  @return  The active configuration
     *  @throws  DriverException
     */
    public UsbConfiguration getActiveConfiguration() throws DriverException
    {
        long cfgd = UsbLib.getActiveConfigDescriptor(devc);
        if (cfgd == 0) return null;
        UsbConfiguration cfg = UsbLib.buildConfiguration(cfgd, this);
        UsbLib.freeConfigDescriptor(cfgd);

        return cfg;
    }


    /**
     *  Gets a configuration by its index.
     *
     *  @param  index  The index of the configuration
     *  @return  The configuration
     *  @throws  DriverException
     */
    public UsbConfiguration getConfiguration(int index) throws DriverException
    {
        long cfgd = UsbLib.getConfigDescriptor(devc, index);
        if (cfgd == 0) return null;
        UsbConfiguration cfg = UsbLib.buildConfiguration(cfgd, this);
        UsbLib.freeConfigDescriptor(cfgd);

        return cfg;
    }


    /**
     *  Gets a configuration by its value.
     *
     *  @param  value  The configuration value
     *  @return  The configuration
     *  @throws  DriverException
     */
    public UsbConfiguration getConfigurationByValue(int value) throws DriverException
    {
        long cfgd = UsbLib.getConfigDescriptorByValue(devc, value);
        if (cfgd == 0) return null;
        UsbConfiguration cfg = UsbLib.buildConfiguration(cfgd, this);
        UsbLib.freeConfigDescriptor(cfgd);

        return cfg;
    }


    /**
     *  Closes the device.
     */
    public void close()
    {
        UsbLib.closeDev(devh);
    }


    /**
     *  Gets the active configuration number.
     *
     *  @return  The active configuration number
     *  @throws  DriverException
     */
    public int getActiveConfigurationNumber() throws DriverException
    {
        return UsbLib.getConfiguration(getLibusbHandle());
    }


    /**
     *  Sets the active configuration (by number).
     *
     *  @param  cfgnum  The configuration number
     *  @throws  DriverException
     */
    public void setActiveConfiguration(int cfgnum) throws DriverException
    {
        UsbLib.setConfiguration(getLibusbHandle(), cfgnum);
    }


    /**
     *  Claims an interface.
     *
     *  @param  ifcnum  The interface number
     *  @param  force   Whether to force the claim
     *  @throws  DriverException
     */
    public void claimInterface(int ifcnum, boolean force) throws DriverException
    {
        if (force) {
            detachDriver(ifcnum);
        }
        UsbLib.claimInterface(getLibusbHandle(), ifcnum);
    }


    /**
     *  Releases an interface.
     *
     *  @param  ifcnum  The interface number
     *  @throws  DriverException
     */
    public void releaseInterface(int ifcnum) throws DriverException
    {
        UsbLib.releaseInterface(getLibusbHandle(), ifcnum);
    }


    /**
     *  Detaches a kernel driver from an interface.
     *
     *  @param  ifcnum  The interface number
     *  @throws  DriverException
     */
    public void detachDriver(int ifcnum) throws DriverException
    {
        UsbLib.detachKernelDriver(getLibusbHandle(), ifcnum);
    }


    /**
     *  Re-attaches a kernel driver to an interface.
     *
     *  @param  ifcnum  The interface number
     *  @throws  DriverException
     */
    public void attachDriver(int ifcnum) throws DriverException
    {
        UsbLib.attachKernelDriver(getLibusbHandle(), ifcnum);
    }


    /**
     *  Sets the auto-detach kernel driver option.
     *
     *  @param  enable  Whether to enable the auto-detach option
     *  @throws  DriverException
     */
    public void setAutoDetachDriver(boolean enable) throws DriverException
    {
        UsbLib.setAutoDetachKernelDriver(getLibusbHandle(), enable);
    }


    /**
     *  Selects an alternate setting.
     *
     *  @param  ifcnum  The interface number
     *  @param  altnum  The alternate setting number
     *  @throws  DriverException
     */
    public void setAltSetting(int ifcnum, int altnum) throws DriverException
    {
        UsbLib.setInterfaceAltSetting(getLibusbHandle(), ifcnum, altnum);
        altMap.put(ifcnum, altnum);
    }


    /**
     *  Gets an alternate setting.
     *
     *  @param  ifcnum  The interface number
     *  @return  The alternate setting number, or 0 if the interface number doesn't exist
     */
    public int getAltSetting(int ifcnum)
    {
        Integer altnum = altMap.get(ifcnum);
        return altnum == null ? 0 : altnum;
    }


    /**
     *  Gets a string (descriptor).
     *
     *  @param  index  The index of the string
     *  @return  The string, or null if the index is invalid
     */
    public String getString(int index)
    {
        try {
            return UsbLib.getStringDescriptorAscii(getLibusbHandle(), index);
        }
        catch (DriverException e) {
            return null;
        }
    }


    /**
     *  Gets the manufacturer string.
     *
     *  @return  The manufacturer string
     */
    public String getManufacturer()
    {
        return getString(desc.iManufacturer());
    }


    /**
     *  Gets the product string.
     *
     *  @return  The product string
     */
    public String getProduct()
    {
        return getString(desc.iProduct());
    }


    /**
     *  Gets the serial number string.
     *
     *  @return  The serial number string
     */
    public String getSerialNumber()
    {
        return getString(desc.iSerialNumber());
    }


    /**
     *  Resets the device.
     *
     *  @throws  DriverException
     */
    public void reset() throws DriverException
    {
        UsbLib.resetDevice(getLibusbHandle());
    }


    /**
     *  Reads from a control endpoint.
     *
     *  @param  type     The transfer type
     *  @param  rqst     The request parameter
     *  @param  value    The value parameter
     *  @param  index    The index parameter
     *  @param  data     The array to which the data is read
     *  @param  offset   The offset to the start of the read data
     *  @param  leng     The number of bytes to read
     *  @param  timeout  The timeout (ms)
     *  @return  The number of bytes read
     *  @throws  DriverException
     */
    public int controlRead(int type, int rqst, int value, int index, byte[] data, int offset,
                           int leng, int timeout) throws DriverException
    {
        if ((type & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        return UsbLib.controlTransfer(getLibusbHandle(), type, rqst, value,
                                      index, data, offset, leng, timeout);
    }


    /**
     *  Writes to a control endpoint.
     *
     *  @param  type     The transfer type
     *  @param  rqst     The request parameter
     *  @param  value    The value parameter
     *  @param  index    The index parameter
     *  @param  data     The array from which the data is written
     *  @param  offset   The offset to the start of the data
     *  @param  leng     The number of bytes to write
     *  @param  timeout  The timeout (ms)
     *  @return  The number of bytes written
     *  @throws  DriverException
     */
    public int controlWrite(int type, int rqst, int value, int index, byte[] data, int offset,
                            int leng, int timeout) throws DriverException
    {
        if ((type & UsbLib.EP_DIRECTION_IN) != 0) {
            throw new DriverException("usb error: write request on read e/p");
        }
        return UsbLib.controlTransfer(getLibusbHandle(), type, rqst, value,
                                      index, data, offset, leng, timeout);
    }


    /**
     *  Reads bulk data from an endpoint.
     *
     *  @param  endp     The endpoint address.
     *  @param  data     The array to which the data is read
     *  @param  offset   The offset to the start of the read data
     *  @param  leng     The number of bytes to read
     *  @param  timeout  The timeout (ms)
     *  @return  The number of bytes read
     *  @throws  DriverException
     */
    public int bulkRead(int endp, byte[] data, int offset, int leng, int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        return UsbLib.bulkTransfer(getLibusbHandle(), endp, data, offset, leng,
                                   timeout);
    }


    /**
     *  Flushes bulk data from an endpoint.
     *
     *  @param  endp     The endpoint address.
     *  @throws  DriverException
     */
    public void bulkFlush(int endp) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        int leng = 64;
        byte[] buff = new byte[leng];
        for (int nread = leng; nread == leng;) {
            nread = UsbLib.bulkTransfer(getLibusbHandle(), endp, buff, 0, leng, 10);
        }
    }


    /**
     *  Writes bulk data to an endpoint.
     *
     *  @param  endp     The endpoint address.
     *  @param  data     The array from which the data is written
     *  @param  offset   The offset to the start of the written data
     *  @param  leng     The number of bytes to write
     *  @param  timeout  The timeout (ms)
     *  @return  The number of bytes written
     *  @throws  DriverException
     */
    public int bulkWrite(int endp, byte[] data, int offset, int leng, int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) != 0) {
            throw new DriverException("usb error: write request on read e/p");
        }
        return UsbLib.bulkTransfer(getLibusbHandle(), endp, data, offset, leng,
                                   timeout);
    }


    /**
     *  Reads interrupt data from an endpoint.
     *
     *  @param  endp     The endpoint address.
     *  @param  data     The array to which the data is read
     *  @param  offset   The offset to the start of the read data
     *  @param  leng     The number of bytes to read
     *  @param  timeout  The timeout (ms)
     *  @return  The number of bytes read
     *  @throws  DriverException
     */
    public int interruptRead(int endp, byte[] data, int offset, int leng, int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        return UsbLib.interruptTransfer(getLibusbHandle(), endp, data, offset,
                                        leng, timeout);
    }


    /**
     *  Writes interrupt data to an endpoint.
     *
     *  @param  endp     The endpoint address.
     *  @param  data     The array from which the data is written
     *  @param  offset   The offset to the start of the written data
     *  @param  leng     The number of bytes to write
     *  @param  timeout  The timeout (ms)
     *  @return  The number of bytes written
     *  @throws  DriverException
     */
    public int interruptWrite(int endp, byte[] data, int offset, int leng, int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) != 0) {
            throw new DriverException("usb error: write request on read e/p");
        }
        return UsbLib.interruptTransfer(getLibusbHandle(), endp, data, offset,
                                        leng, timeout);
    }


    /**
     *  Gets the libusb device.
     */
    long getLibusbDevice()
    {
        return devc;
    }


    /**
     *  Gets the libusb device handle.
     */
    long getLibusbHandle() throws DriverException
    {
        if (devh == 0) {
            devh = UsbLib.openDev(devc);
        }
        return devh;
    }

}
