package org.lsst.ccs.drivers.usb;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Operations for a USB configuration
 *
 *  @author Owen Saxton
 */
public class UsbConfiguration {
    
    private final UsbConfigurationDescriptor desc;
    private final List<List<UsbInterface>> ifcs;
    private final UsbDevice devc;


    /**
     *  Constructor.
     */
    UsbConfiguration(UsbConfigurationDescriptor desc, List<List<UsbInterface>> ifcs, UsbDevice devc)
    {
        this.desc = desc;
        this.ifcs = ifcs;
        this.devc = devc;
    }


    /**
     *  Gets the configuration descriptor.
     *
     *  @return  The configuration descriptor
     */
    public UsbConfigurationDescriptor getDescriptor()
    {
        return desc;
    }


    /**
     *  Gets the configuration string.
     *
     *  @return  The configuration string
     */
    public String getString()
    {
        return devc.getString(desc.iConfiguration());
    }


    /**
     *  Gets whether configuration contains an interface.
     *
     *  @param  iface  The interface number
     *  @return  Whether the configuration contains the interface
     */
    public boolean containsInterface(int iface)
    {
        for (List<UsbInterface> ifc : ifcs) {
            if (iface == ifc.get(0).getDescriptor().bInterfaceNumber()) {
                return true;
            }
        }

        return false;
    }


    /**
     *  Gets the parent device.
     *
     *  @return  The parent device
     */
    public UsbDevice getDevice()
    {
        return devc;
    }


    /**
     *  Gets the specified interface.
     *
     *  @param  iface  The interface number
     *  @return  The interface object
     */
    public UsbInterface getInterface(int iface)
    {
        for (List<UsbInterface> ifcl : ifcs) {
            UsbInterface ifc = ifcl.get(0);
            if (iface == ifc.getDescriptor().bInterfaceNumber()) {
                return ifc;
            }
        }
        return null;
    }


    /**
     *  Gets the list of available interfaces.
     *
     *  @return  The list of interfaces
     */
    public List<UsbInterface> getInterfaces()
    {
        List<UsbInterface> ifca = new ArrayList<>();
        for (List<UsbInterface> ifcl : ifcs) {
            ifca.add(ifcl.get(0));
        }
        return ifca;
    }


    /**
     *  Gets whether configuration is active.
     *
     *  @return  Whether the configuration is active
     *  @throws  DriverException
     */
    public boolean isActive() throws DriverException
    {
        return desc.bConfigurationValue() == devc.getActiveConfigurationNumber();
    }

}
