package org.lsst.ccs.drivers.usb;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Routines to interface to libusb
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class UsbLib {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    /** Device speed - unknown */
    public static final int SPEED_UNKNOWN     = 0;

    /** Device speed - low */
    public static final int SPEED_LOW         = 1;

    /** Device speed - full */
    public static final int SPEED_FULL        = 2;

    /** Device speed - high */
    public static final int SPEED_HIGH        = 3;

    /** Device speed - super */
    public static final int SPEED_SUPER       = 4;

    
    /** Device class - see interface */
    public static final int CLASS_INTERFACE   = 0x00;

    /** Device class - audio */
    public static final int CLASS_AUDIO       = 0x01;

    /** Device class - CDC control */
    public static final int CLASS_COMM        = 0x02;

    /** Device class - human interface */
    public static final int CLASS_HID         = 0x03;

    /** Device class - physical */
    public static final int CLASS_PHYSICAL    = 0x05;

    /** Device class - still image */
    public static final int CLASS_IMAGE       = 0x06;

    /** Device class - printer */
    public static final int CLASS_PRINTER     = 0x07;

    /** Device class - mass storage */
    public static final int CLASS_MASS_STORE  = 0x08;

    /** Device class - hub */
    public static final int CLASS_HUB         = 0x09;

    /** Device class - CDC data */
    public static final int CLASS_DATA        = 0x0a;

    /** Device class - smart card */
    public static final int CLASS_SMART_CARD  = 0x0b;

    /** Device class security x */
    public static final int CLASS_CONTENT_SEC = 0x0d;

    /** Device class - video */
    public static final int CLASS_VIDEO       = 0x0e;

    /** Device class - personal healthcare */
    public static final int CLASS_HEALTHCARE  = 0x0f;

    /** Device class - diagnostic */
    public static final int CLASS_DIAGNOSTIC  = 0xdc;

    /** Device class - wireless */
    public static final int CLASS_WIRELESS    = 0xe0;

    /** Device class - miscellaneous */
    public static final int CLASS_MISC        = 0xef;

    /** Device class - application specific */
    public static final int CLASS_APP_SPEC    = 0xfe;

    /** Device class - vendor specific */
    public static final int CLASS_VENDOR_SPEC = 0xff;

    
    /** Configuration attribute - self-powered */
    public static final int CFG_ATTR_SELF_POWER   = 0x40;

    /** Configuration attribute - remote wakeup */
    public static final int CFG_ATTR_REMOTE_WAKE  = 0x20;

    
    /** Endpoint attribute - control */
    public static final int EP_ATTR_CONTROL       = 0x00;

    /** Endpoint attribute - isochronous */
    public static final int EP_ATTR_ISOCHRONOUS   = 0x01;

    /** Endpoint attribute - bulk data */
    public static final int EP_ATTR_BULK          = 0x02;

    /** Endpoint attribute - interrupt */
    public static final int EP_ATTR_INTERRUPT     = 0x03;

    /** Endpoint attribute - bulk stream */
    public static final int EP_ATTR_BULK_STREAM   = 0x04;


    /** Endpoint direction - in */
    public static final int EP_DIRECTION_IN  = 0x80;

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private static long context;
    private final static Set<Long>
        openDevs = Collections.synchronizedSet(new HashSet<Long>());
    private final static CloseThread closeAll = new CloseThread();

   /**
    ***************************************************************************
    **
    **  Static initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("UsbLib");
        Runtime.getRuntime().addShutdownHook(closeAll);
    }


   /**
    ***************************************************************************
    **
    **  Shutdown hook thread to close all open devices.
    **
    ***************************************************************************
    */
    private static class CloseThread extends Thread {

        @Override
        public void run()
        {
            for (Long devh : openDevs) {
                /*
                try {
                    resetDevice(devh);
                }
                catch (DriverException e) {
                }
                */
                close(devh);
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Gets the list of devices matching vendor, product and serial.
    **
    **  If a non-null serial number is specified and an otherwise matching
    **  device cannot be opened or has no serial number, it is treated as not
    **  matching.
    **
    **  @param  vendor   The vendor ID; or -1 to match all
    **
    **  @param  product  The product ID; or -1 to match all
    **
    **  @param  serial   The serial number, which matches if it is contained
    **                   in the device's number; or null to match all
    **
    **  @return  The list of matching device objects
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public static List<UsbDevice> getDevices(int vendor, int product,
                                             String serial)
        throws DriverException
    {
        List<UsbDevice> devices = new ArrayList<>();
        long[] devs = getDeviceList(getContext());
        for (long dev : devs) {
            boolean match = false;
            long devh = 0;
            UsbDeviceDescriptor desc = newDeviceDescriptor(dev);
            if ((vendor == -1 || vendor == desc.idVendor())
                  && (product == -1 || product == desc.idProduct())) {
                match = true;
                if (serial != null) {
                    try {
                        devh = openDev(dev);
                        String
                          sn = getStringDescriptorAscii(devh,
                                                        desc.iSerialNumber());
                        match = sn.contains(serial);
                    }
                    catch(DriverException e) {
                        match = false;
                    }
                }
            }
            if (match) {
                devices.add(new UsbDevice(dev, devh, desc));
            }
            else {
                if (devh != 0) {
                    closeDev(devh);
                }
                unrefDevice(dev);
            }
        }
        
        return devices;
    }


   /**
    ***************************************************************************
    **
    **  Gets the device matching vendor, product, serial and index.
    **
    **  If a non-null serial number is specified and an otherwise matching
    **  device cannot be opened or has no serial number, it is treated as not
    **  matching.
    **
    **  @param  vendor   The vendor ID, or -1 to match all
    **
    **  @param  product  The product ID, or -1 to match all
    **
    **  @param  serial   The serial number, which matches if it is contained
    **                   in the device's number; or null to match all
    **
    **  @param  index    The index of the device in the list of matching
    **                   devices
    **
    **  @return  The matching device object, or null if no match
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public static UsbDevice getDevice(int vendor, int product, String serial,
                                      int index)
        throws DriverException
    {
        UsbDevice devc = null;
        long[] devs = getDeviceList(getContext());
        for (long dev : devs) {
            boolean match = false;
            long devh = 0;
            UsbDeviceDescriptor desc = newDeviceDescriptor(dev);
            if (index >= 0 && (vendor == -1 || vendor == desc.idVendor())
                  && (product == -1 || product == desc.idProduct())) {
                match = true;
                if (serial != null) {
                    try {
                        devh = openDev(dev);
                        String
                          sn = getStringDescriptorAscii(devh,
                                                        desc.iSerialNumber());
                        match = sn.contains(serial);
                    }
                    catch(DriverException e) {
                        match = false;
                    }
                }
            }
            if (match && index-- == 0) {
                devc = new UsbDevice(dev, devh, desc);
            }
            else {
                if (devh != 0) {
                    closeDev(devh);
                }
                unrefDevice(dev);
            }
        }

        return devc;
    }


   /**
    ***************************************************************************
    **
    **  Opens a device for subsequent I/O operations.
    **
    **  @param  dev  The device handle
    **
    **  @return  The device's open handle
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    static long openDev(long dev) throws DriverException
    {
        long devh = open(dev);
        openDevs.add(devh);

        return devh;
    }


   /**
    ***************************************************************************
    **
    **  Closes a device.
    **
    **  @param  devh  The device's open handle
    **
    ***************************************************************************
    */
    static void closeDev(long devh)
    {
        openDevs.remove(devh);
        close(devh);
    }


   /**
    ***************************************************************************
    **
    **  Builds a complete configuration from its descriptor.
    **
    **  @param  cdesc  The configuration descriptor handle
    **
    **  @param  devc   The associated device object
    **
    **  @return  The configuration object
    **
    ***************************************************************************
    */
    public static UsbConfiguration buildConfiguration(long cdesc,
                                                      UsbDevice devc)
    {
        long[] ifchs = getInterfaces(cdesc);
        List<List<UsbInterface>> ifcs = new ArrayList<>();
        for (long ifch : ifchs) {
            long[] alths = getAltSettingDescriptors(ifch);
            List<UsbInterface> alts = new ArrayList<>();
            for (long alth : alths) {
                long[] ephs = getEndpointDescriptors(alth);
                List<UsbEndpoint> eps = new ArrayList<>();
                for (long eph : ephs) {
                    eps.add(new UsbEndpoint(newEndpointDescriptor(eph), devc));
                }
                alts.add(new UsbInterface(newInterfaceDescriptor(alth), alts,
                                          eps, devc));
            }
            ifcs.add(alts);
        }
        return new UsbConfiguration(newConfigurationDescriptor(cdesc), ifcs,
                                    devc);
    }


   /**
    ***************************************************************************
    **
    **  Gets the context.
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    static long getContext() throws DriverException
    {
        if (context == 0) {
            context = init();
        }

        return context;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the USB library.
    **
    **  @return  The library context
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    native static long init() throws DriverException;


   /**
    ***************************************************************************
    **
    **  Deinitializes the USB library.
    **
    **  @param  ctxt  The library context
    **
    ***************************************************************************
    */
    native static void exit(long ctxt);


   /**
    ***************************************************************************
    **
    **  Gets the list of available USB devices.
    **
    **  @param  ctxt  The library context
    **
    **  @return  The array of handles for each USB device
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static long[] getDeviceList(long ctxt) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Gets the bus number for a device.
    **
    **  @param  devc  The device handle
    **
    **  @return  The device's bus number
    **
    ***************************************************************************
    */
    public native static int getBusNumber(long devc);


   /**
    ***************************************************************************
    **
    **  Gets the port number for a device.
    **
    **  @param  devc  The device handle
    **
    **  @return  The device's port number
    **
    ***************************************************************************
    */
    public native static int getPortNumber(long devc);


   /**
    ***************************************************************************
    **
    **  Gets the address of a device.
    **
    **  @param  devc  The device handle
    **
    **  @return  The device's address
    **
    ***************************************************************************
    */
    public native static int getDeviceAddress(long devc);


   /**
    ***************************************************************************
    **
    **  Gets the speed of a device.
    **
    **  @param  devc  The device handle
    **
    **  @return  The device's speed
    **
    ***************************************************************************
    */
    public native static int getDeviceSpeed(long devc);


   /**
    ***************************************************************************
    **
    **  Gets the maximum packet size for an endpoint.
    **
    **  @param  devc  The device handle
    **
    **  @param  endp  The endpoint address
    **
    **  @return  The device's maximum packet size
    **
    ***************************************************************************
    */
    public native static int getMaxPacketSize(long devc, int endp);


   /**
    ***************************************************************************
    **
    **  Unreferences a device.
    **
    **  @param  devc  The device handle
    **
    ***************************************************************************
    */
    public native static void unrefDevice(long devc);


   /**
    ***************************************************************************
    **
    **  Opens a device for subsequent I/O operations.
    **
    **  @param  devc  The device handle
    **
    **  @return  The device's open handle
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static long open(long devc) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Closes a device.
    **
    **  @param  devh  The open handle
    **
    ***************************************************************************
    */
    public native static void close(long devh);


   /**
    ***************************************************************************
    **
    **  Gets the number of the active configuration.
    **
    **  @param  devh  The open handle
    **
    **  @return  The active configuration number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static int getConfiguration(long devh) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Sets the active configuration.
    **
    **  @param  devh  The open handle
    **
    **  @param  conf  The number of the configuration to make active
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void setConfiguration(long devh, int conf)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Claims an interface.
    **
    **  @param  devh   The open handle
    **
    **  @param  intfc  The number of the interface to claim
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void claimInterface(long devh, int intfc)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Releases an interface.
    **
    **  @param  devh   The open handle
    **
    **  @param  intfc  The number of the interface to release
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void releaseInterface(long devh, int intfc)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Sets an alternate setting for an interface.
    **
    **  @param  devh   The open handle
    **
    **  @param  intfc  The interface number
    **
    **  @param  alt    The alternate setting number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void
        setInterfaceAltSetting(long devh, int intfc, int alt)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Determines if kernel driver is active on an interface.
    **
    **  @param  devh   The open handle
    **
    **  @param  intfc  The interface number
    **
    **  @return  Whether a kernel driver is avtive
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static boolean kernelDriverActive(long devh, int intfc)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Detach kernel driver from an interface.
    **
    **  @param  devh   The open handle
    **
    **  @param  intfc  The interface number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void detachKernelDriver(long devh, int intfc)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Re-attach kernel driver to an interface.
    **
    **  @param  devh   The open handle
    **
    **  @param  intfc  The interface number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void attachKernelDriver(long devh, int intfc)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Enables/disables the auto detach kernel driver capability.
    **
    **  @param  devh  The open handle
    **
    **  @param  enable  Whether to enable (true) or disable (false)
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void setAutoDetachKernelDriver(long devh,
                                                        boolean enable)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Clears a halt condition on an endpoint.
    **
    **  @param  devh  The open handle
    **
    **  @param  endp  The endpoint address
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void clearHalt(long devh, int endp)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Resets a device.
    **
    **  @param  devh  The open handle
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static void resetDevice(long devh) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Gets the device descriptor.
    **
    **  @param  devc  The device handle
    **
    **  @return  The device descriptor object
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static UsbDeviceDescriptor
        newDeviceDescriptor(long devc) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Gets the active configuration descriptor handle.
    **
    **  The memory associated with the handle must be released after it is no
    **  longer needed by calling freeConfigDescriptor.
    **
    **  @param  devc  The device handle
    **
    **  @return  The configuration descriptor handle
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static long
        getActiveConfigDescriptor(long devc) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Gets a configuration descriptor handle.
    **
    **  The memory associated with the handle must be released after it is no
    **  longer needed by calling freeConfigDescriptor.
    **
    **  @param  devc   The device handle
    **
    **  @param  index  The index of the descriptor
    **
    **  @return  The configuration descriptor handle
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static long
        getConfigDescriptor(long devc, int index) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Gets a configuration descriptor handle using its value.
    **
    **  The memory associated with the handle must be released after it is no
    **  longer needed by calling freeConfigDescriptor.
    **
    **  @param  devc   The device handle
    **
    **  @param  value  The descriptor value (number)
    **
    **  @return  The configuration descriptor handle
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static long
        getConfigDescriptorByValue(long devc, int value) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Frees a configuration descriptor.
    **
    **  @param  cdesc  The configuration descriptor handle
    **
    ***************************************************************************
    */
    public native static void freeConfigDescriptor(long cdesc);


   /**
    ***************************************************************************
    **
    **  Creates a configuration descriptor object.
    **
    **  @param  cdesc  The configuration descriptor handle
    **
    **  @return  The configuration descriptor object
    **
    ***************************************************************************
    */
    public native static UsbConfigurationDescriptor
        newConfigurationDescriptor(long cdesc);


   /**
    ***************************************************************************
    **
    **  Gets the interfaces in a configuration.
    **
    **  @param  cdesc  The configuration descriptor handle
    **
    **  @return  The array of interface handles
    **
    ***************************************************************************
    */
    public native static long[] getInterfaces(long cdesc);


   /**
    ***************************************************************************
    **
    **  Gets the alternate settings in an interface.
    **
    **  @param  iface  The interface handle
    **
    **  @return  The array of alternate setting interface descriptor handles
    **
    ***************************************************************************
    */
    public native static long[] getAltSettingDescriptors(long iface);


   /**
    ***************************************************************************
    **
    **  Creates an interface descriptor object.
    **
    **  @param  idesc  The interface descriptor handle
    **
    **  @return  The interface descriptor object
    **
    ***************************************************************************
    */
    public native static UsbInterfaceDescriptor
        newInterfaceDescriptor(long idesc);


   /**
    ***************************************************************************
    **
    **  Gets the endpoints in an interface.
    **
    **  @param  idesc  The interface descriptor handle
    **
    **  @return  The array of endpoint descriptor handles
    **
    ***************************************************************************
    */
    public native static long[] getEndpointDescriptors(long idesc);


   /**
    ***************************************************************************
    **
    **  Creates an endpoint descriptor object.
    **
    **  @param  edesc  The endpoint descriptor handle
    **
    **  @return  The endpoint descriptor object
    **
    ***************************************************************************
    */
    public native static UsbEndpointDescriptor
        newEndpointDescriptor(long edesc);


   /**
    ***************************************************************************
    **
    **  Gets a string descriptor.
    **
    **  @param  devh   The open handle
    **
    **  @param  index  The index of the string descriptor
    **
    **  @return  The string descriptor as an ASCII string
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static String getStringDescriptorAscii(long devh, int index)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Transfers data on the control endpoint.
    **
    **  @param  devh     The open handle
    **
    **  @param  type     The transfer type
    **
    **  @param  rqst     The request parameter
    **
    **  @param  value    The value parameter
    **
    **  @param  index    The index parameter
    **
    **  @param  data     The array to or from which the data is transterred
    **
    **  @param  offset   The offset to the start of the transferred data
    **
    **  @param  leng     The number of bytes to transfer
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes transferred
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static int
        controlTransfer(long devh, int type, int rqst, int value, int index,
                        byte[] data, int offset, int leng, int timeout)
        throws DriverException;


   /**
    ***************************************************************************
    **
    **  Transfers bulk data on an endpoint.
    **
    **  @param  devh     The open handle
    **
    **  @param  endp     The endpoint address.
    **
    **  @param  data     The array to or from which the data is transterred
    **
    **  @param  offset   The offset to the start of the transferred data
    **
    **  @param  leng     The number of bytes to transfer
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes transferred
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static int
        bulkTransfer(long devh, int endp, byte[] data, int offset, int leng,
                     int timeout) throws DriverException;


   /**
    ***************************************************************************
    **
    **  Transfers interrupt data on an endpoint.
    **
    **  @param  devh     The open handle
    **
    **  @param  endp     The endpoint address.
    **
    **  @param  data     The array to or from which the data is transterred
    **
    **  @param  offset   The offset to the start of the transferred data
    **
    **  @param  leng     The number of bytes to transfer
    **
    **  @param  timeout  The timeout (ms)
    **
    **  @return  The number of bytes transferred
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public native static int
        interruptTransfer(long devh, int endp, byte[] data, int offset,
                          int leng, int timeout) throws DriverException;

}
