package org.lsst.ccs.drivers.usb;

import java.util.List;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Operations for a USB interface
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class UsbInterface {
    
    private final UsbInterfaceDescriptor desc;
    private final List<UsbInterface> alts;
    private final List<UsbEndpoint> eps;
    private final UsbDevice devc;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    UsbInterface(UsbInterfaceDescriptor desc, List<UsbInterface> alts,
                 List<UsbEndpoint> eps, UsbDevice devc)
    {
        this.desc = desc;
        this.alts = alts;
        this.eps = eps;
        this.devc = devc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the interface descriptor.
    **
    ***************************************************************************
    */
    public UsbInterfaceDescriptor getDescriptor()
    {
        return desc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the interface string.
    **
    ***************************************************************************
    */
    public String getString()
    {
        return devc.getString(desc.iInterface());
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of alternate settings.
    **
    ***************************************************************************
    */
    public int getNumSettings()
    {
        return alts.size();
    }


   /**
    ***************************************************************************
    **
    **  Gets the list of alternate settings.
    **
    ***************************************************************************
    */
    public List<UsbInterface> getSettings()
    {
        return alts;
    }


   /**
    ***************************************************************************
    **
    **  Gets an alternate setting.
    **
    ***************************************************************************
    */
    public UsbInterface getSetting(int altNum)
    {
        for (UsbInterface alt : alts) {
            if (altNum == alt.getDescriptor().bAlternateSetting()) {
                return alt;
            }
        }
        return null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the active setting number.
    **
    ***************************************************************************
    */
    public int getActiveSettingNumber()
    {
        return devc.getAltSetting(desc.bInterfaceNumber());
    }


   /**
    ***************************************************************************
    **
    **  Gets the active setting.
    **
    ***************************************************************************
    */
    public UsbInterface getActiveSetting()
    {
        int altnum = devc.getAltSetting(desc.bInterfaceNumber());
        for (UsbInterface alt : alts) {
            if (alt.desc.bAlternateSetting() == altnum) return alt;
        }
        return null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the list of endpoints.
    **
    ***************************************************************************
    */
    public List<UsbEndpoint> getEndpoints()
    {
        return eps;
    }


   /**
    ***************************************************************************
    **
    **  Gets an endpoint.
    **
    ***************************************************************************
    */
    public UsbEndpoint getEndpoint(int epAddr)
    {
        for (UsbEndpoint ep : eps) {
            if (epAddr == ep.getDescriptor().bEndpointAddress()) {
                return ep;
            }
        }
        return null;
    }

}
