package org.lsst.ccs.drivers.usb;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ******************************************************************************
 **
 **  Operations for a USB device
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class UsbDevice {
    
    private final long devc;       // libusb device
    private long devh;             // libusb device handle
    private final UsbDeviceDescriptor desc;
    private Map<Integer, Integer> altMap = new HashMap<>();


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    UsbDevice(long devc, long devh, UsbDeviceDescriptor desc)
        throws DriverException
    {
        this.devc = devc;
        this.devh = devh;
        this.desc = desc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the device descriptor.
    **
    ***************************************************************************
    */
    public UsbDeviceDescriptor getDescriptor()
    {
        return desc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the bus number.
    **
    ***************************************************************************
    */
    public int getBusNumber()
    {
        return UsbLib.getBusNumber(devc);
    }


   /**
    ***************************************************************************
    **
    **  Gets the port number.
    **
    ***************************************************************************
    */
    public int getPortNumber()
    {
        return UsbLib.getPortNumber(devc);
    }


   /**
    ***************************************************************************
    **
    **  Gets the device address.
    **
    ***************************************************************************
    */
    public int getAddress()
    {
        return UsbLib.getDeviceAddress(devc);
    }


   /**
    ***************************************************************************
    **
    **  Gets the negotiated device speed.
    **
    ***************************************************************************
    */
    public int getSpeed()
    {
        return UsbLib.getDeviceSpeed(devc);
    }


   /**
    ***************************************************************************
    **
    **  Gets the list of available configurations.
    **
    ***************************************************************************
    */
    public List<UsbConfiguration> getConfigurations() throws DriverException
    {
        List<UsbConfiguration> cfgs = new ArrayList<>();
        for (int j = 0; j < desc.bNumConfigurations(); j++) {
            long cfgd = UsbLib.getConfigDescriptor(devc, j);
            cfgs.add(UsbLib.buildConfiguration(cfgd, this));
            UsbLib.freeConfigDescriptor(cfgd);
        }

        return cfgs;
    }


   /**
    ***************************************************************************
    **
    **  Gets the active configuration.
    **
    ***************************************************************************
    */
    public UsbConfiguration getActiveConfiguration() throws DriverException
    {
        long cfgd = UsbLib.getActiveConfigDescriptor(devc);
        if (cfgd == 0) return null;
        UsbConfiguration cfg = UsbLib.buildConfiguration(cfgd, this);
        UsbLib.freeConfigDescriptor(cfgd);

        return cfg;
    }


   /**
    ***************************************************************************
    **
    **  Gets a configuration by its index.
    **
    ***************************************************************************
    */
    public UsbConfiguration getConfiguration(int index)
             throws DriverException
    {
        long cfgd = UsbLib.getConfigDescriptor(devc, index);
        if (cfgd == 0) return null;
        UsbConfiguration cfg = UsbLib.buildConfiguration(cfgd, this);
        UsbLib.freeConfigDescriptor(cfgd);

        return cfg;
    }


   /**
    ***************************************************************************
    **
    **  Gets a configuration by its value.
    **
    ***************************************************************************
    */
    public UsbConfiguration getConfigurationByValue(int value)
             throws DriverException
    {
        long cfgd = UsbLib.getConfigDescriptorByValue(devc, value);
        if (cfgd == 0) return null;
        UsbConfiguration cfg = UsbLib.buildConfiguration(cfgd, this);
        UsbLib.freeConfigDescriptor(cfgd);

        return cfg;
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    ***************************************************************************
    */
    public void close()
    {
        UsbLib.closeDev(devh);
    }


   /**
    ***************************************************************************
    **
    **  Gets the active configuration number.
    **
    ***************************************************************************
    */
    public int getActiveConfigurationNumber() throws DriverException
    {
        return UsbLib.getConfiguration(getLibusbHandle());
    }


   /**
    ***************************************************************************
    **
    **  Sets the active configuration (by number).
    **
    ***************************************************************************
    */
    public void setActiveConfiguration(int cfgnum) throws DriverException
    {
        UsbLib.setConfiguration(getLibusbHandle(), cfgnum);
    }


   /**
    ***************************************************************************
    **
    **  Claims an interface.
    **
    ***************************************************************************
    */
    public void claimInterface(int ifcnum, boolean force)
        throws DriverException
    {
        if (force) {
            detachDriver(ifcnum);
        }
        UsbLib.claimInterface(getLibusbHandle(), ifcnum);
    }


   /**
    ***************************************************************************
    **
    **  Releases an interface.
    **
    ***************************************************************************
    */
    public void releaseInterface(int ifcnum) throws DriverException
    {
        UsbLib.releaseInterface(getLibusbHandle(), ifcnum);
    }


   /**
    ***************************************************************************
    **
    **  Detaches a kernel driver from an interface.
    **
    ***************************************************************************
    */
    public void detachDriver(int ifcnum) throws DriverException
    {
        UsbLib.detachKernelDriver(getLibusbHandle(), ifcnum);
    }


   /**
    ***************************************************************************
    **
    **  Re-attaches a kernel driver to an interface.
    **
    ***************************************************************************
    */
    public void attachDriver(int ifcnum) throws DriverException
    {
        UsbLib.attachKernelDriver(getLibusbHandle(), ifcnum);
    }


   /**
    ***************************************************************************
    **
    **  Sets the auto-detach kernel driver option.
    **
    ***************************************************************************
    */
    public void setAutoDetachDriver(boolean enable) throws DriverException
    {
        UsbLib.setAutoDetachKernelDriver(getLibusbHandle(), enable);
    }


   /**
    ***************************************************************************
    **
    **  Selects an alternate setting.
    **
    ***************************************************************************
    */
    public void setAltSetting(int ifcnum, int altnum) throws DriverException
    {
        UsbLib.setInterfaceAltSetting(getLibusbHandle(), ifcnum, altnum);
        altMap.put(ifcnum, altnum);
    }


   /**
    ***************************************************************************
    **
    **  Gets an alternate setting.
    **
    ***************************************************************************
    */
    public int getAltSetting(int ifcnum)
    {
        Integer altnum = altMap.get(ifcnum);
        return altnum == null ? 0 : altnum;
    }


   /**
    ***************************************************************************
    **
    **  Gets a string (descriptor).
    **
    ***************************************************************************
    */
    public String getString(int index)
    {
        try {
            return UsbLib.getStringDescriptorAscii(getLibusbHandle(), index);
        }
        catch (DriverException e) {
            return null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the manufacturer string.
    **
    ***************************************************************************
    */
    public String getManufacturer()
    {
        return getString(desc.iManufacturer());
    }


   /**
    ***************************************************************************
    **
    **  Gets the product string.
    **
    ***************************************************************************
    */
    public String getProduct()
    {
        return getString(desc.iProduct());
    }


   /**
    ***************************************************************************
    **
    **  Gets the serial number string.
    **
    ***************************************************************************
    */
    public String getSerialNumber()
    {
        return getString(desc.iSerialNumber());
    }


   /**
    ***************************************************************************
    **
    **  Resets the device.
    **
    ***************************************************************************
    */
    public void reset() throws DriverException
    {
        UsbLib.resetDevice(getLibusbHandle());
    }


   /**
    ***************************************************************************
    **
    **  Reads from a control endpoint.
    **
    ***************************************************************************
    */
    public int controlRead(int type, int rqst, int value, int index,
                           byte[] data, int offset, int leng, int timeout)
                 throws DriverException
    {
        if ((type & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        return UsbLib.controlTransfer(getLibusbHandle(), type, rqst, value,
                                      index, data, offset, leng, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a control endpoint.
    **
    ***************************************************************************
    */
    public int controlWrite(int type, int rqst, int value, int index,
                            byte[] data, int offset, int leng, int timeout)
                 throws DriverException
    {
        if ((type & UsbLib.EP_DIRECTION_IN) != 0) {
            throw new DriverException("usb error: write request on read e/p");
        }
        return UsbLib.controlTransfer(getLibusbHandle(), type, rqst, value,
                                      index, data, offset, leng, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads bulk data from an endpoint.
    **
    ***************************************************************************
    */
    public int bulkRead(int endp, byte[] data, int offset, int leng,
                        int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        return UsbLib.bulkTransfer(getLibusbHandle(), endp, data, offset, leng,
                                   timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes bulk data to an endpoint.
    **
    ***************************************************************************
    */
    public int bulkWrite(int endp, byte[] data, int offset, int leng,
                         int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) != 0) {
            throw new DriverException("usb error: write request on read e/p");
        }
        return UsbLib.bulkTransfer(getLibusbHandle(), endp, data, offset, leng,
                                   timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads interrupt data from an endpoint.
    **
    ***************************************************************************
    */
    public int interruptRead(int endp, byte[] data, int offset, int leng,
                             int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) == 0) {
            throw new DriverException("usb error: read request on write e/p");
        }
        return UsbLib.interruptTransfer(getLibusbHandle(), endp, data, offset,
                                        leng, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Writes interrupt data to an endpoint.
    **
    ***************************************************************************
    */
    public int interruptWrite(int endp, byte[] data, int offset, int leng,
                              int timeout) throws DriverException
    {
        if ((endp & UsbLib.EP_DIRECTION_IN) != 0) {
            throw new DriverException("usb error: write request on read e/p");
        }
        return UsbLib.interruptTransfer(getLibusbHandle(), endp, data, offset,
                                        leng, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Gets the libusb device.
    **
    ***************************************************************************
    */
    long getLibusbDevice()
    {
        return devc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the libusb device handle.
    **
    ***************************************************************************
    */
    long getLibusbHandle() throws DriverException
    {
        if (devh == 0) {
            devh = UsbLib.openDev(devc);
        }
        return devh;
    }

}
