package org.lsst.ccs.drivers.usb;

import java.util.ArrayList;
import java.util.List;
import javax.usb.UsbConfiguration;
import javax.usb.UsbConfigurationDescriptor;
import javax.usb.UsbDevice;
import javax.usb.UsbDeviceDescriptor;
import javax.usb.UsbEndpoint;
import javax.usb.UsbEndpointDescriptor;
import javax.usb.UsbException;
import javax.usb.UsbInterface;
import javax.usb.UsbInterfaceDescriptor;
import jline.console.ConsoleReader;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test the UsbComm methods
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestUsbCommNew implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_FIND        = 0,
        CMD_OPEN        = 1,
        CMD_CLOSE       = 2,
        CMD_FLUSH       = 3,
        CMD_READ        = 4,
        CMD_WRITE       = 5,
        CMD_SHOWDEV     = 6,
        CMD_SHOWCFG     = 7,
        CMD_SHOWIF      = 8,
        CMD_SHOWEP      = 9,
        CMD_SHOWBUS     = 10,
        CMD_SHOWOPEN    = 11,
        CMD_RCNTRL      = 12,
        CMD_WCNTRL      = 13,
        CMD_SETCFG      = 14,
        CMD_SETIF       = 15,
        CMD_CLAIM       = 16,
        CMD_RELEASE     = 17,
        NUM_CMDS        = 18;

    /*
    **  Command help text
    */
    private final static String[] helpFind = {
        "Find a device",
        "find <vendId> <devId> [<serial>]",
        "vendId  The vendor ID of the device",
        "devId   The device ID of the device",
        "serial  The serial number of the device (default any)",
    };

    private final static String[] helpOpen = {
        "Open a pipe to a device endpoint",
        "open <iface> <epaddr>",
        "iface   The number of the interface to use",
        "epaddr  The address of the endpoint to open",
    };

    private final static String[] helpClose = {
        "Close a pipe",
        "close <pipe>",
        "pipe  The number of the pipe (reported at open time) to close",
    };

    private final static String[] helpFlush = {
        "Flush any unread data from an input pipe",
        "flush <pipe>",
        "pipe  The number of the pipe (reported at open time) to flush",
    };

    private final static String[] helpRead = {
        "Read data from an input pipe and display it",
        "read <pipe> <count> [<timeout>]",
        "pipe     The number of the pipe (reported at open time) to read from",
        "count    The number of bytes to read",
        "timeout  The timeout in seconds (default 1)",
    };

    private final static String[] helpWrite = {
        "Write data to an output pipe",
        "write <pipe> [<data1>]... [<datan>]",
        "pipe   The number of the pipe (reported at open time) to write to",
        "datan  Up to 8 hexadecimal strings",
    };

    private final static String[] helpRcntrl = {
        "Read data from the control endpoint and display it",
        "rcntrl  <type> <rqst> <value> <index> <count> [<timeout>]",
        "type     The request type",
        "rqst     The request",
        "value    The associated value",
        "index    The associated index",
        "count    The number of bytes to read",
        "timeout  The timeout in seconds (default 1)",
    };

    private final static String[] helpWcntrl = {
        "Write data to the control endpoint",
        "wcntrl <type> <rqst> <value> <index> [<data1>]... [<datan>]",
        "type   The request type",
        "rqst   The request",
        "value  The associated value",
        "index  The associated index",
        "datan  Up to 8 hexadecimal strings",
    };

    private final static String[] helpClaim = {
        "Claim an interface",
        "claim <iface> [<force>]",
        "iface  The interface number",
        "force  If present and non-zero, force the claim",
    };

    private final static String[] helpRelease = {
        "Release an interface",
        "release <iface>",
        "iface  The interface number",
    };

    private final static String[] helpSetcfg = {
        "Set the current configuration",
        "setcfg <number>",
        "number  The number of the configuration to set",
    };

    private final static String[] helpSetif = {
        "Set the alternate setting for an interface",
        "setif <iface> <altnum>",
        "iface   The interface number",
        "altnum  The alternate setting number",
    };

    private final static String[] helpShowopen = {
        "Show all the open pipes",
        "showopen",
    };

    private final static String[] helpShowbus = {
        "Show all devices on the USB bus",
        "showbus",
    };

    private final static String[] helpShowdev = {
        "Show device information",
        "showdev [<devnum>]",
        "devnum  Device number from showbus list (default: current device)",
    };

    private final static String[] helpShowcfg = {
        "Show configuration information",
        "showcfg [<cfgnum>] [<devnum>]",
        "cfgnum  Configuration number (default: active configuration)",
        "devnum  Device number from showbus list (default: current device)",
    };

    private final static String[] helpShowif = {
        "Show interface information",
        "showif <iface> [<altnum>] [<cfgnum>] [<devnum>]",
        "iface   The interface number", 
        "altnum  The alternate setting number (default: active one)",
        "cfgnum  Configuration number (default: active configuration)",
        "devnum  Device number from showbus list (default: current device)",
    };

    private final static String[] helpShowep = {
        "Show endpoint information",
        "showep <epaddr> <iface> [<altnum>] [<cfgnum>] [<devnum>]",
        "epaddr  The endpoint address",
        "iface   The interface number", 
        "altnum  The alternate setting number (default: active one)",
        "cfgnum  Configuration number (default: active configuration)",
        "devnum  Device number from showbus list (default: current device)",
    };

    /*
    **  Command table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("find",        CMD_FIND,        helpFind,     "IIs");
        cmnd.add("open",        CMD_OPEN,        helpOpen,     "II");
        cmnd.add("close",       CMD_CLOSE,       helpClose,    "I");
        cmnd.add("flush",       CMD_FLUSH,       helpFlush,    "I");
        cmnd.add("read",        CMD_READ,        helpRead,     "IIf");
        cmnd.add("write",       CMD_WRITE,       helpWrite,    "Issssssss");
        cmnd.add("rcntrl",      CMD_RCNTRL,      helpRcntrl,   "IIIIIf");
        cmnd.add("wcntrl",      CMD_WCNTRL,      helpWcntrl,   "IIIIssssssss");
        cmnd.add("claim",       CMD_CLAIM,       helpClaim,    "Ii");
        cmnd.add("release",     CMD_RELEASE,     helpRelease,  "I");
        cmnd.add("setcfg",      CMD_SETCFG,      helpSetcfg,   "I");
        cmnd.add("setif",       CMD_SETIF,       helpSetif,    "II");
        cmnd.add("showdev",     CMD_SHOWDEV,     helpShowdev,  "i");
        cmnd.add("showcfg",     CMD_SHOWCFG,     helpShowcfg,  "ii");
        cmnd.add("showif",      CMD_SHOWIF,      helpShowif,   "Iiii");
        cmnd.add("showep",      CMD_SHOWEP,      helpShowep,   "IIiii");
        cmnd.add("showbus",     CMD_SHOWBUS,     helpShowbus,  "");
        cmnd.add("showopen",    CMD_SHOWOPEN,    helpShowopen, "");
    }

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup speedNames;
    static {
        speedNames = new CmndProcess.Lookup(4);
        speedNames.add("Undefined",    UsbComm.DEV_SPEED_UNDEFINED);
        speedNames.add("Low",          UsbComm.DEV_SPEED_LOW);
        speedNames.add("Full",         UsbComm.DEV_SPEED_FULL);
        speedNames.add("Unknown",      UsbComm.DEV_SPEED_UNKNOWN);
    }
    private final static CmndProcess.Lookup devcNames;
    static {
        devcNames = new CmndProcess.Lookup(6);
        devcNames.add("See interface", UsbComm.DEV_CLASS_INTERFACE);
        devcNames.add("Comm control",  UsbComm.DEV_CLASS_CDC_CTRL);
        devcNames.add("Hub",           UsbComm.DEV_CLASS_HUB);
        devcNames.add("Diagnostic",    UsbComm.DEV_CLASS_DIAGNOSTIC);
        devcNames.add("Miscellaneous", UsbComm.DEV_CLASS_MISC);
        devcNames.add("Vendor spec",   UsbComm.DEV_CLASS_VENDOR_SPEC);
    }
    private final static CmndProcess.Lookup cfgaNames;
    static {
        cfgaNames = new CmndProcess.Lookup(2);
        cfgaNames.add("Self-powered",  UsbComm.CFG_ATTR_SELF_POWER);
        cfgaNames.add("Remote wakeup", UsbComm.CFG_ATTR_REMOTE_WAKE);
    }
    private final static CmndProcess.Lookup ifccNames;
    static {
        ifccNames = new CmndProcess.Lookup(17);
        ifccNames.add("Audio",         UsbComm.IFC_CLASS_AUDIO);
        ifccNames.add("Comm control",  UsbComm.IFC_CLASS_CDC_CTRL);
        ifccNames.add("Human intface", UsbComm.IFC_CLASS_HID);
        ifccNames.add("Physical",      UsbComm.IFC_CLASS_PHYSICAL);
        ifccNames.add("Still image",   UsbComm.IFC_CLASS_IMAGE);
        ifccNames.add("Printer",       UsbComm.IFC_CLASS_PRINTER);
        ifccNames.add("Mass storage",  UsbComm.IFC_CLASS_MASS_STORE);
        ifccNames.add("Comm data",     UsbComm.IFC_CLASS_CDC_DATA);
        ifccNames.add("Smart card",    UsbComm.IFC_CLASS_SMART_CARD);
        ifccNames.add("Content secty", UsbComm.IFC_CLASS_CONTENT_SEC);
        ifccNames.add("Video",         UsbComm.IFC_CLASS_VIDEO);
        ifccNames.add("Healthcare",    UsbComm.IFC_CLASS_HEALTHCARE);
        ifccNames.add("Diagnostic",    UsbComm.IFC_CLASS_DIAGNOSTIC);
        ifccNames.add("Wireless",      UsbComm.IFC_CLASS_WIRELESS);
        ifccNames.add("Miscellaneous", UsbComm.IFC_CLASS_MISC);
        ifccNames.add("App specific",  UsbComm.IFC_CLASS_APP_SPEC);
        ifccNames.add("Vendor spec",   UsbComm.IFC_CLASS_VENDOR_SPEC);
    }
    private final static CmndProcess.Lookup epaNames;
    static {
        epaNames = new CmndProcess.Lookup(4);
        epaNames.add("Control",        UsbComm.EP_ATTR_CONTROL);
        epaNames.add("Isochronous",    UsbComm.EP_ATTR_ISOCHRONOUS);
        epaNames.add("Bulk",           UsbComm.EP_ATTR_BULK);
        epaNames.add("Interrupt",      UsbComm.EP_ATTR_INTERRUPT);
    }

    /*
    **  Private fields
    */
    private final Output out = new ConsOut();
    private final CmndProcess proc = new CmndProcess();
    private final boolean debug;
    private UsbComm com;
    private ArrayList<Integer> openEps = new ArrayList<Integer>();
    private ArrayList<UsbCommPipe> openPipes = new ArrayList<UsbCommPipe>();


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public TestUsbCommNew(boolean debug)
    {
        this.debug = debug;
    }

    public TestUsbCommNew()
    {
        this(false);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestUsbCommNew(args.length > 0)).run();
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            com = new UsbComm();

            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null) break;
                if (!proc.process(line)) break;
            }
        }
        catch (Exception e) {
            out.println(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_FIND:
                procFind(found, args); break;
            case CMD_RCNTRL:
                procRcntrl(found, args); break;
            case CMD_WCNTRL:
                procWcntrl(found, args); break;
            case CMD_OPEN:
                procOpen(found, args); break;
            case CMD_CLOSE:
                procClose(found, args); break;
            case CMD_FLUSH:
                procFlush(found, args); break;
            case CMD_READ:
                procRead(found, args); break;
            case CMD_WRITE:
                procWrite(found, args); break;
            case CMD_CLAIM:
                procClaim(found, args); break;
            case CMD_RELEASE:
                procRelease(found, args); break;
            case CMD_SETCFG:
                procSetCfg(found, args); break;
            case CMD_SETIF:
                procSetIf(found, args); break;
            case CMD_SHOWOPEN:
                procShowOpen(found, args); break;
            case CMD_SHOWBUS:
                procShowBus(found, args); break;
            case CMD_SHOWDEV:
                procShowDev(found, args); break;
            case CMD_SHOWCFG:
                procShowCfg(found, args); break;
            case CMD_SHOWIF:
                procShowIf(found, args); break;
            case CMD_SHOWEP:
                procShowEp(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (UsbException e) {
            out.println(e.toString());
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the FIND command
    **
    ***************************************************************************
    */
    @Command(name = "find", description = "Find USB a device")
    public void findDevice(@Argument(name = "vendId", description = "The vendor ID of the device") int vendId,
            @Argument(name = "devId", description = "The device ID of the device") int devId,
            @Argument(name = "serial", description = "The serial number of the device (default any)", defaultValue=Argument.NULL) String serial) throws UsbException {
        com.findDevice(devId, devId, serial);
    }

    private void procFind(int found, Object[] args) throws UsbException
    {
        String serial = ((found & 0x04) != 0) ? (String)args[2] : null;
        if (com.findDevice((Integer)args[0], (Integer)args[1], serial) == null)
            out.println("Device not found");
    }


   /**
    ***************************************************************************
    **
    **  Processes the RCNTRL command
    **
    ***************************************************************************
    */
    private void procRcntrl(int found, Object[] args) throws UsbException
    {
        float timeout = ((found & 0x20) != 0) ? (Float)args[5] : 1F;
        byte[] data = new byte[(Integer)args[4]];
        int nRead = com.read((Integer)args[0], (Integer)args[1],
                             (Integer)args[2], (Integer)args[3],
                             data, (int)(1000f * timeout));
        dispData(data, nRead);
    }


   /**
    ***************************************************************************
    **
    **  Processes the WCNTRL command
    **
    ***************************************************************************
    */
    private void procWcntrl(int found, Object[] args) throws UsbException
    {
        byte[] data = genData(found & 0xff0, args);
        if (data == null) return;
        int nWrite = com.write((Integer)args[0], (Integer)args[1],
                               (Integer)args[2], (Integer)args[3], data);
        out.println(nWrite + " bytes written");
    }


   /**
    ***************************************************************************
    **
    **  Processes the OPEN command
    **
    ***************************************************************************
    */
    @Command(name = "open", description = "Open a pipe to a device endpoint")
    public void openDevice(@Argument(name = "iface", description = "The number of the interface to use") int iface,
            @Argument(name = "epaddr", description = "The address of the endpoint to open") int epaddr
            ) throws UsbException {
        procOpen(0, new Object[]{new Integer(iface), new Integer(epaddr)});
    }    

    private void procOpen(int found, Object[] args) throws UsbException
    {
        UsbCommPipe pipe = com.open((Integer)args[0], (Integer)args[1]);
        int pnum = openEps.size();
        for (int j = 0; j < pnum; j++) {
            int ep = openEps.get(j);
            if (ep < 0) {
                pnum = j;
                break;
            }
        }
        if (pnum < openEps.size()) {
            openEps.set(pnum, (Integer)args[1]);
            openPipes.set(pnum, pipe);
        }
        else {
            openEps.add((Integer)args[1]);
            openPipes.add(pipe);
        }
        out.println("Pipe number = " + pnum);
    }


   /**
    ***************************************************************************
    **
    **  Processes the CLOSE command
    **
    ***************************************************************************
    */
    private void procClose(int found, Object[] args) throws UsbException
    {
        int pnum = (Integer)args[0];
        if (pnum >= openEps.size())
            out.println("Invalid pipe number");
        else {
            UsbCommPipe pipe = openPipes.get(pnum);
            openPipes.set(pnum, null);
            openEps.set(pnum, -1);
            pipe.close();
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the FLUSH command
    **
    ***************************************************************************
    */
    private void procFlush(int found, Object[] args) throws UsbException
    {
        int pnum = (Integer)args[0];
        if (pnum >= openEps.size())
            out.println("Invalid pipe number");
        else {
            UsbCommPipe pipe = openPipes.get(pnum);
            out.println(pipe.flush() + " bytes flushed");
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ command
    **
    ***************************************************************************
    */
    private void procRead(int found, Object[] args) throws UsbException
    {
        int pnum = (Integer)args[0];
        int count = (Integer)args[1];
        float timeout = 1f;
        if ((found & 0x04) != 0) timeout = (Float)args[2];
        if (pnum >= openEps.size())
            out.println("Invalid pipe number");
        else {
            UsbCommPipe pipe = openPipes.get(pnum);
            byte[] data = new byte[count];
            int nRead = pipe.read(data, (int)(1000f * timeout));
            dispData(data, nRead);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITE command
    **
    ***************************************************************************
    */
    private void procWrite(int found, Object[] args) throws UsbException
    {
        int pnum = (Integer)args[0];
        byte[] data = genData(found & 0x1fe, args);
        if (data == null) return;
        if (pnum >= openEps.size())
            out.println("Invalid pipe number");
        else {
            UsbCommPipe pipe = openPipes.get(pnum);
            out.println(pipe.write(data) + " bytes written");
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the CLAIM command
    **
    ***************************************************************************
    */
    private void procClaim(int found, Object[] args) throws UsbException
    {
        com.claim((Integer)args[0],
                  (found & 0x02) != 0 && (Integer)args[1] != 0);
    }


   /**
    ***************************************************************************
    **
    **  Processes the RELEASE command
    **
    ***************************************************************************
    */
    private void procRelease(int found, Object[] args) throws UsbException
    {
        com.release((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETCFG command
    **
    ***************************************************************************
    */
    private void procSetCfg(int found, Object[] args) throws UsbException
    {
        com.setConfiguration((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETIF command
    **
    ***************************************************************************
    */
    private void procSetIf(int found, Object[] args) throws UsbException
    {
        com.setInterface((Integer)args[0], (Integer)args[1]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWOPEN command
    **
    ***************************************************************************
    */
    private void procShowOpen(int found, Object[] args) throws UsbException
    {
        int nOpen = 0;
        for (int j = 0; j < openEps.size(); j++) {
            int epAddr = openEps.get(j);
            if (epAddr < 0) continue;
            nOpen++;
            out.format("Pipe %s: address 0x%02x (%s)\n", j, epAddr,
                       (epAddr & 0x80) != 0 ? "in" : "out");
        }
        if (nOpen == 0)
            out.println("No pipes open");
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWBUS command
    **
    ***************************************************************************
    */
    private void procShowBus(int found, Object[] args) throws UsbException
    {
        List devs = UsbComm.findDevices(-1, -1, null);
        if (!devs.isEmpty()) {
            for (int j = 0; j < devs.size(); j++) {
                UsbDevice dev = (UsbDevice)devs.get(j);
                if (j > 0) out.println();
                out.println("Device " + j + ":");
                showDevice(dev);
            }
        }
        else
            out.println("No devices found");
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWDEV command
    **
    ***************************************************************************
    */
    private void procShowDev(int found, Object[] args) throws UsbException
    {
        int devNum = (found & 0x01) != 0 ? (Integer)args[0] : -1;
        UsbDevice dev = getDevice(devNum);
        if (dev != null)
            showDevice(dev);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWCFG command
    **
    ***************************************************************************
    */
    private void procShowCfg(int found, Object[] args) throws UsbException
    {
        int devNum = (found & 0x02) != 0 ? (Integer)args[1] : -1;
        UsbDevice dev = getDevice(devNum);
        if (dev == null) return;
        int cfgNum = (found & 0x01) != 0 ? (Integer)args[0] : -1;
        UsbConfiguration cfg = getConfiguration(dev, cfgNum);
        if (cfg == null) return;
        UsbConfigurationDescriptor desc = cfg.getUsbConfigurationDescriptor();
        out.format("ConfigNum (name) : %s (%s)\n",
                   desc.bConfigurationValue(), cfg.getConfigurationString());
        int attr = desc.bmAttributes() & 0xff;
        String sAttr = "";
        for (int j = 0; j < cfgaNames.count(); j++) {
            int code = cfgaNames.code(j);
            if ((attr & code) != 0) {
                if (!sAttr.equals("")) sAttr += ", ";
                sAttr += cfgaNames.name(j);
            }
        }
        out.format("Attributes       : 0x%02x (%s)\n", attr, sAttr);
        out.format("Maximum power    : %s mA\n",
                   2 * (desc.bMaxPower() & 0xff));
        out.println("Number interfaces: " + desc.bNumInterfaces());
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWIF command
    **
    ***************************************************************************
    */
    private void procShowIf(int found, Object[] args) throws UsbException
    {
        int devNum = (found & 0x08) != 0 ? (Integer)args[3] : -1;
        UsbDevice dev = getDevice(devNum);
        if (dev == null) return;
        int cfgNum = (found & 0x04) != 0 ? (Integer)args[2] : -1;
        UsbConfiguration cfg = getConfiguration(dev, cfgNum);
        if (cfg == null) return;
        int setNum = (found & 0x02) != 0 ? (Integer)args[1] : -1;
        UsbInterface ifc = getInterface(cfg, (Integer)args[0], setNum);
        if (ifc == null) return;
        UsbInterfaceDescriptor desc = ifc.getUsbInterfaceDescriptor();
        out.format("Interface no. (name): %s (%s)\n",
                   desc.bInterfaceNumber(), ifc.getInterfaceString());
        out.println("Setting number      : " + desc.bAlternateSetting());
        out.println("Active setting no.  : " + ifc.getActiveSettingNumber());
        out.println("Number of settings  : " + ifc.getNumSettings());
        int cls = desc.bInterfaceClass() & 0xff;
        out.format("Interface class     : %s (%s)\n", cls,
                   ifccNames.decode(cls));
        out.println("Interface subclass  : "
                    + (desc.bInterfaceSubClass() & 0xff));
        out.println("Interface protocol  : "
                    + (desc.bInterfaceProtocol() & 0xff));
        out.println("Number of endpoints : " + desc.bNumEndpoints());
        out.print("Endpoint addressess :");
        List eps = ifc.getUsbEndpoints();
        for (int j = 0; j < eps.size(); j++) {
            UsbEndpoint ep = (UsbEndpoint)eps.get(j);
            UsbEndpointDescriptor epDesc = ep.getUsbEndpointDescriptor();
            out.format(" 0x%02x", epDesc.bEndpointAddress());
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWEP command
    **
    ***************************************************************************
    */
    private void procShowEp(int found, Object[] args) throws UsbException
    {
        int devNum = (found & 0x10) != 0 ? (Integer)args[4] : -1;
        UsbDevice dev = getDevice(devNum);
        if (dev == null) return;
        int cfgNum = (found & 0x08) != 0 ? (Integer)args[3] : -1;
        UsbConfiguration cfg = getConfiguration(dev, cfgNum);
        if (cfg == null) return;
        int setNum = (found & 0x04) != 0 ? (Integer)args[2] : -1;
        UsbInterface ifc = getInterface(cfg, (Integer)args[1], setNum);
        if (ifc == null) return;
        UsbEndpoint ep = ifc.getUsbEndpoint((byte)(int)(Integer)args[0]);
        if (ep == null) {
            out.println("Endpoint not found");
            return;
        }
        UsbEndpointDescriptor desc = ep.getUsbEndpointDescriptor();
        out.format("Address          : 0x%02x\n",
                   desc.bEndpointAddress());
        int attr = desc.bmAttributes() & 0xff;
        out.format("Attributes       : 0x%02x (%s)\n",
                   attr, epaNames.decode(attr & 0x03));
        out.format("Interval         : %s msec\n",
                   desc.bInterval());
        out.println("Maxm packet size : " + desc.wMaxPacketSize());
    }


   /**
    ***************************************************************************
    **
    **  Gets a device using its index in the system
    **
    ***************************************************************************
    */
    private UsbDevice getDevice(int index) throws UsbException
    {
        if (index < 0) {
            UsbDevice dev = com.getDevice();
            if (dev == null) out.println("No device active");
            return dev;
        }
        List devs = UsbComm.findDevices(-1, -1, null);
        if (index < devs.size())
            return (UsbDevice)devs.get(index);
        else {
            out.println("Device not found");
            return null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets a configuration
    **
    ***************************************************************************
    */
    private UsbConfiguration getConfiguration(UsbDevice dev, int cfgNum)
    {
        UsbConfiguration cfg;
        if (cfgNum < 0) {
            cfg = dev.getActiveUsbConfiguration();
            if (cfg == null) out.println("No configuration active");
        }
        else {
            cfg = dev.getUsbConfiguration((byte)cfgNum);
            if (cfg == null) out.println("Configuration not found");
        }

        return cfg;
    }


   /**
    ***************************************************************************
    **
    **  Gets an interface
    **
    ***************************************************************************
    */
    private UsbInterface getInterface(UsbConfiguration cfg, int ifNum,
                                      int setNum)
        throws UsbException
    {
        UsbInterface ifc = cfg.getUsbInterface((byte)ifNum);
        if (ifc != null) {
            if (setNum < 0)
                ifc = ifc.getActiveSetting();
            else
                ifc = ifc.getSetting((byte)setNum);
        }
        if (ifc == null) out.println("Interface not found");

        return ifc;
    }


   /**
    ***************************************************************************
    **
    **  Displays read data
    **
    ***************************************************************************
    */
    private void dispData(byte[] data, int count)
    {
        String cnt = count + " bytes read";
        String pad = String.format("%24s", "").substring(0, cnt.length() + 1);
        out.print(cnt);
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j == 0) out.print(":");
                else {
                    out.println();
                    out.print(pad);
                }
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Generates write data from hexadecimal string arguments
    **
    ***************************************************************************
    */
    private byte[] genData(int found, Object[] args)
    {
        int count = 0, mask = found;
        ArrayList list = new ArrayList();
        for (int j = 0; found != 0; found >>= 1, j++) {
            if ((found & 1) == 0) continue;
            String hex = (String)args[j];
            int leng = hex.length();
            if ((leng & 1) != 0) break;
            byte[] sData = new byte[leng / 2];
            try {
                for (int k = 0; k < leng; k += 2) {
                    String num = hex.substring(k, k + 2);
                    sData[k / 2] = (byte)Integer.parseInt(num, 16);
                }
            }
            catch (NumberFormatException e) {
                break;
            }
            count += leng / 2;
            list.add(sData);
            mask ^= (1 << j);
        }
        if (mask != 0) {
            out.println("Invalid hexadecimal string");
            return null;
        }
        byte[] data = new byte[count];
        int l = 0;
        for (int j = 0; j < list.size(); j++) {
            byte[] sData = (byte [])list.get(j);
            for (int k = 0; k < sData.length; k++) {
                data[l++] = sData[k];
            }
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Shows device information
    **
    ***************************************************************************
    */
    private void showDevice(UsbDevice dev) throws UsbException
    {
        UsbDeviceDescriptor desc = dev.getUsbDeviceDescriptor();
        out.format("Vendor ID (name)    : 0x%04x (%s)\n",
                   desc.idVendor(), dev.getManufacturerString());
        out.format("Product ID (name)   : 0x%04x (%s)\n",
                   desc.idProduct(), dev.getProductString());
        out.println("Serial number       : " + dev.getSerialNumberString());
        out.println("Device speed        : "
                      + speedNames.decode(UsbComm.getSpeed(dev)));
        int cls = desc.bDeviceClass() & 0xff;
        out.format("Device class        : %s (%s)\n", cls,
                   devcNames.decode(cls));
        out.println("Device subclass     : "
                    + (desc.bDeviceSubClass() & 0xff));
        out.println("Device protocol     : "
                    + (desc.bDeviceProtocol() & 0xff));
        out.format("USB release         : %04x\n", desc.bcdUSB());
        out.println("Number of configs   : " + desc.bNumConfigurations());
        out.println("Active config number: "
                    + dev.getActiveUsbConfigurationNumber());
        out.println("Maximum packet size : "
                    + (desc.bMaxPacketSize0() & 0xff));
    }

}
