package org.lsst.ccs.drivers.usb;

import java.util.ArrayList;
import java.util.List;
import javax.usb.UsbConfiguration;
import javax.usb.UsbConst;
import javax.usb.UsbControlIrp;
import javax.usb.UsbDevice;
import javax.usb.UsbDeviceDescriptor;
import javax.usb.UsbEndpoint;
import javax.usb.UsbException;
import javax.usb.UsbHostManager;
import javax.usb.UsbHub;
import javax.usb.UsbInterface;
import javax.usb.UsbInterfacePolicy;
import javax.usb.event.UsbDeviceDataEvent;
import javax.usb.event.UsbDeviceErrorEvent;
import javax.usb.event.UsbDeviceEvent;
import javax.usb.event.UsbDeviceListener;
import javax.usb.util.StandardRequest;

/**
 ***************************************************************************
 **
 **  Communicates with a USB device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class UsbComm {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    /** Device speed - undefined*/
    public static final int DEV_SPEED_UNDEFINED   = 0;

    /** Device speed - unknown */
    public static final int DEV_SPEED_UNKNOWN     = 1;

    /** Device speed - low */
    public static final int DEV_SPEED_LOW         = 2;

    /** Device speed - full */
    public static final int DEV_SPEED_FULL        = 3;

    
    /** Device class - see interface */
    public static final int DEV_CLASS_INTERFACE   = 0x00;

    /** Device class - CDC control */
    public static final int DEV_CLASS_CDC_CTRL    = 0x02;

    /** Device class - hub */
    public static final int DEV_CLASS_HUB         = 0x09;

    /** Device class - diagnostic */
    public static final int DEV_CLASS_DIAGNOSTIC  = 0xdc;

    /** Device class - miscellaneous */
    public static final int DEV_CLASS_MISC        = 0xef;

    /** Device class - vendor specific */
    public static final int DEV_CLASS_VENDOR_SPEC = 0xff;

    
    /** Configuration attribute - self-powered */
    public static final int CFG_ATTR_SELF_POWER   = 0x40;

    /** Configuration attribute - remote wakeup */
    public static final int CFG_ATTR_REMOTE_WAKE  = 0x20;

    
    /** Interface class - audio */
    public static final int IFC_CLASS_AUDIO       = 0x01;

    /** Interface class - CDC control */
    public static final int IFC_CLASS_CDC_CTRL    = 0x02;

    /** Interface class - human interface */
    public static final int IFC_CLASS_HID         = 0x03;

    /** Interface class - physical */
    public static final int IFC_CLASS_PHYSICAL    = 0x05;

    /** Interface class - still image */
    public static final int IFC_CLASS_IMAGE       = 0x06;

    /** Interface class - printer */
    public static final int IFC_CLASS_PRINTER     = 0x07;

    /** Interface class - mass storage */
    public static final int IFC_CLASS_MASS_STORE  = 0x08;

    /** Interface class - CDC data */
    public static final int IFC_CLASS_CDC_DATA    = 0x0a;

    /** Interface class - smart card */
    public static final int IFC_CLASS_SMART_CARD  = 0x0b;

    /** Interface class security x */
    public static final int IFC_CLASS_CONTENT_SEC = 0x0d;

    /** Interface class - video */
    public static final int IFC_CLASS_VIDEO       = 0x0e;

    /** Interface class - personal healthcare */
    public static final int IFC_CLASS_HEALTHCARE  = 0x0f;

    /** Interface class - diagnostic */
    public static final int IFC_CLASS_DIAGNOSTIC  = 0xdc;

    /** Interface class - wireless */
    public static final int IFC_CLASS_WIRELESS    = 0xe0;

    /** Interface class - miscellaneous */
    public static final int IFC_CLASS_MISC        = 0xef;

    /** Interface class - application specific */
    public static final int IFC_CLASS_APP_SPEC    = 0xfe;

    /** Interface class - vendor specific */
    public static final int IFC_CLASS_VENDOR_SPEC = 0xff;

    
    /** Endpoint attribute - control */
    public static final int EP_ATTR_CONTROL       = 0x00;

    /** Endpoint attribute - isochronous */
    public static final int EP_ATTR_ISOCHRONOUS   = 0x01;

    /** Endpoint attribute - bulk data */
    public static final int EP_ATTR_BULK          = 0x02;

    /** Endpoint attribute - interrupt */
    public static final int EP_ATTR_INTERRUPT     = 0x03;


    private static UsbHub rootHub;
    private UsbDevice dev;
    private final Listener listener = new Listener();


   /**
    ***************************************************************************
    **
    **  Implements interface claim policy
    **
    ***************************************************************************
    */
    static class Policy implements UsbInterfacePolicy {

        private final boolean force;

        public Policy(boolean force)
        {
            this.force = force;
        }

        @Override
        public boolean forceClaim(UsbInterface iface)
        {
            return force;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class that implements device listener
    **
    ***************************************************************************
    */
    static class Listener implements UsbDeviceListener {

        private Thread mainThread = null;
        private UsbControlIrp irp = null;

        @Override
        public void dataEventOccurred(UsbDeviceDataEvent event)
        {
            if (event.getUsbControlIrp().equals(irp)) mainThread.interrupt();
        }

        @Override
        public void errorEventOccurred(UsbDeviceErrorEvent event)
        {
            if (event.getUsbControlIrp().equals(irp)) mainThread.interrupt();
        }

        @Override
        public void usbDeviceDetached(UsbDeviceEvent event)
        {
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public UsbComm() throws UsbException
    {
        if (rootHub == null) {
            rootHub = UsbHostManager.getUsbServices().getRootUsbHub();
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds USB devices
    **
    **  <p>
    **  The tree of USB devices is searched, looking for devices whose vendor
    **  and product IDs match the ones specified.  If a non-null serial number
    **  is provided, this must be matched as well.
    **
    **  @param  vid     The vendor ID, or -1 if any vendor ID matches.
    **
    **  @param  pid     The product ID, or -1 if any product ID matches.
    **
    **  @param  serial  The serial number, or null if the serial number is
    **                  not to be matched.
    **
    **  @return  The list of matching UsbDevices, empty if no matches found.
    **
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    public static List findDevices(int vid, int pid, String serial)
        throws UsbException
    {
        if (rootHub == null) {
            rootHub = UsbHostManager.getUsbServices().getRootUsbHub();
        }
        ArrayList list = new ArrayList();
        find(rootHub, vid, pid, serial, list);

        return list;
    }


   /**
    ***************************************************************************
    **
    **  Finds a USB device and sets it as current
    **
    **  <p>
    **  The tree of USB devices is searched, looking for the first device
    **  whose vendor & product IDs match the ones specified.  The current
    **  device is set to the resultant device, possibly null.
    **
    **  @param  vid     The vendor ID
    **
    **  @param  pid     The product ID
    **
    **  @return  The matching UsbDevice or null if no match found.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public UsbDevice findDevice(int vid, int pid) throws UsbException
    {
        return findDevice(vid, pid, null, 0);
    }


   /**
    ***************************************************************************
    **
    **  Finds a USB device and sets it as current
    **
    **  <p>
    **  The tree of USB devices is searched, looking for the first device
    **  whose vendor & product IDs, and serial number, match the ones
    **  specified.  The serial number may be null to indicate that any serial
    **  number will match.  The current device is set to the resultant device,
    **  possibly null.
    **
    **  @param  vid     The vendor ID
    **
    **  @param  pid     The product ID
    **
    **  @param  serial  The serial number, or null if the serial number is
    **                  not to be matched.
    **
    **  @return  The matching UsbDevice or null if no match found.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public UsbDevice findDevice(int vid, int pid, String serial)
        throws UsbException
    {
        return findDevice(vid, pid, serial, 0);
    }


   /**
    ***************************************************************************
    **
    **  Finds a USB device and sets it as current
    **
    **  <p>
    **  The tree of USB devices is searched, looking for devices whose vendor
    **  & product IDs match the ones specified.  From this list of matches,
    **  the one with the specified index, starting at 0, is selected.  The
    **  current device is set to the resultant device, possibly null.
    **
    **  @param  vid     The vendor ID
    **
    **  @param  pid     The product ID
    **
    **  @param  index   The index, starting at 0, of the instance of matching
    **                  devices to select.
    **
    **  @return  The matching UsbDevice or null if no match found.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public UsbDevice findDevice(int vid, int pid, int index) throws UsbException
    {
        return findDevice(vid, pid, null, index);
    }


   /**
    ***************************************************************************
    **
    **  Finds a USB device and sets it as current
    **
    **  <p>
    **  The tree of USB devices is searched, looking for devices whose vendor
    **  & product IDs, and serial number, match the ones specified.  From
    **  this list of matches, the one with the specified index, starting at 0,
    **  is selected.  The serial number may be null to indicate that any serial
    **  number will match.  The current device is set to the resultant device,
    **  possibly null.
    **
    **  @param  vid     The vendor ID
    **
    **  @param  pid     The product ID
    **
    **  @param  serial  The serial number, or null if the serial number is
    **                  not to be matched.
    **
    **  @param  index   The index, starting at 0, of the instance of matching
    **                  devices to select.
    **
    **  @return  The matching UsbDevice or null if no match found.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public UsbDevice findDevice(int vid, int pid, String serial, int index)
        throws UsbException
    {
        UsbDevice devc = null;
        if (vid != -1 && pid != -1) {
            ArrayList list = new ArrayList();
            find(rootHub, vid, pid, serial, list);
            devc = (list.size() <= index) ? null : (UsbDevice)list.get(index);
        }
        setDevice(devc);

        return devc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the current USB device
    **
    **  @return  The USB device object
    **
    ***************************************************************************
    */
    public UsbDevice getDevice()
    {
        return dev;
    }


   /**
    ***************************************************************************
    **
    **  Sets the current USB device
    **
    **  @param  device  The UsbDevice object to set.  Can be null.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public void setDevice(UsbDevice device) throws UsbException
    {
        if (dev != null)
            dev.removeUsbDeviceListener(listener);
        dev = device;
        if (dev != null)
            dev.addUsbDeviceListener(listener);
    }


   /**
    ***************************************************************************
    **
    **  Gets the active USB configuration for the current device
    **
    **  @return  The USB configuration object
    **
    ***************************************************************************
    */
    public UsbConfiguration getConfiguration()
    {
        return (dev != null) ? dev.getActiveUsbConfiguration() : null;
    }


   /**
    ***************************************************************************
    **
    **  Sets the active USB configuration for the current device
    **
    **  <p>
    **  Note: fails if an interface has been claimed
    **
    **  @param  cfgNum  The number of the configuration to set
    **
    **  @throws  UsbException if no device set, configuration number is
    **                        invalid, or interface claimed
    **
    ***************************************************************************
    */
    public void setConfiguration(int cfgNum) throws UsbException
    {
        checkDevice();
        StandardRequest.setConfiguration(dev, (short)cfgNum);
    }


   /**
    ***************************************************************************
    **
    **  Gets the active setting of the specified USB interface of the current
    **  device
    **
    **  @param  iface  The number, starting from 0, of the interface .
    **
    **  @return  The USB interface object, or null
    **
    ***************************************************************************
    */
    public UsbInterface getInterface(int iface)
    {
        if (dev != null) return null;
        UsbConfiguration cfg = dev.getActiveUsbConfiguration();
        if (cfg == null) return null;
        UsbInterface ifc = cfg.getUsbInterface((byte)iface);
        return (ifc != null) ? ifc.getActiveSetting() : null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the active setting of the specified USB interface of the
    **  specified device.   *** Deprecated ***
    **
    **  @param  device  The USB device
    **
    **  @param  iface   The number, starting from 0, of the interface.
    **
    **  @return  The USB interface object, or null
    **
    ***************************************************************************
    */
    public static UsbInterface getInterface(UsbDevice device, int iface)
    {
        UsbConfiguration cfg = device.getActiveUsbConfiguration();
        if (cfg == null) return null;
        UsbInterface ifc = cfg.getUsbInterface((byte)iface);
        return (ifc != null) ? ifc.getActiveSetting() : null;
    }


   /**
    ***************************************************************************
    **
    **  Sets the active setting of the specified USB interface of the current
    **  device
    **
    **  <p>
    **  Note: fails if the interface has not been claimed
    **
    **  @param  iface    The number, starting from 0, of the interface.
    **
    **  @param  setting  The setting number, starting from 0, to use.
    **
    **  @throws  UsbException if no device set, setting or interface number is
    **                        invalid, or interface not claimed
    **
    ***************************************************************************
    */
    public void setInterface(int iface, int setting) throws UsbException
    {
        checkDevice();
        StandardRequest.setInterface(dev, (short)iface, (short)setting);
    }


   /**
    ***************************************************************************
    **
    **  Gets a USB endpoint for the specified interface.  *** Deprecated ***
    **
    **  @param  iface   The USB interface to use
    **
    **  @param  epAddr  The address of the endpoint to get.
    **
    **  @return  The USB endpoint object, or null
    **
    ***************************************************************************
    */
    public static UsbEndpoint getEndpoint(UsbInterface iface, int epAddr)
    {
        return iface.getUsbEndpoint((byte)epAddr);
    }


   /**
    ***************************************************************************
    **
    **  Gets all USB endpoints for the specified interface.  *** Deprecated ***
    **
    **  @param  iface   The USB interface to use
    **
    **  @return  The list of USB endpoint objects
    **
    ***************************************************************************
    */
    public static List getEndpoints(UsbInterface iface)
    {
        return iface.getUsbEndpoints();
    }


   /**
    ***************************************************************************
    **
    **  Gets the speed of the current device  *** Re-do ***
    **
    **  @return  The speed of the device, encoded locally.
    **
    ***************************************************************************
    */
    public int getSpeed()
    {
        return (dev != null) ? getSpeed(dev) : DEV_SPEED_UNDEFINED;
    }


   /**
    ***************************************************************************
    **
    **  Gets the speed of a device   *** Deprecated ***
    **
    **  @param  device  The USB device
    **
    **  @return  The speed of the device, encoded locally.
    **
    ***************************************************************************
    */
    public static int getSpeed(UsbDevice device)
    {
        Object speed = device.getSpeed();
        if (speed.equals(UsbConst.DEVICE_SPEED_UNKNOWN))
            return DEV_SPEED_UNKNOWN;
        if (speed.equals(UsbConst.DEVICE_SPEED_LOW))
            return DEV_SPEED_LOW;
        if (speed.equals(UsbConst.DEVICE_SPEED_FULL))
            return DEV_SPEED_FULL;
        return DEV_SPEED_UNDEFINED;
    }


   /**
    ***************************************************************************
    **
    **  Claims an interface
    **
    **  @param  iface  The number, starting from 0, of the interface.
    **
    **  @param  force  Set to true if claiming the device interface is to be
    **                 forced, allowing the use of devices claimed by another
    **                 driver or user.
    **
    **  @throws  UsbException if no device set, device not configured,
    **                        interface number invalid, etc
    **
    ***************************************************************************
    */
    public void claim(int iface, boolean force) throws UsbException
    {
        // Check that there is a device set
        checkDevice();

        // Get the active interface and claim it
        UsbConfiguration cfg = dev.getActiveUsbConfiguration();
        if (cfg == null)
            throw new UsbException("Device not configured");
        UsbInterface ifc = cfg.getUsbInterface((byte)iface);
        if (ifc != null) ifc = ifc.getActiveSetting();
        if (ifc == null)
            throw new UsbException("Interface not found");
        ifc.claim(new Policy(force));
    }

        
   /**
    ***************************************************************************
    **
    **  Releases an interface
    **
    ***************************************************************************
    */
    public void release(int iface) throws UsbException
    {
        // Check that there is a device set
        checkDevice();

        // Get the active interface and release it
        UsbConfiguration cfg = dev.getActiveUsbConfiguration();
        UsbInterface ifc = cfg.getUsbInterface((byte)iface).getActiveSetting();
        if (ifc == null)
            throw new UsbException("Interface not found");
        ifc.release();
    }

        
   /**
    ***************************************************************************
    **
    **  Opens a connection to a device's endpoint
    **
    **  <p>
    **  The device is assumed to be configured, and the specified interface
    **  is used.
    **
    **  @param  iface   The number of the interface to use, starting at 0.
    **
    **  @param  epaddr  The address of the endpoint to be opened.
    **
    **  @throws  UsbException if the device has not been set or the requested
    **                        interface or endpoints do not exist.
    **
    **  @throws  UsbClaimException if the interface cannot be claimed.
    **
    ***************************************************************************
    */
    public UsbCommPipe open(int iface, int epAddr)
        throws UsbException
    {
        // Check that there is a device to open
        checkDevice();

        // Get the active interface
        UsbConfiguration cfg = dev.getActiveUsbConfiguration();
        UsbInterface ifc = cfg.getUsbInterface((byte)iface).getActiveSetting();
        if (ifc == null)
            throw new UsbException("Interface not found");

        // Get the endpoint and open its pipe
        UsbEndpoint ep = ifc.getUsbEndpoint((byte)epAddr);
        if (ep == null)
            throw new UsbException("Endpoint not found");

        return new UsbCommPipe(ep.getUsbPipe());
    }


   /**
    ***************************************************************************
    **
    **  Writes data to the control endpoint
    **
    **  @param  type   The request type
    **
    **  @param  rqst   The request
    **
    **  @param  value  The value
    **
    **  @param  index  The index
    **
    **  @param  data   Byte array of data to be written.
    **
    **  @return  The number of bytes of data actually written.
    **
    **  @throws  UsbException if the out endpoint is not open.
    **
    ***************************************************************************
    */
    public int write(int type, int rqst, int value, int index, byte[] data)
        throws UsbException
    {
        return write(type, rqst, value, index, data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data to the control endpoint
    **
    **  @param  type   The request type
    **
    **  @param  rqst   The request
    **
    **  @param  value  The value
    **
    **  @param  index  The index
    **
    **  @param  data   Byte array containing the data to be written.
    **
    **  @param  offs   The offset of the first byte to be written.
    **
    **  @param  leng   The number of bytes to be written.
    **
    **  @return  The number of bytes of data actually written.
    **
    **  @throws  UsbException if the out endpoint is not open.
    **
    ***************************************************************************
    */
    public int write(int type, int rqst, int value, int index, byte[] data,
                     int offs, int leng) throws UsbException
    {
        // Check that there is a device to write to
        checkDevice();

        // Check the direction
        if ((type & 0x80) != 0) {
            throw new UsbException("Cannot write to IN endpoint");
        }

        // Create the request packet
        UsbControlIrp irp = dev.createUsbControlIrp((byte)type, (byte)rqst,
                                                    (short)value, (short)index);
        irp.setData(data, offs, leng);
        irp.setComplete(false);
        irp.setUsbException(null);

        // Submit the request synchronously
        dev.syncSubmit(irp);

        return irp.getActualLength();
    }


   /**
    ***************************************************************************
    **
    **  Reads data from the control endpoint
    **
    **  @param  type     The request type
    **
    **  @param  rqst     The request
    **
    **  @param  value    The value
    **
    **  @param  index    The index
    **
    **  @param  data     Byte array to receive the read data.
    **
    **  @param  timeout  The maximum time, in milliseconds, to wait for data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  UsbException if the in endpoint is not open.
    **
    **  @throws  UsbTimeoutException if the read times out.
    **
    ***************************************************************************
    */
    public int read(int type, int rqst, int value, int index, byte[] data,
                    int timeout)
        throws UsbException, UsbTimeoutException
    {
        return read(type, rqst, value, index, data, 0, data.length, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads data from the control endpoint
    **
    **  @param  type     The request type
    **
    **  @param  rqst     The request
    **
    **  @param  value    The value
    **
    **  @param  index    The index
    **
    **  @param  data     Byte array to receive the read data.
    **
    **  @param  offs     The offset in {@code data} where the read data is to
    **                   start.
    **
    **  @param  leng     The number of bytes of data to read.
    **
    **  @param  timeout  The maximum time, in milliseconds, to wait for data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  UsbException if the in endpoint is not open.
    **
    **  @throws  UsbTimeoutException if the read times out.
    **
    ***************************************************************************
    */
    public int read(int type, int rqst, int value, int index, byte[] data,
                    int offs, int leng, int timeout)
        throws UsbException, UsbTimeoutException
    {
        // Check that there is a device to read from
        checkDevice();

        // Check the direction
        if ((type & 0x80) == 0)
            throw new UsbException("Cannot read from OUT endpoint");

        // Create the request packet
        UsbControlIrp irp = dev.createUsbControlIrp((byte)type, (byte)rqst,
                                                    (short)value, (short)index);
        irp.setData(data, offs, leng);
        irp.setComplete(false);
        irp.setUsbException(null);

        // Submit the request
        if (timeout == 0) {
            dev.syncSubmit(irp);
        }
        else {
            Thread.interrupted();   // Clears any interrupt
            listener.mainThread = Thread.currentThread();
            listener.irp = irp;
            try {
                dev.asyncSubmit(irp);
                try {
                    Thread.sleep((long)timeout);
                }
                catch (InterruptedException e) {
                }
            }
            finally {
                listener.irp = null;
            }
        }

        // Check the result
        UsbException re = irp.getUsbException();
        if (re != null) throw re;
        if (!irp.isComplete()) {
            throw new UsbTimeoutException();
        }

        // Return the number of bytes read
        return irp.getActualLength();
    }


   /**
    ***************************************************************************
    **
    **  Finds all the USB devices matching vendor and product IDs
    **
    **  <p>
    **  This routine is used recursively to travel the tree of USB devices
    **  to locate the one with matching vendor and product IDs and optional
    **  serial number.
    **
    **  @param  hub     The hub where the search is to begin.
    **
    **  @param  vid     The vendor ID to be matched, or -1 if all match.
    **
    **  @param  pid     The product ID to be matched, -1 if all match.
    **
    **  @param  serial  If non-null, the serial number to be matched.
    **
    **  @param  list    The list to fill with matching USB devices.
    **
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    private static void find(UsbHub hub, int vid, int pid, String serial,
                             List list)
        throws UsbException
    {
        int nPort = hub.getNumberOfPorts() & 0xff;
        for (int j = 1; j <= nPort; j++) {
            UsbDevice dev = hub.getUsbPort((byte)j).getUsbDevice();
            if (dev == null) continue;
            if (dev.isUsbHub()) {
                find((UsbHub)dev, vid, pid, serial, list);
            }
            else {
                UsbDeviceDescriptor desc = dev.getUsbDeviceDescriptor();
                if ((vid == -1 || (short)vid == desc.idVendor())
                      && (pid == -1 || (short)pid == desc.idProduct())) {
                    if (serial == null) list.add(dev);
                    else {
                        String serno = dev.getSerialNumberString();
                        if (serno != null && serno.equals(serial))
                            list.add(dev);
                    }
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks that the device has been set
    **
    ***************************************************************************
    */
    private void checkDevice() throws UsbException
    {
        if (dev == null)
            throw new UsbException("Device has not been set");
    }

}
