package org.lsst.ccs.drivers.usb;

import javax.usb.UsbBabbleException;
import javax.usb.UsbException;
import javax.usb.UsbIrp;
import javax.usb.UsbPipe;
import javax.usb.event.UsbPipeDataEvent;
import javax.usb.event.UsbPipeErrorEvent;
import javax.usb.event.UsbPipeListener;

/**
 ***************************************************************************
 **
 **  Performs I/O operations on a UsbPipe
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class UsbCommPipe {

    private UsbPipe pipe;
    private boolean input;
    private final Listener listener = new Listener();
    private Thread exitHook =
        new Thread() {
            @Override
            public void run() {
                try {
                    close();
                }
                catch (Exception e) {
                }
            }
        };

   /**
    ***************************************************************************
    **
    **  Inner class that implements pipe listener
    **
    ***************************************************************************
    */
    static class Listener implements UsbPipeListener {

        private Thread mainThread = null;
        private UsbIrp irp = null;

        @Override
        public void dataEventOccurred(UsbPipeDataEvent event)
        {
            if (event.getUsbIrp().equals(irp)) mainThread.interrupt();
        }

        @Override
        public void errorEventOccurred(UsbPipeErrorEvent event)
        {
            if (event.getUsbIrp().equals(irp)) mainThread.interrupt();
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public UsbCommPipe(UsbPipe pipe) throws UsbException
    {
        this.pipe = pipe;
        input = pipe.getUsbEndpoint().getDirection() != 0;
        pipe.addUsbPipeListener(listener);
        pipe.open();
        Runtime.getRuntime().addShutdownHook(exitHook);
    }


   /**
    ***************************************************************************
    **
    **  Closes the pipe
    **
    **  @throws  UsbException if the device is not open.
    **
    ***************************************************************************
    */
    public void close() throws UsbException
    {
        pipe.close();
        Runtime.getRuntime().removeShutdownHook(exitHook);
    }


   /**
    ***************************************************************************
    **
    **  Writes data to the pipe
    **
    **  @param  data  Byte array of data to be written.
    **
    **  @return  The number of bytes of data actually written.
    **
    **  @throws  UsbException if the out endpoint is not open.
    **
    ***************************************************************************
    */
    public int write(byte[] data) throws UsbException
    {
        return write(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data to the pipe
    **
    **  @param  data  Byte array containing the data to be written.
    **
    **  @param  offs  The offset of the first byte to be written.
    **
    **  @param  leng  The number of bytes to be written.
    **
    **  @return  The number of bytes of data actually written.
    **
    **  @throws  UsbException if the out endpoint is not open.
    **
    ***************************************************************************
    */
    public int write(byte[] data, int offs, int leng) throws UsbException
    {
        // Check the direction
        if (input) {
            throw new UsbException("Cannot write to IN endpoint");
        }

        // Create the request packet
        UsbIrp irp = pipe.createUsbIrp();
        irp.setData(data, offs, leng);
        irp.setComplete(false);
        irp.setUsbException(null);

        // Submit the request synchronously
        pipe.syncSubmit(irp);

        return irp.getActualLength();
    }


   /**
    ***************************************************************************
    **
    **  Reads data from the pipe
    **
    **  @param  data     Byte array to receive the read data.
    **
    **  @param  timeout  The maximum time, in milliseconds, to wait for data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  UsbException if the in endpoint is not open.
    **
    **  @throws  UsbTimeoutException if the read times out.
    **
    ***************************************************************************
    */
    public int read(byte[] data, int timeout)
        throws UsbException, UsbTimeoutException
    {
        return read(data, 0, data.length, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads data from the pipe
    **
    **  @param  data     Byte array to receive the read data.
    **
    **  @param  offs     The offset in {@code data} where the read data is to
    **                   start.
    **
    **  @param  leng     The number of bytes of data to read.
    **
    **  @param  timeout  The maximum time, in milliseconds, to wait for data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  UsbException if the in endpoint is not open.
    **
    **  @throws  UsbTimeoutException if the read times out.
    **
    ***************************************************************************
    */
    public int read(byte[] data, int offs, int leng, int timeout)
        throws UsbException, UsbTimeoutException
    {
        // Check the direction
        if (!input) {
            throw new UsbException("Cannot read from OUT endpoint");
        }

        // Create the request packet
        UsbIrp irp = pipe.createUsbIrp();
        irp.setData(data, offs, leng);
        irp.setComplete(false);
        irp.setUsbException(null);

        // Submit the request
        if (timeout == 0) {
            pipe.syncSubmit(irp);
        }
        else {
            Thread.interrupted();   // Clears any interrupt
            listener.mainThread = Thread.currentThread();
            listener.irp = irp;
            try {
                pipe.asyncSubmit(irp);
                try {
                    Thread.sleep((long)timeout);
                }
                catch (InterruptedException e) {
                }
            }
            finally {
                listener.irp = null;
            }
        }

        // Check the result
        UsbException re = irp.getUsbException();
        if (re != null) throw re;
        if (!irp.isComplete()) {
            try {
                pipe.abortAllSubmissions();
            }
            catch (Exception e) {
            }
            throw new UsbTimeoutException();
        }

        // Return the number of bytes read
        return irp.getActualLength();
    }


   /**
    ***************************************************************************
    **
    **  Flushes any pending read data
    **
    **  @return  The number of bytes flushed.
    **
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    public int flush() throws UsbException
    {
        int leng = 0;
        byte[] data = new byte[64];

        while (true) {
            try {
                leng += read(data, 100);
            }
            catch (UsbTimeoutException e) {
                break;
            }
            catch (UsbBabbleException e) {
                data = new byte[2 * data.length];
            }
        }

        return leng;
    }

}
