package org.lsst.ccs.drivers.usb;

import javax.usb.*;
import javax.usb.event.*;
import java.util.*;

/**
 ***************************************************************************
 **
 **  Communicates with a USB device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class UsbComm {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    /** Device speed - undefined*/
    public static final int DEV_SPEED_UNDEFINED   = 0;

    /** Device speed - unknown */
    public static final int DEV_SPEED_UNKNOWN     = 1;

    /** Device speed - low */
    public static final int DEV_SPEED_LOW         = 2;

    /** Device speed - full */
    public static final int DEV_SPEED_FULL        = 3;

    
    /** Device class - see interface */
    public static final int DEV_CLASS_INTERFACE   = 0x00;

    /** Device class - CDC control */
    public static final int DEV_CLASS_CDC_CTRL    = 0x02;

    /** Device class - hub */
    public static final int DEV_CLASS_HUB         = 0x09;

    /** Device class - diagnostic */
    public static final int DEV_CLASS_DIAGNOSTIC  = 0xdc;

    /** Device class - miscellaneous */
    public static final int DEV_CLASS_MISC        = 0xef;

    /** Device class - vendor specific */
    public static final int DEV_CLASS_VENDOR_SPEC = 0xff;

    
    /** Configuration attribute - self-powered */
    public static final int CFG_ATTR_SELF_POWER   = 0x40;

    /** Configuration attribute - remote wakeup */
    public static final int CFG_ATTR_REMOTE_WAKE  = 0x20;

    
    /** Interface class - audio */
    public static final int IFC_CLASS_AUDIO       = 0x01;

    /** Interface class - CDC control */
    public static final int IFC_CLASS_CDC_CTRL    = 0x02;

    /** Interface class - human interface */
    public static final int IFC_CLASS_HID         = 0x03;

    /** Interface class - physical */
    public static final int IFC_CLASS_PHYSICAL    = 0x05;

    /** Interface class - still image */
    public static final int IFC_CLASS_IMAGE       = 0x06;

    /** Interface class - printer */
    public static final int IFC_CLASS_PRINTER     = 0x07;

    /** Interface class - mass storage */
    public static final int IFC_CLASS_MASS_STORE  = 0x08;

    /** Interface class - CDC data */
    public static final int IFC_CLASS_CDC_DATA    = 0x0a;

    /** Interface class - smart card */
    public static final int IFC_CLASS_SMART_CARD  = 0x0b;

    /** Interface class security x */
    public static final int IFC_CLASS_CONTENT_SEC = 0x0d;

    /** Interface class - video */
    public static final int IFC_CLASS_VIDEO       = 0x0e;

    /** Interface class - personal healthcare */
    public static final int IFC_CLASS_HEALTHCARE  = 0x0f;

    /** Interface class - diagnostic */
    public static final int IFC_CLASS_DIAGNOSTIC  = 0xdc;

    /** Interface class - wireless */
    public static final int IFC_CLASS_WIRELESS    = 0xe0;

    /** Interface class - miscellaneous */
    public static final int IFC_CLASS_MISC        = 0xef;

    /** Interface class - application specific */
    public static final int IFC_CLASS_APP_SPEC    = 0xfe;

    /** Interface class - vendor specific */
    public static final int IFC_CLASS_VENDOR_SPEC = 0xff;

    
    /** Endpoint attribute - control */
    public static final int EP_ATTR_CONTROL       = 0x00;

    /** Endpoint attribute - isochronous */
    public static final int EP_ATTR_ISOCHRONOUS   = 0x01;

    /** Endpoint attribute - bulk data */
    public static final int EP_ATTR_BULK          = 0x02;

    /** Endpoint attribute - interrupt */
    public static final int EP_ATTR_INTERRUPT     = 0x03;


    private static UsbHub rootHub;

    private final Listener listener = new Listener();
    private UsbDevice dev;
    private UsbConfiguration conf;
    private UsbInterface ifc;
    private UsbEndpoint iEp, oEp;
    private UsbPipe iPipe, oPipe;


   /**
    ***************************************************************************
    **
    **  Implements interface claim policy
    **
    ***************************************************************************
    */
    static class Policy implements UsbInterfacePolicy {

        private final boolean force;

        public Policy(boolean force)
        {
            this.force = force;
        }

        @Override
        public boolean forceClaim(UsbInterface iface)
        {
            return force;
        }

    }


   /**
    ***************************************************************************
    **
    **  Implements pipe listener
    **
    ***************************************************************************
    */
    static class Listener implements UsbPipeListener {

        private Thread mainThread;

        @Override
        public void dataEventOccurred(UsbPipeDataEvent event)
        {
            mainThread.interrupt();
        }

        @Override
        public void errorEventOccurred(UsbPipeErrorEvent event)
        {
            mainThread.interrupt();
        }
    }


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public UsbComm() throws UsbException {
        Runtime.getRuntime().addShutdownHook(
            new Thread() {
                @Override
                public void run()
                {
                    try {
                        close();
                    }
                    catch (Exception e) {
                    }
                }
            }
        );

        if (rootHub == null) {
            rootHub = UsbHostManager.getUsbServices().getRootUsbHub();
        }
    }


   /**
    ***************************************************************************
    **
    **  Finds USB devices
    **
    **  <p>
    **  The tree of USB devices is searched, looking for devices whose vendor
    **  and product IDs match the ones specified.  If a non-null serial number
    **  is provided, this must be matched as well; otherwise the first ID
    **  match is used.
    **
    **  @param  vid     The vendor ID, or -1 if any vendor ID matches.
    **
    **  @param  pid     The product ID, or -1 if any product ID matches.
    **
    **  @param  serial  The serial number, or null if the serial number is
    **                  not to be matched.
    **
    **  @return  The list of matching UsbDevices, empty if no matches found.
    **
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    public static List findDevices(int vid, int pid, String serial)
        throws UsbException
    {
        if (rootHub == null) {
            rootHub = UsbHostManager.getUsbServices().getRootUsbHub();
        }
        ArrayList list = new ArrayList();
        find(rootHub, vid, pid, serial, list);

        return list;
    }


   /**
    ***************************************************************************
    **
    **  Finds a USB device and sets it
    **
    **  <p>
    **  The tree of USB devices is searched, looking for devices whose vendor
    **  and product IDs match the ones specified.  If a non-null serial number
    **  is provided, this must be matched as well; otherwise the first ID
    **  match is used.  The current device is set to the resultant device,
    **  possibly null.
    **
    **  @param  vid     The vendor ID
    **
    **  @param  pid     The product ID
    **
    **  @param  serial  The serial number, or null if the serial number is
    **                  not to be matched.
    **
    **  @return  The matching UsbDevice or null if no match found.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public UsbDevice findDevice(int vid, int pid, String serial)
        throws UsbException
    {
        UsbDevice devc = null;
        if (vid != -1 && pid != -1) {
            ArrayList list = new ArrayList();
            find(rootHub, vid, pid, serial, list);
            devc = (list.isEmpty()) ? null : (UsbDevice)list.get(0);
        }
        setDevice(devc);

        return dev;
    }


   /**
    ***************************************************************************
    **
    **  Gets the USB device
    **
    **  @return  The USB device object
    **
    ***************************************************************************
    */
    public UsbDevice getDevice()
    {
        return dev;
    }


   /**
    ***************************************************************************
    **
    **  Sets the USB device
    **
    **  @param  device  The UsbDevice object to set.  Can be null.
    **
    **  @throws  UsbException if a device is already open.
    **
    ***************************************************************************
    */
    public void setDevice(UsbDevice device) throws UsbException
    {
        if (iPipe != null || oPipe != null)
            throw new UsbException("Device is open");
        dev = device;
        conf = (dev != null) ? dev.getActiveUsbConfiguration() : null;
        ifc = null;
        iEp = null;
        oEp = null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the current USB configuration
    **
    **  @return  The USB configuration object
    **
    ***************************************************************************
    */
    public UsbConfiguration getConfiguration()
    {
        return conf;
    }


   /**
    ***************************************************************************
    **
    **  Gets the USB interface being used
    **
    **  @return  The USB interface object
    **
    ***************************************************************************
    */
    public UsbInterface getInterface()
    {
        return ifc;
    }


   /**
    ***************************************************************************
    **
    **  Gets the specified USB interface of the current device
    **
    **  @param  iface  The number, starting from 0, of the interface .
    **
    **  @return  The USB interface object, or null
    **
    ***************************************************************************
    */
    public UsbInterface getInterface(int iface)
    {
        if (conf != null)
            return conf.getUsbInterface((byte)iface); //.getActiveSetting();
        else
            return null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the specified USB interface of the specified device
    **
    **  @param  device  The USB device
    **
    **  @param  iface   The number, starting from 0, of the interface .
    **
    **  @return  The USB interface object, or null
    **
    ***************************************************************************
    */
    public static UsbInterface getInterface(UsbDevice device, int iface)
    {
        UsbConfiguration cfg = device.getActiveUsbConfiguration();
        return cfg.getUsbInterface((byte)iface); //.getActiveSetting();
    }


   /**
    ***************************************************************************
    **
    **  Gets a USB endpoint for the current open device
    **
    **  @param  epnum  The endpoint to get: 0 = in, 1 = out.
    **
    **  @return  The USB endpoint object, or null
    **
    ***************************************************************************
    */
    public UsbEndpoint getEndpoint(int epnum)
    {
        return epnum == 0 ? iEp : oEp;
    }


   /**
    ***************************************************************************
    **
    **  Gets a USB endpoint for the specified interface
    **
    **  @param  iface  The USB interface to use
    **
    **  @param  epnum  The endpoint to get: 0 = in, anything else = out.
    **
    **  @return  The USB endpoint object
    **
    ***************************************************************************
    */
    public static UsbEndpoint getEndpoint(UsbInterface iface, int epnum)
    {
        List eps = iface.getUsbEndpoints();
        for (int j = 0; j < eps.size(); j++) {
            UsbEndpoint ep = (UsbEndpoint)eps.get(j);
            if (epnum == 0 ^ ep.getDirection() == 0) return ep;
        }

        return null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the speed of the current device
    **
    **  @return  The speed of the device, encoded locally.
    **
    ***************************************************************************
    */
    public int getSpeed()
    {
        return getSpeed(dev);
    }


   /**
    ***************************************************************************
    **
    **  Gets the speed of a device
    **
    **  @return  The speed of the device, encoded locally.
    **
    ***************************************************************************
    */
    public static int getSpeed(UsbDevice devc)
    {
        Object speed = devc.getSpeed();
        if (speed.equals(UsbConst.DEVICE_SPEED_UNKNOWN))
            return DEV_SPEED_UNKNOWN;
        if (speed.equals(UsbConst.DEVICE_SPEED_LOW))
            return DEV_SPEED_LOW;
        if (speed.equals(UsbConst.DEVICE_SPEED_FULL))
            return DEV_SPEED_FULL;
        return DEV_SPEED_UNDEFINED;
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to a device
    **
    **  <p>
    **  The device is assumed to be configured, and the specified interface
    **  is used.  The interface must have an endpoint of each requested type.
    **
    **  @param  iface  The number of the interface to use, starting at 0.
    **
    **  @param  ends   The endpoints to be opened: 0 = in and out, 1 = in,
    **                 2 = out, otherwise in and out.
    **
    **  @param  force  Set to true if claiming the device interface is to be
    **                 forced, allowing the use of devices claimed by another
    **                 driver or user.
    **
    **  @throws  UsbException if the device has not been set or the requested
    **                        interface or endpoints do not exist.
    **
    **  @throws  UsbClaimException if the interface cannot be claimed.
    **
    ***************************************************************************
    */
    public void open(int iface, int ends, boolean force)
        throws UsbException
    {
        // Check that there is a device to open
        if (dev == null)
            throw new UsbException("No device set");

        // Check that it's not already open
        if (iPipe != null || oPipe != null)
            throw new UsbException("Device is open");

        // Get the interface and claim it
        ifc = conf.getUsbInterface((byte)iface); //.getActiveSetting();
        if (ifc == null)
            throw new UsbException("Interface not found");
        ifc.claim(new Policy(force));

        // Get the interface's endpoints and check availability
        List eps = ifc.getUsbEndpoints();
        iEp = oEp = null;
        for (int j = 0; j < eps.size(); j++) {
            UsbEndpoint ep = (UsbEndpoint)eps.get(j);
            if (ep.getDirection() == 0) {
                if (oEp == null) oEp = ep;
            }
            else {
                if (iEp == null) iEp = ep;
            }
        }
        if (iEp == null && ends != 2 || oEp == null && ends != 1) {
            ifc.release();
            throw new UsbException("Endpoint(s) not found");
        }

        // Get the pipes for the endpoints to be used
        if (ends != 2) {
            iPipe = iEp.getUsbPipe();
            iPipe.addUsbPipeListener(listener);
        }
        if (ends != 1) {
            oPipe = oEp.getUsbPipe();
        }

        // Open the pipes: clean up if any problem
        try {
            if (iPipe != null) iPipe.open();
            if (oPipe != null) oPipe.open();
        }
        catch (UsbException e) {
            close();
            throw e;
        }
        catch (RuntimeException e) {
            close();
            throw e;
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes the connection to the device
    **
    **  @throws  UsbException if the device is not open.
    **
    ***************************************************************************
    */
    public void close() throws UsbException
    {
        UsbException ue = null;
        RuntimeException re = null;

        // Check that device is open
        if (iPipe == null && oPipe == null)
            throw new UsbException("Device is not open");

        // Close the input pipe
        if (iPipe != null) {
            try {
                iPipe.abortAllSubmissions();
                if (iPipe.isOpen()) iPipe.close();
            }
            catch (UsbException e) {
                if (ue == null) ue = e;
            }
            catch (RuntimeException e) {
                if (re == null) re = e;
            }
            iPipe = null;
        }

        // Close the output pipe
        if (oPipe != null) {
            try {
                if (oPipe.isOpen()) oPipe.close();
            }
            catch (UsbException e) {
                if (ue == null) ue = e;
            }
            catch (RuntimeException e) {
                if (re == null) re = e;
            }
            oPipe = null;
        }

        // Release the interface
        try {
            ifc.release();
        }
        catch (UsbException e) {
            if (ue == null) ue = e;
        }
        catch (RuntimeException e) {
            if (re == null) re = e;
        }

        // Throw the first caught exception, if any
        if (re != null) throw re;
        if (ue != null) throw ue;
    }


   /**
    ***************************************************************************
    **
    **  Writes data to the device
    **
    **  @param  data  Byte array of data to be written.
    **
    **  @return  The number of bytes of data actually written.
    **
    **  @throws  UsbException if the out endpoint is not open.
    **
    ***************************************************************************
    */
    public int write(byte[] data) throws UsbException
    {
        return write(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data to the device
    **
    **  @param  data  Byte array containing the data to be written.
    **
    **  @param  offs  The offset of the first byte to be written.
    **
    **  @param  leng  The number of bytes to be written.
    **
    **  @return  The number of bytes of data actually written.
    **
    **  @throws  UsbException if the out endpoint is not open.
    **
    ***************************************************************************
    */
    public int write(byte[] data, int offs, int leng) throws UsbException
    {
        if (oPipe == null)
            throw new UsbException("Out endpoint is not open");
        UsbIrp irp = oPipe.createUsbIrp();
        irp.setComplete(false);
        irp.setUsbException(null);
        irp.setData(data, offs, leng);
        oPipe.syncSubmit(irp);

        return irp.getActualLength();
    }


   /**
    ***************************************************************************
    **
    **  Reads data from the device
    **
    **  @param  data     Byte array to receive the read data.
    **
    **  @param  timeout  The maximum time, in milliseconds, to wait for data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  UsbException if the in endpoint is not open.
    **
    **  @throws  UsbTimeoutException if the read times out.
    **
    ***************************************************************************
    */
    public int read(byte[] data, int timeout)
        throws UsbException, UsbTimeoutException
    {
        return read(data, 0, data.length, timeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads data from the device
    **
    **  @param  data     Byte array to receive the read data.
    **
    **  @param  offs     The offset in {@code data} where the read data is to
    **                   start.
    **
    **  @param  leng     The number of bytes of data to read.
    **
    **  @param  timeout  The maximum time, in milliseconds, to wait for data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws  UsbException if the in endpoint is not open.
    **
    **  @throws  UsbTimeoutException if the read times out.
    **
    ***************************************************************************
    */
    public synchronized int read(byte[] data, int offs, int leng, int timeout)
        throws UsbException, UsbTimeoutException
    {
        if (iPipe == null)
            throw new UsbException("In endpoint is not open");
        UsbIrp irp = iPipe.createUsbIrp();
        irp.setComplete(false);
        irp.setUsbException(null);
        irp.setData(data, offs, leng);
        listener.mainThread = Thread.currentThread();
        if (timeout == 0)
            iPipe.syncSubmit(irp);
        else {
            Thread.interrupted();   // Clears any interrupt
            iPipe.asyncSubmit(irp);
            try {
                Thread.sleep((long)timeout);
            }
            catch (InterruptedException e) {
            }
        }
        UsbException re = irp.getUsbException();
        if (re != null) throw re;
        if (!irp.isComplete()) {
            try {
                iPipe.abortAllSubmissions();
            }
            catch (Exception e) {
            }
            throw new UsbTimeoutException();
        }

        return irp.getActualLength();
    }


   /**
    ***************************************************************************
    **
    **  Flushes any pending read data
    **
    **  @return  The number of bytes flushed.
    **
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    public int flush() throws UsbException
    {
        int leng = 0;
        byte[] data = new byte[64];

        while (true) {
            try {
                leng += read(data, 100);
            }
            catch (UsbTimeoutException e) {
                break;
            }
            catch (UsbBabbleException e) {
                data = new byte[2 * data.length];
            }
        }

        return leng;
    }


   /**
    ***************************************************************************
    **
    **  Finds all the USB devices matching vendor and product IDs
    **
    **  <p>
    **  This routine is used recursively to travel the tree of USB devices
    **  to locate the one with matching vendor and product IDs and optional
    **  serial number.
    **
    **  @param  hub     The hub where the search is to begin.
    **
    **  @param  vid     The vendor ID to be matched, or -1 if all match.
    **
    **  @param  pid     The product ID to be matched, -1 if all match.
    **
    **  @param  serial  If non-null, the serial number to be matched.
    **
    **  @param  list    The list to fil with matching USB devices.
    **
    **  @throws  UsbException
    **
    ***************************************************************************
    */
    private static void find(UsbHub hub, int vid, int pid, String serial,
                             List list)
        throws UsbException
    {
        int nPort = hub.getNumberOfPorts() & 0xff;
        for (int j = 1; j <= nPort; j++) {
            UsbDevice dev = hub.getUsbPort((byte)j).getUsbDevice();
            if (dev == null) continue;
            if (dev.isUsbHub()) {
                find((UsbHub)dev, vid, pid, serial, list);
            }
            else {
                UsbDeviceDescriptor desc = dev.getUsbDeviceDescriptor();
                if ((vid == -1 || (short)vid == desc.idVendor())
                      && (pid == -1 || (short)pid == desc.idProduct())) {
                    if (serial == null) list.add(dev);
                    else {
                        String serno = dev.getSerialNumberString();
                        if (serno != null && serno.equals(serial))
                            list.add(dev);
                    }
                }
            }
        }
    }

}
