package org.lsst.ccs.drivers.usb;

import javax.usb.*;
import org.lsst.ccs.utilities.sa.*;
import java.util.*;
import java.io.IOException;
import jline.ConsoleReader;

/**
 ***************************************************************************
 **
 **  Program to test the UsbComm methods
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestUsbComm implements CmndProc.Dispatch {

    private final static int
        CMD_FIND        = 0,
        CMD_OPEN        = 1,
        CMD_CLOSE       = 2,
        CMD_FLUSH       = 3,
        CMD_READ        = 4,
        CMD_WRITE       = 5,
        CMD_SHOWDEV     = 6,
        CMD_SHOWCFG     = 7,
        CMD_SHOWIF      = 8,
        CMD_SHOWEP      = 9,
        CMD_SHOWBUS     = 10,
        NUM_CMDS        = 11;

    private final static String[] helpFind = {
        "Find a device",
        "find <vendId> <devId> [<serial>]",
        "vendId  The vendor ID of the device",
        "devId   The device ID of the device",
        "serial  The serial number of the device (default any)",
    };

    private final static String[] helpOpen = {
        "Open device connection",
        "open [<iface>] [<ends>] [<force>]",
        "iface  The number of the interface to open (default 0)",
        "ends   The endpoints to open (default 0)",
        "force  If present and non-zero, force the interface claim",
    };

    private final static String[] helpClose = {
        "Close device connection",
        "close",
    };

    private final static String[] helpFlush = {
        "Flush any unread data",
        "flush",
    };

    private final static String[] helpRead = {
        "Read data from the device and display it",
        "read <count> [<timeout>]",
        "count    The number of bytes to read",
        "timeout  The timeout in seconds (default 1)",
    };

    private final static String[] helpWrite = {
        "Write data to the device",
        "write [<data1>]... [<datan>]",
        "datan  Up to 8 hexadecimal strings",
    };

    private final static String[] helpShowdev = {
        "Show device information",
        "showdev",
    };

    private final static String[] helpShowcfg = {
        "Show configuration information",
        "showcfg [<device>]",
        "device  Device number from showbus list (default: current device)",
    };

    private final static String[] helpShowif = {
        "Show interface information",
        "showif [<iface>] [<device>]",
        "iface   The interface number (default: the current open one)", 
        "device  Device number from showbus list (default: current device)",
    };

    private final static String[] helpShowep = {
        "Show endpoint information",
        "showep <epnum> [<iface>] [<device>]",
        "epnum   The number of the endpoint to show: 0 = in, 1 = out",
        "iface   The interface number (default: the current open one)", 
        "device  Device number from showbus list (default: current device)",
    };

    private final static String[] helpShowbus = {
        "Show all devices on the USB bus",
        "showbus",
    };

    private final static CmndProc.Command cmnd;
    private final static CmndProc.Lookup
        speedNames, devcNames, cfgaNames, ifccNames, epaNames;
    static {
        cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("find",        CMD_FIND,        helpFind);
        cmnd.add("open",        CMD_OPEN,        helpOpen);
        cmnd.add("close",       CMD_CLOSE,       helpClose);
        cmnd.add("flush",       CMD_FLUSH,       helpFlush);
        cmnd.add("read",        CMD_READ,        helpRead);
        cmnd.add("write",       CMD_WRITE,       helpWrite);
        cmnd.add("showdev",     CMD_SHOWDEV,     helpShowdev);
        cmnd.add("showcfg",     CMD_SHOWCFG,     helpShowcfg);
        cmnd.add("showif",      CMD_SHOWIF,      helpShowif);
        cmnd.add("showep",      CMD_SHOWEP,      helpShowep);
        cmnd.add("showbus",     CMD_SHOWBUS,     helpShowbus);

        speedNames = new CmndProc.Lookup(4);
        speedNames.add("Undefined",    UsbComm.DEV_SPEED_UNDEFINED);
        speedNames.add("Low",          UsbComm.DEV_SPEED_LOW);
        speedNames.add("Full",         UsbComm.DEV_SPEED_FULL);
        speedNames.add("Unknown",      UsbComm.DEV_SPEED_UNKNOWN);

        devcNames = new CmndProc.Lookup(6);
        devcNames.add("See interface", UsbComm.DEV_CLASS_INTERFACE);
        devcNames.add("Comm control",  UsbComm.DEV_CLASS_CDC_CTRL);
        devcNames.add("Hub",           UsbComm.DEV_CLASS_HUB);
        devcNames.add("Diagnostic",    UsbComm.DEV_CLASS_DIAGNOSTIC);
        devcNames.add("Miscellaneous", UsbComm.DEV_CLASS_MISC);
        devcNames.add("Vendor spec",   UsbComm.DEV_CLASS_VENDOR_SPEC);

        cfgaNames = new CmndProc.Lookup(2);
        cfgaNames.add("Self-powered",  UsbComm.CFG_ATTR_SELF_POWER);
        cfgaNames.add("Remote wakeup", UsbComm.CFG_ATTR_REMOTE_WAKE);

        ifccNames = new CmndProc.Lookup(17);
        ifccNames.add("Audio",         UsbComm.IFC_CLASS_AUDIO);
        ifccNames.add("Comm control",  UsbComm.IFC_CLASS_CDC_CTRL);
        ifccNames.add("Human intface", UsbComm.IFC_CLASS_HID);
        ifccNames.add("Physical",      UsbComm.IFC_CLASS_PHYSICAL);
        ifccNames.add("Still image",   UsbComm.IFC_CLASS_IMAGE);
        ifccNames.add("Printer",       UsbComm.IFC_CLASS_PRINTER);
        ifccNames.add("Mass storage",  UsbComm.IFC_CLASS_MASS_STORE);
        ifccNames.add("Comm data",     UsbComm.IFC_CLASS_CDC_DATA);
        ifccNames.add("Smart card",    UsbComm.IFC_CLASS_SMART_CARD);
        ifccNames.add("Content secty", UsbComm.IFC_CLASS_CONTENT_SEC);
        ifccNames.add("Video",         UsbComm.IFC_CLASS_VIDEO);
        ifccNames.add("Healthcare",    UsbComm.IFC_CLASS_HEALTHCARE);
        ifccNames.add("Diagnostic",    UsbComm.IFC_CLASS_DIAGNOSTIC);
        ifccNames.add("Wireless",      UsbComm.IFC_CLASS_WIRELESS);
        ifccNames.add("Miscellaneous", UsbComm.IFC_CLASS_MISC);
        ifccNames.add("App specific",  UsbComm.IFC_CLASS_APP_SPEC);
        ifccNames.add("Vendor spec",   UsbComm.IFC_CLASS_VENDOR_SPEC);

        epaNames = new CmndProc.Lookup(4);
        epaNames.add("Control",        UsbComm.EP_ATTR_CONTROL);
        epaNames.add("Isochronous",    UsbComm.EP_ATTR_ISOCHRONOUS);
        epaNames.add("Bulk",           UsbComm.EP_ATTR_BULK);
        epaNames.add("Interrupt",      UsbComm.EP_ATTR_INTERRUPT);
    }

    private final Output out = new ConsOut();
    private final ConsoleReader reader = new ConsoleReader();
    private final CmndProc proc = new CmndProc();
    private final UsbComm com = new UsbComm();
    private final boolean debug;


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public TestUsbComm(boolean dbg) throws UsbException, IOException
    {
        debug = dbg;
        proc.add(this, cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        TestUsbComm tcom = null;

        try {
            tcom = new TestUsbComm(args.length > 0);
            tcom.run();
        }
        catch (UsbException e) {
            System.out.println(e);
        }
        catch (IOException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run() throws IOException
    {
        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            try {
                if (!proc.process(line)) break;
            }
            catch (RuntimeException e) {
                if (debug) e.printStackTrace(System.out);
                else out.println(e);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing, exception-free version
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, Scanner scan)
    {
        try {
            return dispatchCmnd(code, scan);
        }
        catch (UsbException e) {
            out.println(e.toString());
            return true;
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatch command for processing
    **
    ***************************************************************************
    */
    private boolean dispatchCmnd(int code, Scanner scan) throws UsbException
    {
        int found;
        Object[] args = new Object[16];

        switch (code) {

        case CMD_FIND:
            if ((found = CmndProc.scanArgs(scan, "IIs", args)) >= 0) {
                String serial = null;
                if ((found & 0x04) != 0) serial = (String)args[2];
                if (com.findDevice((Integer)args[0], (Integer)args[1],
                                   serial) == null)
                    out.println("Device not found");
            }
            break;

        case CMD_OPEN:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                com.open((found & 0x01) != 0 ? (Integer)args[0] : 0,
                         (found & 0x02) != 0 ? (Integer)args[1] : 0,
                         (found & 0x04) != 0 && (Integer)args[2] != 0);
            }
            break;

        case CMD_CLOSE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                com.close();
            }
            break;

        case CMD_FLUSH:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println(com.flush() + " bytes flushed");
            }
            break;

        case CMD_READ:
            if ((found = CmndProc.scanArgs(scan, "If", args)) >= 0) {
                int count = (Integer)args[0];
                float timeout = 1f;
                if ((found & 0x02) != 0) timeout = (Float)args[1];
                byte[] data = new byte[count];
                int nRead = com.read(data, (int)(1000f * timeout));
                dispData(data, nRead);
            }
            break;

        case CMD_WRITE:
            if ((found = CmndProc.scanArgs(scan, "ssssssss", args)) >= 0) {
                int count = 0, mask = 0;
                ArrayList list = new ArrayList();
                for (int j = 0; j < 8; j++) {
                    if ((found & (1 << j)) == 0) continue;
                    String hex = (String)args[j];
                    int leng = hex.length();
                    if ((leng & 1) != 0) break;
                    byte[] sData = new byte[leng / 2];
                    try {
                        for (int k = 0; k < leng; k += 2) {
                            String num = hex.substring(k, k + 2);
                            sData[k / 2] = (byte)Integer.parseInt(num, 16);
                        }
                    }
                    catch (NumberFormatException e) {
                        break;
                    }
                    count += leng / 2;
                    list.add(sData);
                    mask |= (1 << j);
                }
                if (mask != found) {
                    out.println("Invalid hexadecimal string");
                    break;
                }
                byte[] data = new byte[count];
                int l = 0;
                for (int j = 0; j < list.size(); j++) {
                    byte[] sData = (byte [])list.get(j);
                    for (int k = 0; k < sData.length; k++) {
                        data[l++] = sData[k];
                    }
                }
                int nWrite = com.write(data);
            }
            break;

        case CMD_SHOWDEV:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                UsbDevice dev = com.getDevice();
                if (dev != null)
                    showDevice(dev);
                else
                    out.println("No device active");
            }
            break;

        case CMD_SHOWBUS:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                List devs = UsbComm.findDevices(-1, -1, null);
                if (!devs.isEmpty()) {
                    for (int j = 0; j < devs.size(); j++) {
                        UsbDevice dev = (UsbDevice)devs.get(j);
                        if (j > 0) out.println();
                        out.println("Device " + j + ":");
                        showDevice(dev);
                    }
                }
                else
                    out.println("No devices found");
            }
            break;

        case CMD_SHOWCFG:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) == 0) {
                    UsbConfiguration cfg = com.getConfiguration();
                    if (cfg != null)
                        showConfiguration(cfg);
                    else
                        out.println("No device active");
                }
                else {
                    UsbDevice dev = getDevice((Integer)args[0]);
                    if (dev != null)
                        showConfiguration(dev.getActiveUsbConfiguration());
                    else
                        out.println("Device not found");
                }
            }
            break;

        case CMD_SHOWIF:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                int inum = (found & 0x01) != 0 ? (Integer)args[0] : 0;
                int dnum = (found & 0x02) != 0 ? (Integer)args[1] : 0;
                UsbInterface ifc = getInterface(found, inum, dnum);
                if (ifc != null)
                    showInterface(ifc);
            }
            break;

        case CMD_SHOWEP:
            if ((found = CmndProc.scanArgs(scan, "Iii", args)) >= 0) {
                int inum = (found & 0x02) != 0 ? (Integer)args[1] : 0;
                int dnum = (found & 0x04) != 0 ? (Integer)args[2] : 0;
                UsbInterface ifc = getInterface(found >> 1, inum, dnum);
                if (ifc != null) {
                    UsbEndpoint ep = UsbComm.getEndpoint(ifc, (Integer)args[0]);
                    if (ep != null)
                        showEndpoint(ep);
                    else
                        out.println("Endpoint not found");
                }
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Get a device using its index in the system
    **
    ***************************************************************************
    */
    private UsbDevice getDevice(int index) throws UsbException
    {
        List devs = UsbComm.findDevices(-1, -1, null);
        if (index >= 0 && index < devs.size())
            return (UsbDevice)devs.get(index);
        else
            return null;
    }


   /**
    ***************************************************************************
    **
    **  Get an interface
    **
    ***************************************************************************
    */
    private UsbInterface getInterface(int mask, int inum, int dnum)
        throws UsbException
    {
        UsbInterface ifc = null;
        if ((mask & 0x02) == 0) {
            if ((mask & 0x01) == 0) {
                ifc = com.getInterface();
                if (ifc == null)
                    out.println("Device not open");
            }
            else {
                ifc = com.getInterface(inum);
                if (ifc == null)
                    out.println("Interface not found");
            }
        }
        else {
            UsbDevice dev = getDevice(dnum);
            if (dev != null) {
                int iface = ((mask & 0x01) != 0) ? inum : 0;
                ifc = UsbComm.getInterface(dev, iface);
                if (ifc == null)
                    out.println("Interface not found");
            }
            else
                out.println("Device not found");
        }

        return ifc;
    }


   /**
    ***************************************************************************
    **
    **  Display read data
    **
    ***************************************************************************
    */
    private void dispData(byte[] data, int count)
    {
        String cnt = count + " bytes read";
        String pad = String.format("%24s", "").substring(0, cnt.length() + 1);
        out.print(cnt);
        for (int j = 0; j < count; j++) {
            if ((j & 0x0f) == 0) {
                if (j == 0) out.print(":");
                else {
                    out.println();
                    out.print(pad);
                }
            }
            out.format(" %02x", data[j] & 0xff);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Show device information
    **
    ***************************************************************************
    */
    private void showDevice(UsbDevice dev) throws UsbException
    {
        UsbDeviceDescriptor desc = dev.getUsbDeviceDescriptor();
        out.format("Vendor ID (name) : 0x%04x (%s)\n",
                   desc.idVendor(), dev.getManufacturerString());
        out.format("Product ID (name): 0x%04x (%s)\n",
                   desc.idProduct(), dev.getProductString());
        out.println("Serial Number    : " + dev.getSerialNumberString());
        out.println("Device Speed     : "
                      + speedNames.decode(UsbComm.getSpeed(dev)));
        int cls = desc.bDeviceClass() & 0xff;
        out.format("Device class     : %s (%s)\n", cls, devcNames.decode(cls));
        out.println("Device Subclass  : " + desc.bDeviceSubClass());
        out.println("Device Protocol  : " + desc.bDeviceProtocol());
        out.format("USB Release      : %04x\n", desc.bcdUSB());
        out.println("Number Configs   : " + desc.bNumConfigurations());
        out.println("Maxm packet size : " + desc.bMaxPacketSize0());
    }


   /**
    ***************************************************************************
    **
    **  Show configuration information
    **
    ***************************************************************************
    */
    private void showConfiguration(UsbConfiguration cfg) throws UsbException
    {
        UsbConfigurationDescriptor desc = cfg.getUsbConfigurationDescriptor();
        out.format("ConfigNum (name) : %s (%s)\n",
                   desc.bConfigurationValue(), cfg.getConfigurationString());
        int attr = desc.bmAttributes() & 0xff;
        String sAttr = "";
        for (int j = 0; j < cfgaNames.count(); j++) {
            int code = cfgaNames.code(j);
            if ((attr & code) != 0) {
                if (!sAttr.equals("")) sAttr += ", ";
                sAttr += cfgaNames.name(j);
            }
        }
        out.format("Attributes       : 0x%02x (%s)\n", attr, sAttr);
        out.format("Maximum power    : %s mA\n", desc.bMaxPower());
        out.println("Number interfaces: " + desc.bNumInterfaces());
    }


   /**
    ***************************************************************************
    **
    **  Show interface information
    **
    ***************************************************************************
    */
    private void showInterface(UsbInterface ifc) throws UsbException
    {
        UsbInterfaceDescriptor desc = ifc.getUsbInterfaceDescriptor();
        out.format("Interface (name) : %s (%s)\n",
                   desc.bInterfaceNumber(), ifc.getInterfaceString());
        out.println("Alt. setting     : " + desc.bAlternateSetting());
        int cls = desc.bInterfaceClass() & 0xff;
        out.format("Interface class  : %s (%s)\n", cls, ifccNames.decode(cls));
        out.println("Intface subclass : " + desc.bInterfaceSubClass());
        out.println("Intface protocol : " + desc.bInterfaceProtocol());
        out.println("Number endpoints : " + desc.bNumEndpoints());
    }


   /**
    ***************************************************************************
    **
    **  Show endpoint information
    **
    ***************************************************************************
    */
    private void showEndpoint(UsbEndpoint ep)
    {
        UsbEndpointDescriptor desc = ep.getUsbEndpointDescriptor();
        out.format("Address          : 0x%02x\n",
                   desc.bEndpointAddress());
        int attr = desc.bmAttributes() & 0xff;
        out.format("Attributes       : 0x%02x (%s)\n",
                   attr, epaNames.decode(attr & 0x03));
        out.format("Interval         : %s msec\n",
                   desc.bInterval());
        out.println("Maxm packet size : " + desc.wMaxPacketSize());
    }

}
