package org.lsst.ccs.drivers.twistorr;

import java.util.logging.Logger;
import java.util.logging.Level;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.SupportedOption;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Option;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.TestAscii;
import java.util.Date;
import java.time.Instant;

/**
 *  Program to test the TwisTorr turbo pump device driver.
 *
 *  @author Al Eisner with other CCS Team contributions
 */

public class TestTT extends TestAscii {

    private final TwisTorr ttdev;
    private double pulseTime = 0.0;

    private Thread openMoment;
    private Boolean ventValveOpen = false;

    private static final Logger LOG = Logger.getLogger(TestTT.class.getName());

    protected volatile TwisTorr.CntlrModels model;
    private final SupportedOption isoOption = SupportedOption.getSupportedOption("iso");
    private final SupportedOption forceOption = SupportedOption.getSupportedOption("force");

   /**
    *  Constructor
    */
    public TestTT()
    {
        super(new TwisTorr());
        ttdev = (TwisTorr)dev;
    }


    /**
     * Determines current time
     *
     * @param options    The provided command options
     * @return current time as a String
     */
    @Command(name="timestamp", description="Prints current time, optionally in iso-8601")
    @Option(name = "iso", description = "format the date as iso-8601=s")
    public String timestamp(Options options)
    {
        boolean hasIso= options.hasOption(isoOption);
        Date now = new Date();
        if (hasIso) {
            return Instant.now().toString();
        } else {
            return new Date().toString();
        }
    }

   /**
    *  Set model
    * 
    *  @param 
    *  @throws
    */
    @Command(name="setModel", description="Set Controller Model")
    public String setModel(@Argument(name="FS74OB|FS84OB|FS304OB|FS305OB",
                            description="which model") String model_value) throws DriverException 
    {
        if (model_value.matches("FS74OB")) {
            setModel(TwisTorr.CntlrModels.FS74OB);
        } else if (model_value.matches("FS84OB")) {
            setModel(TwisTorr.CntlrModels.FS84OB);
        } else if (model_value.matches("FS304OB")) {
            setModel(TwisTorr.CntlrModels.FS304OB);
        } else if (model_value.matches("FS305OB")) {
            setModel(TwisTorr.CntlrModels.FS305OB);
        } else {
            return "Unknown Model";
        }
        return this.model.toString();
    }

   /**
    *  Opens connection to a device.
    * 
    *  @param  ident     The identification (serial port name)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        ttdev.open(ident);
    }

   /**
    * Read all settings and data from pump controller.
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    * @throws InterruptedException
    */
    @Command(name="readAll", description="Read all supported controller settings and data")
    @Option(name = "force", description = "Read all known registers, regardless of controller model")
    public String readAll(Options options) throws InterruptedException
    {
        long startTime = System.currentTimeMillis();
        boolean forceRead = options.hasOption(forceOption);
        //String table = "Read all turbo-pump settings and data\n" + timestamp() +"\n";
        String table = "Read all turbo-pump settings and data\n" + Instant.now().toString() +"\n";
        table += String.format("Controller Model Set to: %s", this.model.toString());

        /* Boolean commands */

        TwisTorr.CmndBool cmndB[] = TwisTorr.CmndBool.values();
        int nB = cmndB.length;
        for (int i = 0; i < nB; i++) {
            if (forceRead || (model.getModelMask() & cmndB[i].getModels()) != 0) {
                table += String.format("\n%3s    %-22s", cmndB[i].getWindow(), cmndB[i]);
                try {
                    boolean respB = ttdev.readBool(cmndB[i]);
                    table += Boolean.toString(respB);
                } catch (DriverException ex) {
                    table += ex.getMessage();
                }
            }
        }
        table += "\n";

        /* Numeric commands */

        TwisTorr.CmndNumeric cmndN[] = TwisTorr.CmndNumeric.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            if (forceRead || (model.getModelMask() & cmndN[i].getModels()) != 0) {
                table += String.format("\n%3s    %-22s", cmndN[i].getWindow(), cmndN[i]);
                try {
                    int respN = ttdev.readNumeric(cmndN[i]);
                    table += Integer.toString(respN);
                } catch (DriverException ex) {
                    table += ex.getMessage();
                }
            }
        }
        table += "\n";

        TwisTorr.CmndAlpha cmndA[] = TwisTorr.CmndAlpha.values();
        int nA = cmndA.length;
        for (int i = 0; i < nA; i++) {
            if (forceRead || (model.getModelMask() & cmndA[i].getModels()) != 0) {
                table += String.format("\n%3s    %-22s", cmndA[i].getWindow(), cmndA[i]);
                try {
                    String respA = ttdev.readAlpha(cmndA[i]);
                    table += respA;
                } catch (DriverException ex) {
                    table += ex.getMessage();
                }
            }
        }
        long dt = System.currentTimeMillis() - startTime;
        table += String.format("\nreadAll() time: %d\n", dt);

        return table;
    }

   /** 
    *  Generic read command for boolean quantities
    *
    *  @param   Enumerated command name
    *  @return  String showing response
    *  @throws  DriverException
    */
    @Command(name="readBool", description="Read specified boolean quantity")
    public String readBool(@Argument(name="enum quantity", 
     description="name of boolean quantity") TwisTorr.CmndBool quantity) 
     throws DriverException
    {
        boolean value = ttdev.readBool(quantity);
        return Boolean.toString(value);
    }

   /** 
    *  Generic read command for numberic (integer) quantities
    *
    *  @param   Enumerated command name
    *  @return  String showing response
    *  @throws  DriverException
    */
    @Command(name="readNumeric", description="Read specified numeric quantity")
    public String readNumeric(@Argument(name="enum quantity", 
     description="name of numeric quantity") TwisTorr.CmndNumeric quantity) 
     throws DriverException
    {
        int value = ttdev.readNumeric(quantity);
        return Integer.toString(value);
    }

   /** A few single-item read commands */

   /**
    *  Read pump temperature
    *
    *  @return  String showing pump temperature in degrees C
    *  @throws  Driver Exception
    */
    @Command(name="readTemp", description="Read pump temperature in degrees C")
    public String readTemp()  throws DriverException
    {
        int temp = ttdev.readNumeric(TwisTorr.CmndNumeric.PUMP_TEMP);
        return Integer.toString(temp);
    }

   /**
    *  Repeat read of pump temperature n times
    *
    *  @param   Number of times to repeat reading
    *  @return  Formatted String containing all read results
    *  @throws  DriverException
    */
    @Command(name="readTempRepeat", description="Read pump T multiple times")
    public String readTempRepeat(@Argument(name="repeats",
       description="number of times") int nRepeat) throws DriverException
    {
        String output = "";
        for (int i = 0; i < nRepeat; i++) {
            output += (" " + readTemp());
        }
        return output;
    }

   /**
    *  Read active-stop setting
    *
    *  @return  String showing active-stop state
    *  @throws  Driver Exception
    */
    @Command(name="readActiveStop", description="Read active-stop setting")
    public String readActiveStop()  throws DriverException
    {
        boolean as = ttdev.readBool(TwisTorr.CmndBool.ACTIVE_STOP_MODE);
        return Boolean.toString(as);
    }

   /** 
    *  Read and decode pump status
    *
    *  @return  String indicating pump state or failure
    *  @throws  DriverException
    */
    @Command(name="readStatus", description="Get pump status, including operating modes and failure")
    public TwisTorr.PumpStatus readStatus()  throws DriverException
    {
        int status = ttdev.readNumeric(TwisTorr.CmndNumeric.STATUS);
        return TwisTorr.PumpStatus.decodeStatus(status);
    }

   /** 
    *  Read error code (bits show reason for pump failure)
    *
    *  @return  String giving error code in hexadecimal and decoding
    *  @throws  DriverException
    */
    @Command(name="readErrorCode", description="Read pump error code")
    public String readErrorCode()  throws DriverException
    {
        int errcode = ttdev.readNumeric(TwisTorr.CmndNumeric.ERRCODE);
        return Integer.toString(errcode,16) + TwisTorr.decodeError(errcode);
    }


   /** 
    *  Test error code (bits show reason for pump failure)
    *
    *  @param   Test code (0 to 0xff) to be decoded
    *  @return  String giving test code in hexadecimal and decoding
    */
    @Command(name="testErrorCode", description="Test decoding of error code")
        public String testErrorCode(@Argument(name="testcode", description="code to test, 0 to 0xff") int testcode) 
    {
        return Integer.toString(testcode,16) + TwisTorr.decodeError(testcode);
    }


   /** Pump-operation settings, Boolean commands */

    // Several writeBoool and writeNumeric ("set")  Commands are also
    // imp[ementee as "verify" Commands, to test writeAndVeriify versions.

   /**
    *  Start pump
    * 
    *  @throws  DriverException
    */
    @Command(name="startPump", description="Start the pump")
    public void startPump() throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.START_STOP, true);
    }

   /**
    *  Stop pump
    * 
    *  @throws  DriverException
    */
    @Command(name="stopPump", description="Stop the pump")
    public void stopPump() throws DriverException {
        ttdev.writeBool(TwisTorr.CmndBool.START_STOP, false);
    }

   /**
    *  Select speed mode (low vs. normal)
    * 
    *  @param value  True for low-speed mode, false for normal mode.
    *  @throws  DriverException
    */
    @Command(name="setLowSpeedMode", description="Low-speed mode on/off")
    public void setLowSpeedMode(@Argument(name="<true|false>",description=
     "Low-speed mode true or false") boolean value) throws DriverException 
    {
        ttdev.writeBool(TwisTorr.CmndBool.LOW_SPEED_MODE, value);
    }

   /**
    *  Select soft start vs. normal start
    * 
    *  @param value  True for soft-start mode, false for normal mode.
    *  @throws  DriverException
    */
    @Command(name="setSoftStart", description="Soft-start mode on/off")
    public void setSoftStart(@Argument(name="<true|false>",description=
     "Soft-start mode true or false") boolean value) throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.SOFT_START_MODE, value);
    }

   /**
    *  Select active stop (with braking) vs. normal stop mode
    * 
    *  @param value  True for active-stop mode, false for normal-stop mode.
    *  @throws  DriverException
    */
    @Command(name="setActiveStop", description="Active-stop mode on/off")
    public void setActiveStop(@Argument(name="<true|false>",description=
     "Active-stop mode true or false") boolean value) throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.ACTIVE_STOP_MODE, value);
    }

   /**
    *  Select "Remote" (I/O) mode vs. Serial mode for controller
    * 
    *  @param value  True for remote mode, false for serial mode.
    *  @throws  DriverException
    */
    @Command(name="setRemote", description="Select remote vs. serial mode")
    public void setRemote(@Argument(name="<true|false>",description=
     "Remote I/O mode (true) vs. Serial mode (false)") boolean value) throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.REMOTE, value);
    }

   /**
    *  Set water cooling on or off
    * 
    *  @param value  True/false for water cooling on/off
    *  @throws  DriverException
    */
    @Command(name="setWaterCooling", description="Water cooling on/off")
    public void setWaterCooling(@Argument(name="<true|false>",description=
     "Water cooling true or false") boolean value) throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.WATER_COOLING, value);
    }

   /**
    *  Set signal-type for interlock
    * 
    *  @param value  True/false for continuous signal / impulse signal
    *  @throws  DriverException
    */
    @Command(name="setInterlockType", description="Interlock signal continuos vs. impulse")
    public void setInterlockType(@Argument(name="<true|false>",description=
     "Continuous (true, default) or impulse (false)") boolean value) 
     throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.INTERLOCK_TYPE, value);
    }

   /**
    *  Enable/disable pump-speed reading after Stop
    * 
    *  @param value  True/false for enable/disable
    *  @throws  DriverException
    */
    @Command(name="enableSpeedRead", description="Enable/disable pump-speed read after Stop")
    public void enableSpeedRead(@Argument(name="<true|false>",description=
     "Enable (true) or disable(false, default)") boolean value) 
     throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.SPEED_READ_ACTIVATE, value);
    }

   /**
    *  Select gas load type
    * 
    *  @param value   0 for N, 1 for Ar
    *  @throws  DriverException
    */
    @Command(name="setGasType", description="Select gas load type")
    public void setGasType(@Argument(name="<int>",
     description="false for N2, true for Ar") boolean value)
     throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.GAS_TYPE_ARGON, value);
    }


   /**
    *  Select gas load type
    * 
    *  @param value   0 for N, 1 for Ar
    *  @throws  DriverException
    */
    @Command(name="setGas", description="Select gas")
    public void setGas(@Argument(name="<int>",
     description="0 for Ar, 1 for N2, 2 for He") int value)
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.GAS_TYPE, value);
    }



   /** Pump-operation settings, Numeric commands */

   /**
    *  Select baud rate, 600 to 9600 (default)
    * 
    *  @param value   0 to 4 for 600, 1200, 2400, 4800, 9600
    *  @throws  DriverException
    */
    @Command(name="setBaudRate", description="Select baud rate")
    public void setBaudRate(@Argument(name="<int>",
     description="0 to 4 for 600, 1200, 2400, 4800, 9600 (default)") 
     int value) throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.BAUD_RATE, value);
    }

   /**
    *  Set normal roational frequency
    * 
    *  @param value   1100 to 1350 (default)
    *  @throws  DriverException
    */
    @Command(name="setRotFreq", description="Set rotational frequency")
    public void setRotFreq(@Argument(name="<int>",
     description="1100 to 1350 (default) Hz") int value)
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.ROTFREQ_SET, value);
    }

   /**
    *  Set low-speed roational frequency
    * 
    *  @param value   1100 (default) to normal frequency
    *  @throws  DriverException
    */
    @Command(name="setLowFreq", description="Set low-speed rotational freq.")
    public void setLowFreq(@Argument(name="<int>",
     description="1100 Hz (default) to normal freq.") int value)
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.ROTFREQ_LOW, value);
    }

   /** Commands related to vent valve  */

   /**
    *  Vent valve automatic or by command (openVentValve, closeVentValve)
    * 
    *  @param value  True/false for on-command / automatic (default)
    *  @throws  DriverException
    */
    @Command(name="setVentOperation", description="Vent valve operation")
    public void setVentOperation(@Argument(name="<true|false>",description=
     "On-command (true) vs. Automatic (false, default)") boolean value) 
     throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.VENTVALVE_BY_CMND, value);
    }

   /**
    *  Vent valve normally-open vs. normally-closed
    * 
    *  @param value  True/false for NO/NC
    *  @throws  DriverException
    */
    @Command(name="setVentValveType", description="Vent valve NO vs NC")
    public void setVentValveType(@Argument(name="<true|false>",description=
     "Normally-open (true) vs. Normally-closed (false)") boolean value) 
     throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.VENTVALVE_TYPE, value);
    }

   /**
    *  Set vent valve true (in on-command mode, setVentOperation above)
    * 
    *  @throws  DriverException
    */
    @Command(name="setVentValveTrue", description="Set vent valve true")
    public void setVentValveTrue() throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.VENTVALVE_OPEN, true);
    }

   /**
    *  Set vent valve false (in on-command mode, setVentOperation above)
    * 
    *  @throws  DriverException
    */
    @Command(name="setVentValveFalse", description="Set vent valve false")
    public void setVentValveFalse() throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.VENTVALVE_OPEN, false);
    }

   /**
    *  Set vent-valve opening delay in automatic mode
    * 
    *  @param value   Delay time in units of 0.2 sec  
    *  @throws  DriverException
    */
    @Command(name="setVentDelay", description="Vent valve-open delay")
    public void setVentDelay(@Argument(name="<int>",
     description="Opening delay after stop in auto mode, unit 0.2 s") 
     int value) throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.VENT_DELAY, value);
    }

   /**
    *  Set vent-valve open time for pulsed open
    * 
    *  @param value   Time open in units of secs
    *  @throws  DriverException
    */
    @Command(name="setVentPulseTime", description="set open time for vent pulse")
    public void setVentPulseTime(@Argument(name="<double>",
     description="Open-time in pulse mode, units are seconds (double)")
     double value) throws DriverException
    {
        pulseTime = value;
    }


   /**
    *  Set vent-valve open time in automatic mode
    * 
    *  @param value   Time open in units of 0.2 sec or 0 for infinity
    *  @throws  DriverException
    */
    @Command(name="setVentOpenTime", description="Vent valve-open time")
    public void setVentOpenTime(@Argument(name="<int>",
     description="Open-time in automatic mode, unit 0.2 s, 0 for infinity") 
     int value) throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.VENT_OPENTIME, value);
    }


   /** Commands related to pressure settings
    *  Inclusion postponed, since we do not use an Agilent pressure gauge.
    */

   /* Commands related to external fan */

   /**
    *  External fan by-command vs. always on
    * 
    *  @param value  1/0 for by-command/always on
    *  @throws  DriverException
    */
    @Command(name="setExtFanConfig",description="Fan by-command or always on")
    public void setExtFanConfig(@Argument(name="<1/0>",description=
     "By command (1) vs. Always on (0)") int value) 
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.EXT_FAN_CONFIG, value);
    }

   /**
    *  External fan on/off (if config by-command)
    * 
    *  @param value  True/false for On/Off
    *  @throws  DriverException
    */
    @Command(name="setExtFanOnOff",description="Fan by-command or always on")
    public void setExtFanOnOff(@Argument(name="<true|false>",description=
     "If by-command. On (true) vs. Off (false)") boolean value) 
     throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.EXT_FAN_SETTING, value);
    }

   /* Commands related to programmable setpoint and analog output (on J1) */

   /**
    *  setpoint threshold
    * 
    *  @param value   Value in hz, W or s 
    *  @throws  DriverException
    */
    @Command(name="setpointThresh", description="programmable setpoint threshold")
    public void setpointThresh(@Argument(name="<int>",
     description="setpoint threshold in Hz, W or s") int value) 
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.SETPOINT_THRESH, value);
    }

   /**
    *  setpoint level
    * 
    *  @param value  Active on high level (false) or low-level (true)
    *  @throws  DriverException
    */
    @Command(name="setpointActive", description="setpoint active level")
    public void setpointActive(@Argument(name="<true|false>",description=
     "Active on low-level (true, default) vs. high-level (false)") 
     boolean value) throws DriverException
    {
        ttdev.writeBool(TwisTorr.CmndBool.SETPOINT_ACTIVE, value);
    }

   /**
    *  setpoint delay 
    * 
    *  @param value   Time (s) between pump start and setpoint check
    *  @throws  DriverException
    */
    @Command(name="setpointDelay", description="setpoint-check delay")
    public void setpointDelay(@Argument(name="<int>",
     description="Seconds after pump start (default 0)") int value) 
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.SETPOINT_DELAY, value);
    }

   /**
    *  setpoint hysteresis (for frequency, power, orpressure)
    * 
    *  @param value   In percent of setpoint value
    *  @throws  DriverException
    */
    @Command(name="setpointHyst", description="setpoint hysteresis")
    public void setpointHyst(@Argument(name="<int>",
     description="Hysteresis in % of value (default = 2)") int value) 
     throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.SETPOINT_HYST, value);
    }

   /**
    *  choice of programmable analog output
    * 
    *  @param value   0 to 3 for frequency, power, pump temp., pressure
    *  @throws  DriverException
    */
    @Command(name="selectAnalog", description="select analog output")
    public void selectAnalog(@Argument(name="<int>",
     description="Select analog output (0 to 3, default = 0, frequency)")
     int value) throws DriverException
    {
        ttdev.writeNumeric(TwisTorr.CmndNumeric.ANALOG_OUTPUT, value);
    }


   /**
    *  Reset pump lifetime and cycles
    *
    *  @param confirm  true to confirm user really wants to do this
    *  @throws  DriverException
    */
    @Command(name="resetPumpHoursCyclesIRREVERSIBLE", 
             description="resets pump total hours and count of cycles")
    public void resetPumpHoursCyclesIRREVERSIBLE(@Argument(name="<true|false>",
    description="true to confirm command") boolean confirm)
    throws DriverException
    {
        if (confirm) 
        {
            ttdev.writeBool(TwisTorr.CmndBool.RESET_CYCLE_TIME, Boolean.TRUE);
        }
    }


   /**
    *  Converts a boolean to on/off.
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }


    /*
    *  Timed open vent valve - open vent valve for a predefined period of time
    *
    *  @throws  DriverException                                                                                                                                                        */
    @Command(type=Command.CommandType.ACTION, name="timedOpenVentValve",
             description="open vent valve (certain models only)")
    public void timedOpenVentValve() throws DriverException
    {
        openMoment = new Thread("openMoment") {
                @Override
                public void run() {
                    try {
                        openVentValve();
                    } catch (DriverException ex) {
			try {
			    closeVentValve();
			} catch (DriverException ec) { }
                        throw new RuntimeException("Driver exception occurred when trying to open the turbo vent valve",ex);
                    }

                    try{
                        Thread.sleep((long)(pulseTime * 1000));
                    } catch (InterruptedException ex) {
			try {
			    closeVentValve();
			} catch (DriverException ec) { }
                        throw new RuntimeException("Unexpected interrupt while waiting for turbo vent valve openning period to end",ex);
                    }
                    try {
                        closeVentValve();
                    } catch (DriverException ex) {
                        throw new RuntimeException("Driver exception occurred when trying to close the turbo vent valve",ex);
                    }

                }
            };
        openMoment.setDaemon(true);
        openMoment.start();
    }

    /**
     *  Set the current turbo model
     *
     *  @return turbo mode
     */

    public void setModel(TwisTorr.CntlrModels model) throws DriverException {
        this.model = model;
    }
    
    /**
     *  Open vent valve
     *
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="openVentValve",
             description="open vent valve (certain models only)")
             public void openVentValve() throws DriverException
    {
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_OPEN.getModels()) != 0) {
            boolean type = ttdev.readBool(TwisTorr.CmndBool.VENTVALVE_TYPE);
            ttdev.writeBool(TwisTorr.CmndBool.VENTVALVE_OPEN, type ? true : false);
            ventValveOpen = true;
        }
    }

    /**
     *  Close vent valve
     *
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="closeVentValve",
             description="close vent valve (certain models only)")
             public void closeVentValve() throws DriverException
    {
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_OPEN.getModels()) != 0) {
            boolean type = ttdev.readBool(TwisTorr.CmndBool.VENTVALVE_TYPE);
            ttdev.writeBool(TwisTorr.CmndBool.VENTVALVE_OPEN, type ? false : true);
            ventValveOpen = false;
        }
    }

}
