package org.lsst.ccs.drivers.twistorr;

import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import java.util.logging.Logger;
import java.util.logging.Level;

/**
 *  Driver for Agilent TwisTorr 84 FS AG turbopump Navigator Controller
 *  (models X3509-64000, X3509-64001)  Only the RS-232 (as opposed to
 *  RS-485) option is supported.
 *
 *  @author Al Eisner
 */


public class TwisTorr84 extends Ascii {

    private static final Logger LOG = Logger.getLogger(TwisTorr84.class.getName());
    private final int BAUD_RATE = 9600;
    //private final int TIMEOUT = 200;    // milliseconds
    private final int TIMEOUT = 500;    // milliseconds

    // Expected length of controller response for Write (acknowledgement)
    // or for an error response to any command
    private final static int ACK_RESP_LEN = 6;  // bytes

    // Offset in bytes to data field in write commands and read responses
    private final static int DATA_OFFSET = 6;
    
    // Constant bytes in messages to and from controller
    private final byte STX = 0x2;
    private final byte ETX = 0x3;
    private final byte ADDR = (byte)0x80;

    // Error-code meanings

    private final static String[] errtypes = {"no_connection", "pump_overTemp", "controller_overTemp", "unused", "unused", "overVoltage", "shortCircuit", "overload"};

    // Label identifying source of messages
    private final static String Lbl = "TwisTorr84:  ";

    /**
     *  Enumeration of controller commands with Logical (boolean) data
     */

    public enum CmndBool{
    
        START_STOP         ("000", true, true),
        LOW_SPEED          ("001", true, true),
        REMOTE             ("008", true, true),
        SOFT_START         ("100", true, true),
        SETPOINT_LEVEL     ("104", true, true),
        WATER_COOLING      ("106", true, true),
        ACTIVE_STOP        ("107", true, true),
        INTERLOCK_TYPE     ("110", true, true),
        VENTVALVE_OPEN     ("122", true, true),
        VENTVALVE_BY_CMND  ("125", true, true),
        VENTVALVE_TYPE     ("140", true, true),
        EXT_FAN_SETTING    ("144", true, true),
       	GAS_TYPE_ARGON     ("157", true, true),  
        SPEED_READ_ACTIVATE("167", true, true),

	SERIAL_TYPE        ("504", true, false),  // Disallow selecting RS-485
    
        RESET_CYCLE_TIME   ("109", false, true);
    
        public final static int DATA_LENGTH = 1;                    // bytes
        public final static int READ_RESP_LEN = DATA_LENGTH + 9;    // bytes
        public final static int WRITE_RESP_LEN = ACK_RESP_LEN;      // bytes
        
        private String window;
        private boolean readAllowed;
        private boolean writeAllowed;
        
        CmndBool(String window, boolean readAllowed, boolean writeAllowed) {
            this.window = window;
            this.readAllowed = readAllowed;
            this.writeAllowed = writeAllowed;
        }
        
        public String getWindow() {return window;}
    }
    
    /**
     *  Enumeration of controller commands with Numeric (int) data
     */

    public enum CmndNumeric{
    
        SETPOINT_TYPE  ("101", true, true),
        SETPOINT_VALUE ("102", true, true),
        SETPOINT_DELAY ("103", true, true),
        SETPOINT_HYST  ("105", true, true),
        BAUD_RATE      ("108", true, true),  
        ANALOG_OUTPUT  ("111", true, true),
        ROTFREQ_LOW    ("117", true, true),  
        ROTFREQ_SET    ("120", true, true),  
        VENT_DELAY     ("126", true, true),  
        EXT_FAN_CONFIG ("143", true, true),
        VENT_OPENTIME  ("147", true, true),
        GAS_TYPE       ("157", true, true),
        PRESSURE_FACTOR("161", true, true),
        PRESSURE_UNIT  ("163", true, true),

	RS485_ADDR     ("503", true, false),  //Disallow address != 0
        
        MAXPOWER       ("155", true, false),
        CURRENT        ("200", true, false),
        VOLTAGE        ("201", true, false),
        POWER	       ("202", true, false),
        DRIVEFREQ      ("203", true, false),
        PUMP_TEMP      ("204", true, false),
        STATUS         ("205", true, false),
        ERRCODE        ("206", true, false),
        CONT_TEMP_SINK ("211", true, false),
        CONT_TEMP_AIR  ("216", true, false),
        RPM	       ("226", true, false),
        CYCLE_TIME     ("300", true, false),
        CYCLE_NUMBER   ("301", true, false),
        PUMP_HOURS     ("302", true, false);
        
        public final static int DATA_LENGTH = 6;                    // bytes
        public final static int READ_RESP_LEN = DATA_LENGTH + 9;    // bytes
        public final static int WRITE_RESP_LEN = ACK_RESP_LEN;      // bytes

        private String window;
        private boolean readAllowed;
        private boolean writeAllowed;

        CmndNumeric(String window, boolean readAllowed, boolean writeAllowed) {
            this.window = window;
            this.readAllowed = readAllowed;
            this.writeAllowed = writeAllowed;
        }
        
        public String getWindow() {return window;}
    }
    
    /**
     *  Enumeration of controller commands with Alphanumeric data
     */

    public enum CmndAlpha{
    
        PRESSURE_SET("162", true, true),
    	PRESSURE("224", true, false);

        public final static int DATA_LENGTH = 10;                    // bytes
        public final static int READ_RESP_LEN = DATA_LENGTH + 9;    // bytes
        public final static int WRITE_RESP_LEN = ACK_RESP_LEN;      // bytes

        private String window;
        private boolean readAllowed;
        private boolean writeAllowed;
        
        CmndAlpha(String window, boolean readAllowed, boolean writeAllowed) {
            this.window = window;
            this.readAllowed = readAllowed;
            this.writeAllowed = writeAllowed;
        }
        
        public String getWindow() {return window;}
    
    }

    /**
     *  Enumeration of single-byte acknowledgement or error responses
     */

    public enum ConResp {

        ACK((byte)0x6),
	NACK((byte)0x15),
	BAD_WINDOW((byte)0x32),
	BAD_DATA_TYPE((byte)0x33),
	OUT_OF_RANGE((byte)0x34),
	DISABLED((byte)0x35);

        private byte response;
  
        ConResp(byte response) {
            this.response = response;
        }

        // public byte getResponse() {return response;}

        /**
         *  Method to decode the meaning of a response byte.
         */

        public static String decode(byte code) throws DriverException {
            ConResp[] con = ConResp.values();
            int n = con.length;
            int found = -1;
            for (int i = 0; i < n; i++) {
                if (con[i].response == code) {
                    found = i;
                    break;
                }
            }
            if (found == -1) throw new DriverException(Lbl + "invalid response code");
	    return con[found].toString();
        }
    }

    /**
     *  Enumeration of pump status codes (returned by Read, window 205)
     */

    public enum PumpStatus {

        STOP(0),
	WAIT_INTLK(1),
	STARTING(2),
	AUTO_TUNING(3),
	BRAKING(4),
	NORMAL(5),
	FAIL(6);

        private int status;

        PumpStatus(int status) {this.status = status;}

        public int getStatus() {return status;}

        /**
         *  Method to decode the meaning of a pump-status code,
         *  returned as a String
         */

        public static String decode(int code) throws DriverException {
            return PumpStatus.decodeStatus(code).toString();
        }

        /**
         *  Method to decode (to an enum) the meaning of a pump-status code
         */

        public static PumpStatus decodeStatus(int code) throws DriverException {
            PumpStatus[] stat = PumpStatus.values();
            int n = stat.length;
            int found = -1;
            for (int i = 0; i < n; i++) {
                if (stat[i].status == code) {
                    found = i;
                    break;
                }
            }
            if (found == -1) throw new DriverException(Lbl + "invalid status code");
	    return stat[found];
        }

    }

    /**
     *  Decode error bit-pattern of reason(s) for pump status = FAIL
     */

    public static String decodeError(int errcode) {
        if (errcode == 0) return "  no error";
        String result = "";
        int code = errcode;
	for (int i = 0; i < 8; i++, code>>>=1) {
            if ((code&1)!=0) {
	        result += "  " + errtypes[i];
            }
        }
        return result;
    }

    /**
     *  Constructor.
     */
    public TwisTorr84()
    {
        super(Option.NO_NET);   // Disallow network connections
    }

    /**
     *  Override underlying open method of Ascii class
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate  The baud rate, or 0 for the default (9600)
     *  @param  commParm  The communications parameters (ignored, 0 used)
     *
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int baudRate, 
                     int commParm) throws DriverException
    {
        super.open(connType, ident, baudRate == 0 ? BAUD_RATE : baudRate, 0);
        setTimeout(TIMEOUT);

        /*
         *   Initialize some controller settings
         */

        try {
            // Serial configuration
	    writeAndVerifyBool(CmndBool.REMOTE, Boolean.FALSE);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }

    /**
     *  Open serial connection using default data characteristics
     *
     *  @param  serialName   Serial device name
     *  @throws DriverException
     */
    public void open(String serialName) throws DriverException {
        openSerial(serialName, BAUD_RATE);
    }

    /**
     *  Methods to send commands to controller and obtain response.
     *  Successful responses will be an acknowledgement for a Write,
     *  and returned data for a Read.  Failures should throw a
     *  DriverException or (in case of timeout) a DriverTimeoutException.
     */

    /** Read Logical (boolean) data 
     *
     *  @param  cmnd  enumerated command identifier
     *  @return data returned from controller
     *  @throws DriverException
     */

    public boolean readBool(CmndBool cmnd) throws DriverException {
        if (!cmnd.readAllowed) {
	    throw new DriverException(Lbl + "Read not allowed for " + cmnd.toString());
	}
        String cmndBody = cmnd.getWindow() + String.valueOf(0);
        String cmndText = "Read " +  cmnd.toString();
	byte[] response = respToCommand(cmndBody, cmndText, CmndBool.READ_RESP_LEN);
        String str = new String(response, DATA_OFFSET, 1);
        int istr = Integer.parseInt(str);
        if (istr == 1) {
            return Boolean.TRUE;
        } else {
            return Boolean.FALSE;
        }
    }

    /** Write Logical (boolean) data 
     *
     *  @param  cmnd   enumerated command identifier
     *  @param  bdata  boolean data to write
     *  @throws DriverException
     */

    public void writeBool(CmndBool cmnd, boolean bdata) throws DriverException {
        if (!cmnd.writeAllowed) {
	    throw new DriverException(Lbl + "Write not allowed for " + cmnd.toString());
	}
        int data = 0;
        if (bdata) data = 1;
        String cmndBody = cmnd.getWindow() + String.valueOf(1) + String.valueOf(data);
        String cmndText = "Write " +  cmnd.toString();
	respToCommand(cmndBody, cmndText, CmndBool.WRITE_RESP_LEN);
        return;
    }

    /** Write and verify Logical (boolean) data 
     *
     *  @param  cmnd   enumerated command identifier
     *  @param  bdata  boolean data to write
     *  @throws DriverException
     */

    public void writeAndVerifyBool(CmndBool cmnd, boolean bdata) throws DriverException {
        writeBool(cmnd, bdata);
        boolean value = readBool(cmnd);
        if (value != bdata) {
            throw new DriverException("Value of " + cmnd + " = " + Boolean.toString(value) + " does not match setting = " + Boolean.toString(bdata));
        }
        return;
    }

    /** Read Numeric (int) data 
     *
     *  @param  cmnd  enumerated command identifier
     *  @return data  returned from controller
     *  @throws DriverException
     */

    public int readNumeric(CmndNumeric cmnd) throws DriverException {

        if (!cmnd.readAllowed) {
	    throw new DriverException(Lbl + "Read not allowed for " + cmnd.toString());
	}
        String cmndBody = cmnd.getWindow() + String.valueOf(0);
        String cmndText = "Read " +  cmnd.toString();
	byte[] response = respToCommand(cmndBody, cmndText, CmndNumeric.READ_RESP_LEN);
        String str = new String(response, DATA_OFFSET, CmndNumeric.DATA_LENGTH);
        return Integer.parseInt(str);
    }

    /** Write Numeric (int) data 
     *
     *  @param  cmnd   enumerated command identifier
     *  @param  ndata  integer data to write
     *  @throws DriverException
     */

    public void writeNumeric(CmndNumeric cmnd, int ndata) throws DriverException {

        if (!cmnd.writeAllowed) {
		throw new DriverException(Lbl + "Write not allowed for " + cmnd.toString());
	}
        String cmndBody = cmnd.getWindow() + String.valueOf(1) + String.format("%06d", ndata);
        String cmndText = "Write " +  cmnd.toString();
	respToCommand(cmndBody, cmndText, CmndNumeric.WRITE_RESP_LEN);
        return;
    }

    /** Write and verify Numeric (int) data 
     *
     *  @param  cmnd   enumerated command identifier
     *  @param  ndata  integer data to write
     *  @throws DriverException
     */

    public void writeAndVerifyNumeric(CmndNumeric cmnd, int ndata) throws DriverException {
        writeNumeric(cmnd, ndata);
        int value = readNumeric(cmnd);
        if (value != ndata) {
            throw new DriverException("Value of " + cmnd + " = " + Integer.toString(value) + " does not match setting = " + Integer.toString(ndata));
        }
        return;
    }

    /** Read Alphanumeric data
     *
     *  @param  cmnd  enumerated command identifier
     *  @return data  returned from controller
     *  @throws DriverException
     */

    public String readAlpha(CmndAlpha cmnd) throws DriverException {
        if (!cmnd.readAllowed) {
	    throw new DriverException(Lbl + "Read not allowed for " + cmnd.toString());
	}
        String cmndBody = cmnd.getWindow() + String.valueOf(0);
        String cmndText = "Read " +  cmnd.toString();
	byte[] response = respToCommand(cmndBody, cmndText, CmndAlpha.READ_RESP_LEN);
        String str = new String(response, DATA_OFFSET, CmndAlpha.DATA_LENGTH);
        return str;
    }

    /** Write Alphanumeric data 
     *
     *  @param  cmnd  enumerated command identifier
     *  @param  adata alphanumeric (String) data to write
     *  @throws DriverException
     */

    public void writeAlpha(CmndAlpha cmnd, String adata) throws DriverException {
        if (!cmnd.readAllowed) {
	    throw new DriverException(Lbl + "Write not allowed for " + cmnd.toString());
	}
	if (adata.length() > 10) {
	    throw new DriverException(Lbl + "String exceeds 10 characters " + cmnd.toString());
	}
        String cmndBody = cmnd.getWindow() + String.valueOf(1) + String.format("%-10s", adata);
        String cmndText = "Write " +  cmnd.toString();
	respToCommand(cmndBody, cmndText, CmndAlpha.WRITE_RESP_LEN);
        return;
    }

    /**
     *  Send command to controller and read response.  Both the
     *  command and the response are byte arrays.  
     *
     *  @param  commandBody   Command String before adding STX, ADDR, ETX 
                              and CRC and converting to a byte array
     *                        will be added here)
     *  @param  cmndText      describes command, used as text for an Exception
     *  @param  expRespLen    expected response length in bytes if no error
     *  @return respToCommand the byte array returned, if no Exception
     *
     *  If the response does not time out (DriverTimeoutException thrown by
     *  the AsciiIO software), a DriverException may be thrown if the response
     *  has a CRC bytes do not match a computation, or if the response has
     *  an unexpected length, or if the data returned for a Write
     *  command is an error code instead of an ACK (acknowledge) byte.
     */

    private synchronized byte[] respToCommand(String commandBody, String cmndText, int expRespLen) throws DriverException {

        /*  Create command byte array; send it to TwisTorr84 controller.
         *  Note that CRC checksum omits the first (STX) byte, as well
         *  as the last two bytes (where the CRC itself will be stored).
         */

        byte[] command = new byte[commandBody.length() + 5];
        System.arraycopy(commandBody.getBytes(), 0, command, 2, commandBody.length());
        command[0] = STX;
        command[1] = ADDR;
        command[command.length - 3] = ETX;
        byte[] commandCRC = computeCRC(command, 1, 2);
        System.arraycopy(commandCRC, 0, command, command.length - 2, 2);
        flush();
        long startTime = System.currentTimeMillis();
        writeBytes(command);

        /*  Get response from controller.  If response has the expected
         *  length for an acknowledgement (i.e., any response to a write
         *  command, or an error response to a read command), check
         *  the acknowledge/error byte and throw exception if any error.
         */

        byte response[] = read(expRespLen);
        // Debugging:  print response in hexadecimal
        // System.out.println("** Command " + cmndText);
        // String dbg = "** Response (length = " + response.length + "): ";
        // for (int ib = 0; ib < response.length; ib++) {
        //     dbg += " " + String.format("%02x", response[ib]);
        // }
        // System.out.println(dbg);
        // End debug block

        // Note:  check for ETX because in testing this driver with the
        // FS304 on-bard controller, which has fewer commands, it was
        // found that an "unknown window" response was padded with 
        // extra 00 (hex) bytes.
        if (response.length == ACK_RESP_LEN || response[3] == ETX) {
            String status = ConResp.decode(response[2]);
	    if (status != ConResp.ACK.toString()) {
		throw new DriverException(Lbl + "error response " + status + " to " + cmndText);
            }
        }

        long dt = System.currentTimeMillis() - startTime;
        LOG.log(Level.FINER, "{0} took {1} ms", new Object[]{cmndText,dt});
        if ( dt >= 50 ) {
	    // LOG.log(Level.INFO, "{0} took {1} ms", new Object[]{cmndText,dt});        
        }
        return response;
    }

    /**
     *  Read response from controller
     * 
     *  @param respLen  Expected response length (bytes) if command successful
     */

    private byte[] read(int respLen) throws DriverException {

        byte[] buff = new byte[respLen];
        int curr = 0;   // index to next unread byte
        while (true) {
            curr += readBytes(buff, curr);
            try {
                curr += readBytes(buff, curr);
            }
            catch (DriverTimeoutException e) {
                throw new DriverTimeoutException(e + String.format(" -- after %d bytes read", curr));
            }
            if ((curr==respLen || curr==ACK_RESP_LEN) && buff[curr-3] == ETX) {
	        byte[] crcByte = computeCRC(buff, 1, respLen - curr + 2);
                if (crcByte[0] == buff[curr-2] && crcByte[1] == buff[curr-1]) {
                    return buff;
                }
                throw new DriverException(Lbl + "incorrect CRC");
                // for debugging:
                // String buffstr = " ";
                // for (int ibuf=0; ibuf<curr; ibuf++) buffstr+= String.format(" %x,", Byte.toUnsignedInt(buff[ibuf]));
	        // throw new DriverException(Lbl + String.format("incorrect CRC:  expect length=%d, length=%d,\n compute CRC = %x, %x   Buffer read =" + buffstr, respLen, curr, (int)crcByte[0], (int)crcByte[1]));
	    } else if (curr >= Math.max(respLen,ACK_RESP_LEN)) {
                throw new DriverException(Lbl + "response too long");
	    }
        }
    }

    /**
     *  Compute CRC checksum (exclusive or) for byte array.  The result
     *  is treated as two hex characters, which are in turn converted
     *  to and returned as a two-byte array of Ascii characters.
     *
     *  @param array      Byte array
     *  @param omitFirst  Number of bytes at start to omit.
     *  @param omitLast   Number of bytes at end to omit.
     */

    private byte[] computeCRC(byte[] array, int omitFirst, int omitLast) {

        byte CRC = 0;
        int length = array.length;
        if (length - omitFirst - omitLast >0) {
            for (int i = omitFirst; i < array.length - omitLast; i++) {
                CRC ^= array[i];
            }
        }
        String crcString = String.format("%02X", CRC);
        byte[] crcBytes = crcString.getBytes();
        return crcBytes;
    }
}
