package org.lsst.ccs.drivers.tripplitepdu;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * General access routines for the TrippLitePDU PDU device
 *
 * @author Homer Neal
 */
public class TrippLitePDU extends Ascii {

    public static int timeout = 200;
    public static final int DEFAULT_BAUD = 9600;         // default baud rate
    public static final String DEFAULT_DEV = "/dev/ttyS1"; // default serial device

    /**
     * Private data
     */
    private final Terminator terminator = Terminator.CR;

    private final Thread readW = new Thread(new TLReader());
    private boolean readWstarted = false;
    private int last_state = -1; // last channel state seen                            
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open = false;

    /**
     * Inner class to implement device reading thread.
     */
    private class TLReader implements Runnable {

        @Override
        public void run() {
            System.out.println(open);
//            while (open) {
            while (true) {
                if (open) {
                    try {
                        String reply = read();
                        if (reply.contains("Turn Load")) {
                            System.out.println("***** FOUND LOAD STATE *****");
                            last_state = 1;
                            if (reply.indexOf("On") >= 1) {
                                last_state = 0;
                            }
                        }
                        System.out.println(reply);
                        nRead++;
                        nByte += reply.length();
                    } catch (DriverTimeoutException e) {
                        nTimeout++;
                    } catch (DriverException e) {
                        if (open) {
                            System.out.println(e);
//                        System.exit(0);
                        }
                    }
                } else {
                    sleep(0.2);
                }
            }
        }
    }

    /**
     * Opens a connection.
     *
     * Overrides the main 4-argument Ascii open method.
     * 
     * @param connType  The connection type
     * @param devName   The host/device name
     * @param parm      The port/baud rate
     * @param dataChar  The serial data characteristics
     * @throws DriverException
     */
    @Override
    public void open(ConnType connType, String devName, int parm, int dataChar) throws DriverException
    {
        super.open(connType, devName, parm, 0);
        setTimeout(timeout);
    }

    /**
     * Opens a serial connection
     *
     * @param devname
     * @param baud
     * @throws DriverException 
     */
    public void open(String devname, int baud) throws DriverException {
        openSerial(devname, baud);
    }

    /**
     * Opens a serial connection.
     *
     * @param devname
     * @param baud
     * @throws DriverException 
     */
    public void openserial(String devname, int baud) throws DriverException {
        System.out.println("openning connection to the PDU");
        openSerial(devname, baud);
        setTerminator(terminator);

        // get logged in
        System.out.println("Sending returns to get its attention.");
        writeTrippLite("");
        System.out.println("Sending username.");
        writeTrippLite("localadmin");
        System.out.println("Sending password.");
        writeTrippLite("localadmin");
//        reset();
    }

    /**
     * Opens an FTDI connection.
     *
     * @param devname
     * @param baud
     * @param major_version 
     * @throws DriverException 
     */
    public void openftdi(String devname, int baud, int major_version) throws DriverException {
        System.out.println("openning connection to the PDU");
        openFtdi(devname, baud);
        setTerminator(terminator);

        // get logged in
        System.out.println("Sending esc to get its attention.");
        System.out.println("Sending username.");
        writeTrippLite("localadmin");
        System.out.println("Sending password.");
        writeTrippLite("localadmin");
//        reset();
    }

    /**
     * Opens a Network connection.
     * 
     * @param netaddress
     * @param port
     * @param major_version
     * @throws DriverException
     */
    public void opennet(String netaddress, int port, int major_version) throws DriverException {

        System.out.println("openning connection to the PDU");
        openNet(netaddress, port);
        setTerminator(terminator);
        /*
        try {
            Thread.sleep(3000);
        } catch (InterruptedException ex) {
            System.out.println("Sleep was interrupted:" + ex);
        }
         */
        // get logged in
        System.out.println("#1:" + readTrippLite(""));
        sleep(1.0);
        System.out.println("#2:" + readTrippLite());
        System.out.println("#2.1:" + readTrippLite());
        System.out.println("#2.2:" + readTrippLite());
        //	System.out.println("#2.3:"+readTrippLite());

        //	flush();
        System.out.println("Sending username.");
        System.out.println("#2.3:" + readTrippLite("localadmin"));
        //	System.out.println("#2.4:"+readTrippLite());

        sleep(1.0);
        System.out.println("Sending password.");
        System.out.println("#3:" + readTrippLite("localadmin"));

        //	flush();
        //flush();
        System.out.println(readTrippLite(""));
        //        System.out.println("Done");
        try {
            Thread.sleep(3000);
        } catch (InterruptedException ex) {
            System.out.println("Sleep was interrupted:" + ex);
        }
        System.out.println(readTrippLite(""));
        System.out.println("Done");
        //        reset();
        open = true;
        if (!readWstarted) {
            readW.setDaemon(true);
            readW.start();
            readWstarted = true;
        }
        flush();
    }

    /**
     * Closes the connection.
     *
     * @throws DriverException
     */
    @Override
    public void close() throws DriverException {
        super.close();
        open = false;
    }

    /**
     * Gets the TrippLitePDU channel power state.
     *
     * @param chan
     * @return The powered state
     * @throws DriverException
     * @throws DriverTimeoutException
     */
    public boolean isOutletOn(int chan) throws DriverException, DriverTimeoutException {

        setTimeout(timeout); // set SCPI timeout

        writeTrippLite("M"); // main menu
        sleep(1.0);
        flush();
        writeTrippLite("1"); // device manager
        sleep(1.0);
        flush();
        writeTrippLite("5"); // outlet management
        sleep(1.0);
        flush();
        writeTrippLite("1"); // outlet control
        sleep(1.0);
        flush();

        last_state = -1;

        writeTrippLite(String.valueOf(chan)); // select outlet

        for (int i = 0; i < 5; i++) {
            if (last_state > -1) {
                break;
            }
            sleep(1.0);
            flush();
        }

        if (last_state < 0) {
            throw new DriverTimeoutException("Failed to retrieve outlet state.");
        }

        return last_state == 1;
    }

    /**
     * Sets the TrippLitePDU channel power state.
     *
     * @param chan
     * @param state The powered state
     * @throws DriverException
     */
    public void setOutletState(int chan, boolean state) throws DriverException {
        String response = null;
        setTimeout(timeout); // set SCPI timeout

        System.out.println("SetOutletState called for outlet " + chan + " to be set to state " + state);
        writeTrippLite("M");
        sleep(1.0);
        flush();
        writeTrippLite("1");
        sleep(1.0);
        flush();
        writeTrippLite("5");
        sleep(1.0);
        flush();
        writeTrippLite("1");
        sleep(1.0);
        flush();

        last_state = -1;

        writeTrippLite(String.valueOf(chan)); // select outlet

        sleep(1.0);
        flush();

        System.out.println("checking state");

        for (int i = 0; i < 5; i++) {
            if (last_state > -1) {
                break;
            }
            sleep(1.0);
        }

        if (last_state < 0) {
            throw new DriverTimeoutException("Failed to retrieve outlet state.");
        }

        if ((last_state == 1 && state == false)
                || (last_state == 0 && state == true)) {
            System.out.println("toggling to desired state");
            writeTrippLite("3"); // toggle state
            sleep(1.0);
            flush();
            System.out.println("confirming:");
            writeTrippLite("YES"); // confirm
            sleep(1.0);
            flush();
        }
    }

    /**
     * Sets the TrippLitePDU channel power state.
     *
     * @param chan
     * @param state The powered state
     * @throws DriverException
     */
    public void setOutletState2(int chan, boolean state) throws DriverException {
        String response = null;
        System.out.println("SetOutletState called for outlet " + chan + " to be set to state " + state);
        System.out.println("going to main menu:" + readTrippLite("M")); // main menu
        sleep(1.0);
        System.out.println(readTrippLite(""));
        sleep(1.0);
        System.out.println("device manager" + readTrippLite("1")); // device manager
        sleep(1.0);
        System.out.println(readTrippLite(""));
        sleep(1.0);
        System.out.println("load manager" + readTrippLite("5")); // load control
        sleep(2.0);
        System.out.println(readTrippLite(""));
        sleep(2.0);
        System.out.println("control load" + readTrippLite("1")); // control outlet
        sleep(2.0);
        System.out.println(readTrippLite(""));
        sleep(1.0);
        System.out.println("specifying outlet to control");
        System.out.println(response = readTrippLite(String.valueOf(chan))); // select outlet
        sleep(1.0);
        System.out.println(readTrippLite(""));
        sleep(1.0);
        System.out.println("checking state");
        if ((last_state == 1 && state == false)
                || (last_state == 0 && state == true)) {
            System.out.println("toggling to desired state" + readTrippLite("3")); // toggle state
            System.out.println(readTrippLite(""));
            System.out.println("confirming: " + readTrippLite("YES")); // confirm
            System.out.println(readTrippLite(""));
        }
    }

    /**
     * Writes a command.
     *
     * @param command The command to write
     * @throws DriverException
     */
    public synchronized void writeTrippLite(String command) throws DriverException {
        write(command);
    }

    /**
     * Reads a response.
     *
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readTrippLite() throws DriverException {
        return read();
    }

    /**
     * Reads a response after writing a command.
     *
     * @param command The command to write, excluding terminator 
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readTrippLite(String command) throws DriverException {
        writeTrippLite(command);
        return readTrippLite();
    }

    /**
     * Sleep - what a waste
     */
    private void sleep(double secs) {
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            System.out.println("Rude awakening!" + ex);
        }
    }

}
