package org.lsst.ccs.drivers.tripplitepdu;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 **************************************************************************
 **
 ** General access routines for the TrippLitePDU PDU device
 *
 ** @author Homer Neal *
 * **************************************************************************
 */
public class TrippLitePDU extends Ascii {

    public static int timeout = 200;
    public static final int DEFAULT_BAUD = 9600;         // default baud rate
    public static final String DEFAULT_DEV = "/dev/ttyS1"; // default serial device

    /**
     **************************************************************************
     **
     ** Public constants *
     * **************************************************************************
     */
    /**
     **************************************************************************
     **
     ** Private constants *
     * **************************************************************************
     */
    private final static byte CR = 0x0d, LF = 0x0a;
    private String terminator = "\r";
    //    private String terminator = String.valueOf(LF);
    private byte[] buff = new byte[1024];
    private int buffIn, buffOut;
    private int major_version;

    private final Thread readW = new Thread(new TLReader());
    private boolean readWstarted = false;
    private int last_state = -1; // last channel state seen                            
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open = false;

    /**
     **************************************************************************
     **
     ** Constructor * *
     * **************************************************************************
     */
    public TrippLitePDU() throws DriverException {
    }

    public void open(String devname, int port) throws DriverException {
        open(Ascii.CONN_TYPE_SERIAL, devname, port);
        setTimeout(timeout); // set SCPI timeout
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Inner class to implement device reading thread. *
     * **************************************************************************
     */
    private class TLReader implements Runnable {

        @Override
        public void run() {
            System.out.println(open);
//            while (open) {
            while (true) {
                if (open) {
                    try {
                        String reply = read();
                        if (reply.indexOf("Turn Load") >= 0) {
                            System.out.println("***** FOUND LOAD STATE *****");
                            last_state = 1;
                            if (reply.indexOf("On") >= 1) {
                                last_state = 0;
                            }
                        }
                        System.out.println(reply);
                        nRead++;
                        nByte += reply.length();
                    } catch (DriverTimeoutException e) {
                        nTimeout++;
                    } catch (DriverException e) {
                        if (open) {
                            System.out.println(e);
//                        System.exit(0);
                        }
                    }
                } else {
                    Sleep(0.2);
                }
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Opens an FTDI connection. *
     * **************************************************************************
     */
    public void openserial(String serialname, int port) throws DriverException {
        System.out.println("openning connection to the PDU");
        open(Ascii.CONN_TYPE_SERIAL, serialname, port);
        setTimeout(timeout); // set SCPI timeout
        setTerminator(terminator);

        // get logged in
        System.out.println("Sending returns to get its attention.");
        writeTrippLite("");
        System.out.println("Sending username.");
        writeTrippLite("localadmin");
        System.out.println("Sending password.");
        writeTrippLite("localadmin");
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Opens an FTDI connection. *
     * **************************************************************************
     */
    public void openftdi(String serialname, int port, int major_version) throws DriverException {
        System.out.println("openning connection to the PDU");
        setTerminator(terminator);
        open(Ascii.CONN_TYPE_FTDI, serialname, port);
        setTimeout(timeout); // set SCPI timeout

        // get logged in
        System.out.println("Sending esc to get its attention.");
        System.out.println("Sending username.");
        writeTrippLite("localadmin");
        System.out.println("Sending password.");
        writeTrippLite("localadmin");
//        reset();
    }

    /**
     ***************************************************************************
     **
     ** Opens a Network connection.
     *
     * **************************************************************************
     */
    public void opennet(String netaddress, int port, int major_version) throws DriverException {

        System.out.println("openning connection to the PDU");
        setTerminator(terminator);
        setTimeout(timeout); // set SCPI timeout
        open(Ascii.CONN_TYPE_NETWORK, netaddress, port);
        /*
        try {
            Thread.sleep(3000);
        } catch (InterruptedException ex) {
            System.out.println("Sleep was interrupted:" + ex);
        }
         */
        // get logged in
        System.out.println("#1:" + readTrippLite(""));
        Sleep(1.0);
        System.out.println("#2:" + readTrippLite());
        System.out.println("#2.1:" + readTrippLite());
        System.out.println("#2.2:" + readTrippLite());
        //	System.out.println("#2.3:"+readTrippLite());

        //	flush();
        System.out.println("Sending username.");
        System.out.println("#2.3:" + readTrippLite("localadmin"));
        //	System.out.println("#2.4:"+readTrippLite());

        Sleep(1.0);
        System.out.println("Sending password.");
        System.out.println("#3:" + readTrippLite("localadmin"));

        //	flush();
        //flush();
        System.out.println(readTrippLite(""));
        //        System.out.println("Done");
        try {
            Thread.sleep(3000);
        } catch (InterruptedException ex) {
            System.out.println("Sleep was interrupted:" + ex);
        }
        System.out.println(readTrippLite(""));
        System.out.println("Done");
        //        reset();
        open = true;
        if (!readWstarted) {
            readW.setDaemon(true);
            readW.start();
            readWstarted = true;
        }
        flush();
    }

    /**
     **************************************************************************
     **
     ** Closes the connection. * * @throws DriverException
     *
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * *************************************************************************
     */
    @Override
    public void close() throws DriverException {
        super.close();
        open = false;
    }

    /**
     ***************************************************************************
     **
     ** Gets the TrippLitePDU channel power state.
     *
     *
     * @param chan
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public boolean isOutletOn(int chan) throws DriverException, DriverTimeoutException {

        setTimeout(timeout); // set SCPI timeout

        writeTrippLite("M"); // main menu
        Sleep(1.0);
        flush();
        writeTrippLite("1"); // device manager
        Sleep(1.0);
        flush();
        writeTrippLite("5"); // outlet management
        Sleep(1.0);
        flush();
        writeTrippLite("1"); // outlet control
        Sleep(1.0);
        flush();

        last_state = -1;

        writeTrippLite(String.valueOf(chan)); // select outlet

        for (int i = 0; i < 5; i++) {
            if (last_state > -1) {
                break;
            }
            Sleep(1.0);
            flush();
        }

        if (last_state < 0) {
            throw new DriverTimeoutException("Failed to retrieve outlet state.");
        }

        return last_state == 1 ? true : false;
    }

    /**
     ***************************************************************************
     **
     ** Gets the TrippLitePDU channel power state. *
     * **************************************************************************
     */
    public void setOutletState(int chan, boolean state) throws DriverException {
        String response = null;
        setTimeout(timeout); // set SCPI timeout

        System.out.println("SetOutletState called for outlet " + chan + " to be set to state " + state);
        writeTrippLite("M");
        Sleep(1.0);
        flush();
        writeTrippLite("1");
        Sleep(1.0);
        flush();
        writeTrippLite("5");
        Sleep(1.0);
        flush();
        writeTrippLite("1");
        Sleep(1.0);
        flush();

        last_state = -1;

        writeTrippLite(String.valueOf(chan)); // select outlet

        Sleep(1.0);
        flush();

        System.out.println("checking state");

        for (int i = 0; i < 5; i++) {
            if (last_state > -1) {
                break;
            }
            Sleep(1.0);
        }

        if (last_state < 0) {
            throw new DriverTimeoutException("Failed to retrieve outlet state.");
        }

        if ((last_state == 1 && state == false)
                || (last_state == 0 && state == true)) {
            System.out.println("toggling to desired state");
            writeTrippLite("3"); // toggle state
            Sleep(1.0);
            flush();
            System.out.println("confirming:");
            writeTrippLite("YES"); // confirm
            Sleep(1.0);
            flush();
        }
    }

    /**
     ***************************************************************************
     **
     ** Gets the TrippLitePDU channel power state. *
     * **************************************************************************
     */
    public void setOutletState2(int chan, boolean state) throws DriverException {
        String response = null;
        System.out.println("SetOutletState called for outlet " + chan + " to be set to state " + state);
        System.out.println("going to main menu:" + readTrippLite("M")); // main menu
        Sleep(1.0);
        System.out.println(readTrippLite(""));
        Sleep(1.0);
        System.out.println("device manager" + readTrippLite("1")); // device manager
        Sleep(1.0);
        System.out.println(readTrippLite(""));
        Sleep(1.0);
        System.out.println("load manager" + readTrippLite("5")); // load control
        Sleep(2.0);
        System.out.println(readTrippLite(""));
        Sleep(2.0);
        System.out.println("control load" + readTrippLite("1")); // control outlet
        Sleep(2.0);
        System.out.println(readTrippLite(""));
        Sleep(1.0);
        System.out.println("specifying outlet to control");
        System.out.println(response = readTrippLite(String.valueOf(chan))); // select outlet
        Sleep(1.0);
        System.out.println(readTrippLite(""));
        Sleep(1.0);
        System.out.println("checking state");
        if ((last_state == 1 && state == false)
                || (last_state == 0 && state == true)) {
            System.out.println("toggling to desired state" + readTrippLite("3")); // toggle state
            System.out.println(readTrippLite(""));
            System.out.println("confirming: " + readTrippLite("YES")); // confirm
            System.out.println(readTrippLite(""));
        }
    }

    /**
     **************************************************************************
     **
     ** Writes a command.
     *
     ** @param The command to write
     ** @throws DriverException *
     * *************************************************************************
     */
    public synchronized void writeTrippLite(String command) throws DriverException {
        write(command);
    }

    /**
     **************************************************************************
     **
     ** Reads a response.
     *
     ** @return The command response string
     ** @throws DriverException
     *
     * *************************************************************************
     */
    public synchronized String readTrippLite() throws DriverException {
        return read();
    }

    /**
     **************************************************************************
     **
     ** Reads a response after writing a command. * * @param command The
     * command to write, excluding terminator * * @return The command response
     * string * * @throws DriverException
     *
     ** @throws DriverTimeoutException *
     * *************************************************************************
     */
    public synchronized String readTrippLite(String command) throws DriverException {
        writeTrippLite(command);
        return readTrippLite();
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            System.out.println("Rude awakening!" + ex);
        }
    }

}
