package org.lsst.ccs.drivers.thorlabs;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import java.util.Date;

/**
 *  Program to test the Thorlabs SC10 shutter driver
 */

public class TestSC10 {

    private final ThorlabsSC10 thsc;

   /**
    *  Constructor
    */
    public TestSC10()
    {
        thsc = new ThorlabsSC10();
    }


    /**
     * Determines current time
     *
     * @return current time as a String
     */
    @Command(name="timestamp", description="Prints current time")
    public String timestamp() {
	Date now = new Date();
        return now.toString();
    }


   /**
    *  Opens connection to a device
    * 
    *  @param  ident     The identification (serial port name)
    *  @param  baudRate  BAUD rate (9600 or 115200)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident, @Argument(name="baudRate",
		     description="Baud rate, 9600 or 115200") int baudRate) 
                     throws DriverException
    {
        thsc.open(ident, baudRate);
    }

   /**
    *  Opens connection to a device using default baud rate
    * 
    *  @param  ident     The identification (serial port name)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        thsc.open(ident);
    }

   /**
    *  Closes connection
    * 
    *  @throws  DriverException
    */
    @Command(name="close", description="Closes connection to device")
    public void close() throws DriverException {
        thsc.close();
    }


    /* Query commands */

   /**
    * Read all settings and data from shutter.
    *
    * Loops over all read commands and returns them in a table format
    * (Raw responses, no decoding; for latter, see additional Read methods.)
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(name="readAll", description="Read all controller settings and data")
    public String readAll()
    {

        String table = "Read all shutter settings and data\n" + timestamp() +"\n";

        ThorlabsSC10.CmndSC cmndN[] = ThorlabsSC10.CmndSC.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                String respN = thsc.querySC(cmndN[i]);
                table += respN;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";
        return table;
    }

   /**
    *  Read shutter position.
    *
    *  @return  String showing whether shutter is open (0) or closed (1)
    *  @throws  DriverException
    */
    @Command(name="readSCpos", description="Read shutter position, <0|1> = <open|closed>")
    public String readSCpos()  throws DriverException
    {
        return thsc.querySC(ThorlabsSC10.CmndSC.CLOSED) + 
	    "   (<0|1>  = <open|closed>)";
    }

   /**
    *  Read and decode the settings for operation and trigger modes
    *
    *  @return  String showing decoded modes
    *  @throws  DriverException
    */
    @Command(name="readMode",description="Decode operation and trigger modes")
	public String readMode() throws DriverException
    {
        String opstring = ThorlabsSC10.OpMode.decode(thsc.querySC(ThorlabsSC10.CmndSC.OPERATING_MODE));
        String trstring = ThorlabsSC10.TrigMode.decode(thsc.querySC(ThorlabsSC10.CmndSC.TRIGGER_MODE));
        return "Operating Mode = "+ opstring + ",  Trigger Mode = "+ trstring;
    }

   /**
    *  List codes for operating modes
    *
    *  @return  List of numerical codes for operating modes
    */
    @Command(name="listOpModes", description="List option codes for operating modes")
    public String listOpModes()
    {
        return ThorlabsSC10.OpMode.listModes();
    }
 

   /* Shutter commands to set modes and mode properties */

   /**
    *  Set trigger mode (internal or external)
    * 
    *  @param   Mode <0|1> for <internal|external>
    *  @throws  DriverException
    */
    @Command(name="setTrigMode", description="Set trigger mode")
    public void setTrigMode(@Argument(name="<0|1>",
     description="trigger mode <internal|external>") int value) 
     throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.TRIGGER_MODE, value);
    }

   /**
    *  Set operating mode (five choices).  This and trigger mode together
    *  define the shutter operation.
    * 
    *  @param   Mode (1 to 5); use listOpModes command to see choices
    *  @throws  DriverException
    */
    @Command(name="setOpMode", description="Set operating mode")
    public void setOpMode(@Argument(name="<1|2|3|4|5>",
     description="operating mode (issue listOpModes for meanings)") int value) 
     throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.OPERATING_MODE, value);
    }

   /**
    *  Set meaning of external TTL trigger output
    * 
    *  @param   Mode <0|1> for output follows <shutter|controller> output>
    *  @throws  DriverException
    */
    @Command(name="setOutputTrig", description="Set trigger mode")
    public void setOutputTrig(@Argument(name="<0|1>",
     description="trigger output follows <shutter|controller>") int value) 
     throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.TRIG_OUT_MODE, value);
    }

   /**
    *  Set repeat count for REPEAT operating mode.
    * 
    *  @param   Repeats, 1 to 99
    *  @throws  DriverException
    */
    @Command(name="setRepeatCount", description="Set count for repeat mode")
    public void setRepeatCount(@Argument(name="1-99 repeats",
     description="in REPEAT mode") int value) throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.REPEAT_COUNT, value);
    }

   /**
    *  Set duration of shutter open in ms
    *  (relevant to SINGLE, REPEAT and AUTO modes)
    * 
    *  @param   Six digits or fewer
    *  @throws  DriverException
    */
    @Command(name="setOpenDuration", description="Set shutter-open duration")
    public void setOpenDuration(@Argument(name="1-999999 ms",
     description="shutter-open duration if SINGLE, REPEAT, AUTO") int value) 
     throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.OPEN_DURATION, value);
    }

   /**
    *  Set duration of shutter closed in ms
    *  (relevant to REPEAT and AUTO modes)
    * 
    *  @param   Six digits or fewer
    *  @throws  DriverException
    */
    @Command(name="setClosedDuration", description="Set shutter-closed duration")
    public void setClosedDuration(@Argument(name="1-999999 ms",
     description="shutter-closed duration if REPEAT or AUTO mode") int value) 
     throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.SHUT_DURATION, value);
    }


    /* Other commands */

   /** 
    *  Command to toggle "enable".  
    *  In the internal trigger mode, "enable" either directly controls the
    *  opening and closing of the shutter (manual operating mode) or
    *  initiates a sequence of operations.  
    *  In the external trigger mode, "enable" enables that trigger.
    * 
    *  @param   Position to move to (1 to maximum)
    *  @throws  DriverException
    */
    @Command(name="toggleEnable", 
             description="Initiate an internal trig or enable external trig")
     public void toggleEnable() throws DriverException 
    {
        thsc.enable();
    }

   /**
    *  Set baud rate
    * 
    *  @param   0 (9600) or 1 (115200)
    *  @throws  DriverException
    */
    @Command(name="setDataRate", 
             description="Set baud rate.  Danger: last-resort reset only!")
     public void setDataRate(@Argument(name=" <0|1>", 
     description="baud rate <9600|115200>") int value) 
     throws DriverException 
    {
        thsc.setSC(ThorlabsSC10.CmndSC.DATA_RATE, value);
    }

   /**
    *  Save (some?) settings to non-volatile flash memory
    * 
    *  @throws  DriverException
    */
    @Command(name="saveSettings", 
             description="Save some settings to non-volatile flash memory")
    public void saveSettings() throws DriverException 
    {
        thsc.saveSC();
    }

   /**
    *  Save some settings to EEPROM 
    *  (trig mode, open and closed durations, probably operating mode)
    * 
    *  @throws  DriverException
    */
    @Command(name="saveToProm", description="Save some settings to EEPROM")
    public void saveToProm() throws DriverException 
    {
        thsc.saveProm();
    }

   /**
    *  Load those settings stored in EEPROM 
    * 
    *  @throws  DriverException
    */
    @Command(name="loadFromProm", description="Load settings stored in EEPROM")
    public void loadFromProm() throws DriverException 
    {
        thsc.loadProm();
    }

    /**
     *  Set debug mode on or off
     *
     *  <true|false>
     */
    @Command(name="setDebug",description="set debug mode <true|false>")
    public void setDebug(@Argument(name="<true|false>",
			 description="debug mode") boolean mode)
    {
        thsc.setDebug(mode);
    }

}
