package org.lsst.ccs.drivers.thorlabs;

import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Driver for Thorlabs FW212C 12-posiion filter whhel. 
 *  (It can also be set for the 6-position FW102C.)
 *  Communication is via a serial port.
 *
 *  This code uses the class Session to define communication formats.
 *  Data characteristics are default (8-bit, no-parity, 1 stop-bit, no flow).
 */

public class ThorlabsFW {

    private final int BAUD_DEFAULT = 9600;
    private final int CHAR_TIMEOUT = 200;    // milliseconds
    private final int INIT_TIMEOUT = 10000;  // allows wait for wheel movement

    // Label identifying source of messages
    private final static String Lbl = "ThorlabsFW ";

    private int initTimeout;
    private int finalTimeout;

    private final Session session;

    /**
     *  Enumeration of filter wheel commands
     */

    public enum CmndFW  {
    
        IDENTIFIER     ("*idn",    true,  false),
	POSITION       ("pos",     true,  true ),
	POSITION_COUNT ("pcount",  true,  true ),  // 6 or 12
        TRIGGER_MODE   ("trig",    true,  true ),  // 0 (input) or 1 (output)
	SPEED_MODE     ("speed",   true,  true ),  // 0 (slow) or 1 (high)
	SENSOR_MODE    ("sensors", true,  true ),  // 0 (inactive), 1 (active)
	DATA_RATE      ("baud",    true,  true ),  // 0 (9600) or 1 (115200)
	SAVE_SETTINGS  ("save",    false, false);  // used at next power-up
        
	private String command;
        private boolean queryAllowed;       // Queries
        private boolean setAllowed;        // "Set" commands (with values)
        
        CmndFW(String command, boolean queryAllowed, boolean setAllowed) {
            this.command = command;
            this.queryAllowed = queryAllowed;
            this.setAllowed = setAllowed;
        }
        
        public String getCommand() {return command;}
    }

    /**
     *  Enumeration of choices for date rate
     */

    enum DataRate {

        BAUD_9600(0),
	BAUD_115200(1);

        private int rateCode;    // Code for filter wheel command

        DataRate(int rateCode) {
            this.rateCode = rateCode;
        }

        private int getCode() {
            return rateCode;
        }

    }

    /**
     *  Constructor.
     */
    public ThorlabsFW()
    {
	initTimeout = INIT_TIMEOUT;
	finalTimeout = CHAR_TIMEOUT;
        session = new Session(0, ">", "", "", null, Ascii.Terminator.CR);
    }

    /**
     *  Open serial connection using default data characteristics
     *
     *  @param  serialName   Serial device name
     *  @param  baudRate     9600 or 115200
     *  @throws DriverException
     */
    public void open(String serialName, int baudRate) throws DriverException {

        /* Check validity of baudRate (throwsIllegal ArgumentException) */
        DataRate dataRate = DataRate.valueOf("BAUD_"+ Integer.toString(baudRate));

        session.open(Session.ConnType.SERIAL, serialName, baudRate,
                     finalTimeout);
        session.setTimeout(initTimeout);

        /*
         *   Initialize some controller settings
         */
        try {
            setFW(CmndFW.DATA_RATE, dataRate.getCode());
	    setFW(CmndFW.TRIGGER_MODE, 1);
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }

    /**
     *  Open connection with default baud rate
     *
     *  @param  serialName   Serial device name
     *  @throws DriverException
     */
    public void open(String serialName) throws DriverException {
        open(serialName, BAUD_DEFAULT);
    }

    /** 
     *  Close connection
     *
     *  @throws DriverException
     */

    public void close() throws DriverException {
        session.close();
    } 

    /** Set a filter-wheel quantity
     *
     *  @param  cmnd    enumerated command identifier
     *  @param  setting integer data to write
     *  @throws DriverException
     */

    public void setFW(CmndFW cmnd, int setting) throws DriverException {
        if (!cmnd.setAllowed) {
	    throw new DriverException(Lbl + "Write not allowed for " + 
				      cmnd.toString());
        }
        String[] reply = session.receive(cmnd.getCommand() + "=" + 
                                         Integer.toString(setting));
        if (reply.length > 1) {
            if (reply[1].startsWith("Command error")) {
                throw new DriverException(Lbl + reply[1]);
	    } else {
                throw new DriverException(Lbl + "unexpected reply: " +
                                      reply[1]);
            }
	}
    }


    /** Reads a filter-wheel quantity
     *
     *  @param  query   enumerated command identifier
     *  @return value   value of quantity requested
     *  @throws DriverException
     */

    public String queryFW(CmndFW query) throws DriverException {
        if (!query.queryAllowed) {
	    throw new DriverException(Lbl + "Read not allowed for " + 
				      query.toString());
        }
        String[] reply = session.receive(query.getCommand() + "?");
        if (reply.length != 2) {
            throw new DriverException(Lbl + "unexpected reply length: " +
                                      reply.length + " lines");
        } 
        if (reply[1].startsWith("Command error")) {
            throw new DriverException(Lbl + reply[1]);
        }
        return reply[1];
    }

    /** Saves settings for use at power-up.
     *
     *  @throws DriverException
     */

    public void saveFW() throws DriverException {
        String[] reply = session.receive(CmndFW.SAVE_SETTINGS.getCommand());
        if (reply.length > 1) {
            if (reply[1].startsWith("Command error")) {
                throw new DriverException(Lbl + reply[1]);
	    } else {
                throw new DriverException(Lbl + "unexpected reply: " +
                                      reply[1]);
            }
	}
    }

    /** Set Session's debug mode on or off
     *
     *  @param debugMode  <true|false> for debug-mode <on|off>
     */

    void setDebug(boolean debugMode) {
        session.setDebug(debugMode);
    }

}
