package org.lsst.ccs.drivers.thorlabs;

import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Driver for Thorlabs_SC10 Shutter Controller
 *  Communication is via a serial port.
 *
 *  This code uses the class Session to define communication formats.
 *  Data characteristics are default (8-bit, no-parity, 1 stop-bit, no flow).
 */

public class ThorlabsSC10 {

    private final int BAUD_DEFAULT = 9600;
    private final int CHAR_TIMEOUT = 200;    // milliseconds
    private final int INIT_TIMEOUT = 10000;  // allows wait for wheel movement

    // Label identifying source of messages
    private final static String Lbl = "ThorlabsSC10 ";

    private int initTimeout;
    private int finalTimeout;

    private final Session session;

    /**
     *  Enumeration of shutter commands
     */

    public enum CmndSC  {
    
        IDENTIFIER     ("id",      true,  false),
	ENABLE         ("ens",     true,  false), // Toggles enable
	                                          // 0 (disabled), 1 (enabled)
	REPEAT_COUNT   ("rep",     true,  true ), // Repeat count (1 to 99)
	OPERATING_MODE ("mode",    true,  true ), // 1, 2, 3, 4, or 5
        TRIGGER_MODE   ("trig",    true,  true ), // 0 (internal), 1 (external)
	TRIG_OUT_MODE  ("xto",     true,  true ), // 0 or 1
	OPEN_DURATION  ("open",    true,  true ), // Open-time (ms), <=999999
	SHUT_DURATION  ("shut",    true,  true ), // Close-time (ms), <=999999
	CLOSED         ("closed",  true,  false), // 0 (open), 1 (closed)
        INTERLOCK      ("interlock",true, false), // 1 (tripped), 0 (not)
	DATA_RATE      ("baud",    true,  true ), // 0 (9600) or 1 (115200)
	SAVE           ("save",    false, false), // Save settings (baud, etc.)
	SAVE_TO_PROM   ("savp",    false, false), // Save some to EEPROM.
        LOAD_FROM_PROM ("resp",    false, false); // Load settings from EEPROM.
        /** 
         *  The "save" command saves settings (including baud and trig) to
         *  non-volatile memory, while "savp" saves settings (including
         *  xto, open, shut) to EEPROM.  I've been told that both commands
         *  save "mode"  There may be more.
         */
         
	private String command;
        private boolean queryAllowed;      // Queries
        private boolean setAllowed;        // "Set" commands (with values)
        
        CmndSC(String command, boolean queryAllowed, boolean setAllowed) {
            this.command = command;
            this.queryAllowed = queryAllowed;
            this.setAllowed = setAllowed;
        }
        
        public String getCommand() {return command;}
    }

    /**
     *  Enumeration of operating modes
     */

    public enum OpMode {

        MANUAL(1),
	AUTO(2),
	SINGLE(3),
	REPEAT(4),
	EXT_GATE(5);

        private int setting;     // Controller setting for operating mode

        OpMode(int setting) {
            this.setting = setting;
        }

        public int getSetting() {return setting;}

        // This method takes the code in String form, from a queryFW
        // command; it is converted to an Integer for comparison.
        public static String decode(String code) throws DriverException {
            OpMode[] mode = OpMode.values();
            int n = mode.length;
            int found = -1;
            for (int i = 0; i < n; i++) {
                if (mode[i].setting == Integer.parseInt(code)) {
                    found = i;
                    break;
                }
            }
            if (found == -1) throw new DriverException(Lbl + "invalid code for operating mode");
            return mode[found].toString();
        }

        public static String listModes()
        {
            String list = "Shutter operating modes:";
            OpMode[] mode = OpMode.values();
            int n = mode.length;
            for (int i = 0; i < n; i++) {
                list += ("\n   " + mode[i].setting + " = " + mode[i]);
            }
            return list;
        }
    }

    /**
     *  Enumeration of trigger modes
     */

    public enum TrigMode {

        INTERNAL(0),
	EXTERNAL(1);

        private int setting;     // Controller setting for trigger mode

        TrigMode(int setting) {
            this.setting = setting;
        }

        public int getSetting() {return setting;}

        public static String decode(String code) throws DriverException {
            TrigMode[] mode = TrigMode.values();
            int n = mode.length;
            int found = -1;
            for (int i = 0; i < n; i++) {
                if (mode[i].setting == Integer.parseInt(code)) {
                    found = i;
                    break;
                }
            }
            if (found == -1) throw new DriverException(Lbl + "invalid code for trigger mode");
            return mode[found].toString();
        }
    }

    /**
     *  Enumeration of choices for date rate
     */

    enum DataRate {

        BAUD_9600(0),
	BAUD_115200(1);

        private int rateCode;    // Code for shutter command

        DataRate(int rateCode) {
            this.rateCode = rateCode;
        }

        private int getCode() {
            return rateCode;
        }

    }

    /**
     *  Constructor.
     */
    public ThorlabsSC10()
    {
	initTimeout = INIT_TIMEOUT;
	finalTimeout = CHAR_TIMEOUT;
        session = new Session(0, ">", "", "", null, Ascii.Terminator.CR);
    }

    /**
     *  Open serial connection using default data characteristics
     *
     *  @param  serialName   Serial device name
     *  @param  baudRate     9600 or 115200
     *  @throws DriverException
     */
    public void open(String serialName, int baudRate) throws DriverException {

        /* Check validity of baudRate (throwsIllegal ArgumentException) */
        DataRate dataRate = DataRate.valueOf("BAUD_"+ Integer.toString(baudRate));

        session.open(Session.ConnType.SERIAL, serialName, baudRate,
                     finalTimeout);
        session.setTimeout(initTimeout);
        // setDebug(true);    // Temporary, later will default to false.

        /*
         *   Initialize some controller settings
         */
        try {
            setSC(CmndSC.DATA_RATE, dataRate.getCode());
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }

    /**
     *  Open connection with default baud rate
     *
     *  @param  serialName   Serial device name
     *  @throws DriverException
     */
    public void open(String serialName) throws DriverException {
        open(serialName, BAUD_DEFAULT);
    }

    /** 
     *  Close connection
     *
     *  @throws DriverException
     */

    public void close() throws DriverException {
        session.close();
    } 


    /** Set a shutter-controller quantity
     *
     *  @param  cmnd    enumerated command identifier
     *  @param  setting integer data to write
     *  @throws DriverException
     */

    public void setSC(CmndSC cmnd, int setting) throws DriverException {
        if (!cmnd.setAllowed) {
	    throw new DriverException(Lbl + "Write not allowed for " + 
				      cmnd.toString());
        }
        String[] reply = session.receive(cmnd.getCommand() + "=" + 
                                         Integer.toString(setting));
        if (reply.length > 1) {
            if (reply[1].startsWith("Command error")) {
                throw new DriverException(Lbl + reply[1]);
	    } else {
                throw new DriverException(Lbl + "unexpected reply: " +
                                      reply[1]);
            }
	}
    }


    /** Reads a shutter-controller quantity
     *
     *  @param  query   enumerated command identifier
     *  @return value   value of quantity requested
     *  @throws DriverException
     */

    public String querySC(CmndSC query) throws DriverException {
        if (!query.queryAllowed) {
	    throw new DriverException(Lbl + "Read not allowed for " + 
				      query.toString());
        }
        String[] reply = session.receive(query.getCommand() + "?");
        if (reply.length != 2) {
            throw new DriverException(Lbl + "unexpected reply length: " +
                                      reply.length + " lines");
        } 
        if (reply[1].startsWith("Command error")) {
            throw new DriverException(Lbl + reply[1]);
        }
        return reply[1];
    }

    /**
     *  Process a void command with no argument (i.e., not setting or query)
     *
     *  @throws DriverException
     */
    public void processSC(CmndSC cmnd) throws DriverException {
        String[] reply = session.receive(cmnd.getCommand());
        if (reply.length > 1) {
            if (reply[1].startsWith("Command error")) {
                throw new DriverException(Lbl + reply[1]);
	    } else {
                throw new DriverException(Lbl + "unexpected reply: " +
                                      reply[1]);
            }
	}
    }


    /**
     *  Toggle enable/disable or initiate an internal trigger
     *  (Effect depends upon selected mode.)
     *
     *  @throws DriverException
     */

    public void enable() throws DriverException {
        processSC(CmndSC.ENABLE);
    }

    /** Saves baud rate and trigger mode
     *
     *  @throws DriverException
     */

    public void saveSC() throws DriverException {
        processSC(CmndSC.SAVE);
    }

    /** Saves external trigger mode, open time, shut time in EEPROM
     *
     *  @throws DriverException
     */

    public void saveProm() throws DriverException {
        processSC(CmndSC.SAVE_TO_PROM);
    }

    /** Loads values from EEPROM
     *
     *  @throws DriverException
     */

    public void loadProm() throws DriverException {
        processSC(CmndSC.LOAD_FROM_PROM);
    }

    /** Set Session's debug mode on or off
     *
     *  @param debugMode  <true|false> for debug-mode <on|off>
     */

    void setDebug(boolean debugMode) {
        session.setDebug(debugMode);
    }

}
